#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>
#include <drive/backend/ut/library/script.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/chat_robots/state/robot_state.pb.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/container_tag.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/rt_background/drop_performer/config.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/registrar/hasher.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/support_center/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <drive/tests/library/database.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(SupportChat) {

    Y_UNIT_TEST(Simple) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }
    }

    Y_UNIT_TEST(SentMessagesId) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there222", false, "active");

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "!how_to_get_higher_class_car", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "!ok", {}));

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            auto messagesArr = messagesJson["messages"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(messagesArr[messagesArr.size() - 2]["author"].GetString(), userId);
        }
    }

    Y_UNIT_TEST(CreateOnRead) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_no_create_on_read.1");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson, NJson::JSON_NULL);
        }

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_no_create_on_read.1", "hello!", {}));

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_no_create_on_read.1");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }
    }

    Y_UNIT_TEST(ManyTopics) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support.1");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
            {
                auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
                UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            }
        }
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support.2");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
            {
                auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
                UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 2);
            }
        }
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support.1", "Test #1", {}));
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support.2", "Test #2", {}));

        TSet<TString> supportChatIds;
        supportChatIds.emplace("support.1");
        supportChatIds.emplace("support.2");
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 5);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
            UNIT_ASSERT_C(supportChatIds.contains(chatsJson["chats"].GetArray()[0]["id"].GetString()), TStringBuilder() << chatsJson);

            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[1]["stats"]["total"].GetInteger(), 5);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[1]["stats"]["unread"].GetInteger(), 0);
            UNIT_ASSERT_C(supportChatIds.contains(chatsJson["chats"].GetArray()[1]["id"].GetString()), TStringBuilder() << chatsJson);
        }
        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get(), USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 5);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 5);

            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[1]["stats"]["total"].GetInteger(), 5);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[1]["stats"]["unread"].GetInteger(), 5);
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support.1");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"], "Test #1");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support.2");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["text"], "Test #2");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }
    }

    Y_UNIT_TEST(SupportTalksWithCustomer) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Hello, Yandex.Drive!", {}));
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Your support chat is really awesome!", {}));
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Check out this funny cat picture.", {}));
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 7);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);

            chatsJson = configGenerator.GetChatsList(userId, server.Get(), USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 7);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 7);
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["last_viewed_others"].GetInteger(), 0);
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 7);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);

            chatsJson = configGenerator.GetChatsList(userId, server.Get(), USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 7);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 7);
        }

        {
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "Hello, skulik-was-there!"));
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "It seems, you haven't sent any picture to me yet :("));
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 9);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 2);

            chatsJson = configGenerator.GetChatsList(userId, server.Get(), USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 9);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT(messagesJson["last_viewed_others"].GetInteger() != 0);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "", {"cat.jpg"}));
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Oh, here it is!", {}));

            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 11);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 11);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);

            chatsJson = configGenerator.GetChatsList(userId, server.Get(), USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 11);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 2);
        }

        {
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetChatResource(userId, "support", "cat.jpg"), "");
        }

        {
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "I still don't see anything"));
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "Maybe your internet is slow"));

            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 13);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 13);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);

            chatsJson = configGenerator.GetChatsList(userId, server.Get(), USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 13);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }

        {
            auto result = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(result["messages"].GetArray()[9]["type"], "media_resources");
            UNIT_ASSERT_VALUES_EQUAL(result["messages"].GetArray()[9]["content_types"].GetArray().size(), 1);
            UNIT_ASSERT_C(result["messages"].GetArray()[9]["content_types"].GetArray()[0].IsNull(), TStringBuilder() << result["messages"].GetArray()[9]);
        }

        {
            auto resourceId = configGenerator.SubmitChatResource(userId, "support", "some picture", "cat.jpg", "image/jpeg");
            server.Get()->GetChatRobotsManager()->GetChatRobotsMediaStorage()->ForceRefresh(Now());

            auto result = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(result["messages"].GetArray()[9]["type"], "media_resources");
            UNIT_ASSERT_VALUES_EQUAL(result["messages"].GetArray()[9]["content_types"].GetArray().size(), 1);
            UNIT_ASSERT_C(result["messages"].GetArray()[9]["content_types"].GetArray()[0].GetString().EndsWith("jpeg"), TStringBuilder() << result["messages"].GetArray()[9]);

            UNIT_ASSERT_VALUES_EQUAL(resourceId, "cat.jpg");

            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetChatResource(userId, "support", "cat.jpg"), "some picture");

            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "Oh, here it is!"));
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 14);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 1);

            chatsJson = configGenerator.GetChatsList(userId, server.Get(), USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["total"].GetInteger(), 14);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["stats"]["unread"].GetInteger(), 0);
        }

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 14);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }
    }

    Y_UNIT_TEST(BadRequests) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            UNIT_ASSERT(!configGenerator.CommitChatAction(userId, "support", "Hello, Yandex.Drive!", {"a", "b", "c"}));

            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            UNIT_ASSERT(!configGenerator.CommitChatAction(userId, "support", "", {}));

            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            auto resourceId = configGenerator.SubmitChatResource(userId, "support", "some picture", "cat.jpg", "");
            UNIT_ASSERT_VALUES_EQUAL(resourceId, "");

            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            auto resourceId = configGenerator.SubmitChatResource(userId, "support", "some picture", "cat.jpg", "image/jpeg");
            UNIT_ASSERT_VALUES_EQUAL(resourceId, "");

            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "", {"cat.jpg"}));
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetChatResource(userId, "support", "cat.jpg"), "");

            resourceId = configGenerator.SubmitChatResource(userId, "support", "some pictureeeee", "catt.jpg", "image/jpeg");
            UNIT_ASSERT_VALUES_EQUAL(resourceId, "");
            resourceId = configGenerator.SubmitChatResource(userId, "support", "some picture", "cat.jpg", "image/jpeg");
            UNIT_ASSERT_VALUES_EQUAL(resourceId, "cat.jpg");

            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetChatResource(userId, "support", "catt.jpg"), "");
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetChatResource(userId, "support", "cat.jpg"), "some picture");
        }

        TString picName = ToString(RandomNumber<ui64>(100000000000000)) + "zxzx.jpg";
        TString vidName = ToString(RandomNumber<ui64>(100000000000000)) + "zxzx.mp4";

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "", {picName}));
            UNIT_ASSERT(!configGenerator.SubmitChatResource(userId, "support", "", picName, "image/jpeg"));
            UNIT_ASSERT(configGenerator.SubmitChatResource(userId, "support", "zzz", picName, "image/jpeg"));
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetChatResource(userId, "support", picName), "zzz");
            configGenerator.GetChatResourcePreview(userId, "support", picName);
        }

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "", {vidName}));
            UNIT_ASSERT(!configGenerator.SubmitChatResource(userId, "support", "", vidName, "video/mp4"));
            UNIT_ASSERT(configGenerator.SubmitChatResource(userId, "support", "zzz", vidName, "video/mp4"));
            UNIT_ASSERT_VALUES_EQUAL(configGenerator.GetChatResource(userId, "support", vidName), "zzz");
            configGenerator.GetChatResourcePreview(userId, "support", vidName);
        }
    }

    Y_UNIT_TEST(StaticChat) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        for (size_t it = 0; it < 5; ++it) {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_intro");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "tree");
        }

        UNIT_ASSERT(!configGenerator.CommitChatAction(userId, "support_intro", "Test #1", {}));

        for (size_t it = 0; it < 5; ++it) {
            auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_intro");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "tree");
        }

        {
            auto staticChatRobot = server->GetChatRobotsManager()->GetChatRobot("support_intro");
            auto searchId = staticChatRobot->GetChatSearchId(userId, "");
            NDrive::NChat::TChat chat;
            UNIT_ASSERT(!server->GetChatEngine()->GetChats().GetChatFromTable(searchId, chat));
        }
    }

    Y_UNIT_TEST(StaticChatForSupport) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        }

        {
            auto userData = driveApi.GetUsersData()->FetchInfo(userId).MutableResult().begin()->second;
            userData.SetStatus("blocked");

            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetUsersData()->UpdateUser(userData, "robot-frontend", session) && session.Commit());

            driveApi.GetUsersData()->FetchInfo(userId);
        }

        for (size_t it = 0; it < 5; ++it) {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_intro");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 4);
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["expected_action"].GetStringRobust());
        }

        {
            auto chatsJson = configGenerator.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }
    }

    Y_UNIT_TEST(LongChatName) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support.857fab7a-4470-4d00-9226-8d0b6b249ac6", "Test #1234", {}));
    }

    Y_UNIT_TEST(TreeFallback) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_treelike.123", "hello world!", {}));

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_treelike.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        // Check that tag of the specific line is created
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
    }

    Y_UNIT_TEST(TreelikeSupport) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_treelike.123", "driver_requirements", {}));

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_treelike.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "context_buttons");
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }

        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support_treelike.123", "yes", {}));

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support_treelike.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "chat_closed");
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
    }

    Y_UNIT_TEST(SendBonuses) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 1));  // 4
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 2));  // 5
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 3));  // 6
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 4));  // 7
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 5));  // 8
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 6));  // 9
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 10));  // 10
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 11));  // 11
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 12));  // 12
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 13));  // 13
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 14));  // 14
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 112));  // 15
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 21));  // 16
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 22));  // 17
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 23));  // 18
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 24));  // 19
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 25));  // 20
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 320));  // 21
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 321));  // 22
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 322));  // 23
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 323));  // 24
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 324));  // 25
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 325));  // 26
        UNIT_ASSERT(configGenerator.SendBonusesToChat(userId, "support.123", 211));  // 27

        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support.123");
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[4]["text"].GetString().find("бонусный рубль") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[5]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[6]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[7]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[8]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[9]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[10]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[11]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[12]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[13]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[14]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[15]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[16]["text"].GetString().find("бонусный рубль") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[17]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[18]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[19]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[20]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[21]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[22]["text"].GetString().find("бонусный рубль") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[23]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[24]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[25]["text"].GetString().find("бонусных рубля") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[26]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
            UNIT_ASSERT_C(messagesJson["messages"].GetArray()[27]["text"].GetString().find("бонусных рублей") != TString::npos, TStringBuilder() << messagesJson);
        }
    }

    Y_UNIT_TEST(AssignOperator) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 0);

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(userId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
        }

        UNIT_ASSERT_VALUES_EQUAL(server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 1);

        auto clientId = eGenerator.CreateUser("zxqfd555-was-there", true, "active");
        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_modern.dsfjhsdgf", "Hello, Yandex.Drive!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_modern.dsfjhsdgf", "Your support chat is really awesome!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_modern.dsfjhsdgf", "Check out this funny cat picture.", {}));
        }

        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_2_line" });
        }

        TString tl;
        TString tagId;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags[0]->GetPerformer(), userId);

            auto impl = dbTags[0].GetTagAs<TSupportChatTag>();
            UNIT_ASSERT(!!impl);
            UNIT_ASSERT_VALUES_EQUAL(impl->GetTopicLink(), "support_modern.dsfjhsdgf");
            tl = impl->GetTopicLink();
            tagId = dbTags[0].GetTagId();
        }

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, tl);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            TVector<std::pair<TString, TString>> chatsRequest;
            chatsRequest.push_back(std::make_pair(clientId, tl));
            auto chatsJson = gServer.GetAdminChatsList(chatsRequest);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["tag_id"].GetString(), tagId);
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_2_line"}, dbTags, session));

            auto permissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());

            auto tag = dbTags.front();
            auto newTag = tag.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
            UNIT_ASSERT(!!newTag);
            newTag->SetName("support_chat_1_line");

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().EvolveTag(tag, newTag.GetData(), *permissions, server.Get(), session) && session.Commit());
        }

        {
            server->GetChatRobot("support_modern")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, tl);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_2_line", "support_chat_1_line"}, dbTags, session));
            }
            for (auto&& tag : dbTags) {
                UNIT_ASSERT(gServer.CloseChat(tag.GetTagId(), userId));
            }
        }

        {
            server->GetChatRobot("support_modern")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, tl);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "context_buttons");
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_modern.dsfjhsdgf", "!normal", {}));
        }

        {
            server->GetChatRobot("support_modern")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, tl);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "context_buttons");
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_modern.dsfjhsdgf", "!other", {}));
        }

        {
            server->GetChatRobot("support_modern")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, tl);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_modern.dsfjhsdgf", "Some nontrivial feedback", {}));
        }

        // Don't return chat to active chats after message
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_dtp", "support_chat_2_line"}, dbTags, session));
            }
            for (auto&& tag : dbTags) {
                ERROR_LOG << tag->GetName() << Endl;
            }
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {TSupportChatTag::FeedbackName}, dbTags, session));
            }

            // Check that feedback is saved
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tag = dbTags[0];
            auto tagImpl = tag.GetTagAs<TSupportChatTag>();
            UNIT_ASSERT(tagImpl);
            UNIT_ASSERT_VALUES_EQUAL(tagImpl->GetFeedback().GetOverallQuality(), "normal");
            UNIT_ASSERT_VALUES_EQUAL(tagImpl->GetFeedback().GetIssues().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tagImpl->GetFeedback().GetIssues().front(), "other");
        }

        {
            server->GetChatRobot("support_modern")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, tl);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "chat_closed");
        }
    }

    Y_UNIT_TEST(StartWithTreePutTagAfter) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 0);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(userId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 1);

        auto clientId = eGenerator.CreateUser("zxqfd555-was-there", true, "active");

        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_treelike.123", "driver_requirements", {}));
        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, "support_treelike.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "context_buttons");

            // Check that there is no tag
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TVector<TDBTag> dbTags;
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_1_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
            }
        }

        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_treelike.123", "no", {}));

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, "support_treelike.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");

            // Check that there is still no tag
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TVector<TDBTag> dbTags;
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_1_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
            }
        }

        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support_treelike.123", "hello", {}));

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, "support_treelike.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                TVector<TDBTag> dbTags;
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_1_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
        }
    }

    Y_UNIT_TEST(StartWithOrder) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        UNIT_ASSERT(gServer.SendOrderToChat(userId, "support.123", "zxzxzx"));

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["type"].GetString(), "order");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }
    }

    Y_UNIT_TEST(Resume) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        UNIT_ASSERT(gServer.SendOrderToChat(userId, "support.123", "zxzxzx"));

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line", "support_chat_1_line"}, dbTags, session));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(dbTags, USER_ROOT_DEFAULT, server.Get(), session, true) && session.Commit());
        }

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["type"].GetString(), "order");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        UNIT_ASSERT(gServer.SendArbitraryMessage("support.123", userId, "Hello, skulik-was-there!"));
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support.123", "xzxzxzx", {}));
        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
    }

    Y_UNIT_TEST(ResumeViaEvolve) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        auto operatorId = eGenerator.CreateUser("operator-was-there", true, "active");

        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("support_chat_dtp", Now()).size(), 0);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(operatorId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, operatorId, session) && session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees("support_chat_dtp", Now()).size(), 1);

        UNIT_ASSERT(gServer.SendOrderToChat(userId, "support.123", "zxzxzx"));

        TString chatTagId = "";
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            chatTagId = dbTags.begin()->GetTagId();

            auto tagOriginal = dbTags.front();
            auto tagDestination = tagOriginal.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
            tagDestination->SetName("support_chat_dtp");
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().DirectEvolveTag("robot-frontend", tagOriginal, tagDestination.GetData(), session) && session.Commit());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_dtp"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }

        // Start performing
        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_dtp" });

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_dtp"}, dbTags, session));

            auto tag = dbTags.front();
            UNIT_ASSERT(tag->GetPerformer() != "");
        }

        // Defer
        {
            UNIT_ASSERT(gServer.DeferRequest(chatTagId, operatorId));
        }

        // There shall be no chats in 2line
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_dtp"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }

        // Yet one deferred, with no performer
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_deferred"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            auto tag = dbTags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), "");
        }

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 6);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[4]["type"].GetString(), "order");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        UNIT_ASSERT(gServer.SendArbitraryMessage("support.123", userId, "Hello, skulik-was-there!"));
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support.123", "xzxzxzx", {}));
        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_dtp"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            auto tag = dbTags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), "");
        }
    }

    Y_UNIT_TEST(ChatWriteAccess) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there", false, "active");
        auto userId2 = eGenerator.CreateUser("skulik-was-there-too", false, "active");

        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 0);

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(userId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
        }

        UNIT_ASSERT_VALUES_EQUAL(server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 1);

        auto clientId = eGenerator.CreateUser("zxqfd555-was-there", true, "active");
        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.dsfjhsdgf", "Hello, Yandex.Drive!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.dsfjhsdgf", "Your support chat is really awesome!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.dsfjhsdgf", "Check out this funny cat picture.", {}));
        }

        UNIT_ASSERT(!gServer.SendArbitraryMessage("support.dsfjhsdgf", clientId, "Hello, skulik-was-there!", userId2));
        UNIT_ASSERT(!gServer.SendArbitraryMessage("support.dsfjhsdgf", clientId, "Hello, skulik-was-there!", userId));

        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_2_line" });
        }

        TString tagId;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags[0]->GetPerformer(), userId);
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(userId2).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId2, session) && session.Commit());
        }

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(clientId, "support.dsfjhsdgf");
            ERROR_LOG << "ChatDebug " << messagesJson << Endl;
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        UNIT_ASSERT(!gServer.SendArbitraryMessage("support.dsfjhsdgf", clientId, "Hello, skulik-was-there!", userId2));
        UNIT_ASSERT(gServer.SendArbitraryMessage("support.dsfjhsdgf", clientId, "Hello, skulik-was-there!", userId));
    }

    Y_UNIT_TEST(DeleteMessages) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Hello, Yandex.Drive!", {}));
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Your support chat is really awesome!", {}));
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Check out this funny cat picture.", {}));
        }

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
        }

        {
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "Hello, skulik-was-there!"));
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "It seems, you haven't sent any picture to me yet :("));
        }
        TVector<i64> orderedMessageIds;
        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = configGenerator.GetChatMessages(userId, "support", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
            for (auto&& message : messagesJson["messages"].GetArray()) {
                orderedMessageIds.push_back(message["id"].GetInteger());
            }
            UNIT_ASSERT(configGenerator.DeleteChatMessage(userId, "support", orderedMessageIds[7]));
        }
        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
            int i = 0;
            for (auto&& message : messagesJson["messages"].GetArray()) {
                UNIT_ASSERT_C(orderedMessageIds[i] == message["id"].GetInteger(), TStringBuilder() <<
                                                 orderedMessageIds[i] << " != " << message["id"].GetInteger() << ", i = " << i);
                if (i == 6) {
                    i += 2;
                } else {
                    ++i;
                }
            }
        }
        {
            UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Hello again, Yandex.Drive!", {}));
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "Hello again, skulik-was-there!"));
        }
        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = configGenerator.GetChatMessages(userId, "support", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 10);
            orderedMessageIds.push_back(messagesJson["messages"].GetArray()[8]["id"].GetInteger());
            orderedMessageIds.push_back(messagesJson["messages"].GetArray()[9]["id"].GetInteger());
            UNIT_ASSERT(configGenerator.DeleteChatMessage(userId, "support", orderedMessageIds[10]));
        }
        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            int i = 0;
            for (auto&& message : messagesJson["messages"].GetArray()) {
                UNIT_ASSERT_C(orderedMessageIds[i] == message["id"].GetInteger(), TStringBuilder() <<
                                                 orderedMessageIds[i] << " != " << message["id"].GetInteger() << ", i = " << i);
                if (i == 6) {
                    i += 2;
                } else {
                    ++i;
                }
            }
        }
    }

    Y_UNIT_TEST(SupportRequestsArchive) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 0);

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(userId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
        }

        UNIT_ASSERT_VALUES_EQUAL(server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 1);

        auto clientId = eGenerator.CreateUser("zxqfd555-was-there", true, "active");
        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.dsfjhsdgf", "Hello, Yandex.Drive!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.dsfjhsdgf", "Your support chat is really awesome!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.dsfjhsdgf", "Check out this funny cat picture.", {}));
        }

        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_2_line" });
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_2_line", "support_chat_1_line"}, dbTags, session));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(dbTags, USER_ROOT_DEFAULT, server.Get(), session, true) && session.Commit());
        }

        {
            // force cache update

            UNIT_ASSERT_VALUES_EQUAL(gServer.GetSupportRequestArchiveByUser(clientId)["tags"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(gServer.GetSupportRequestArchiveByUser(userId)["tags"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(gServer.GetSupportRequestArchiveByPerformer(clientId)["tags"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(gServer.GetSupportRequestArchiveByPerformer(userId)["tags"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(gServer.GetSupportRequestArchiveByTags({"@user_support_chat_tag"}, clientId)["tags"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(gServer.GetSupportRequestArchiveByTags({"@user_support_call_tag"}, clientId)["tags"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(gServer.GetSupportRequestArchiveByTags({"support_chat_2_line"}, clientId)["tags"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(gServer.GetSupportRequestArchiveByTags({"support_chat_1_line"}, clientId)["tags"].GetArray().size(), 0);
        }

        {
            TInstant addTagTime = Now() - TDuration::Minutes(20);
            auto userWithCall = eGenerator.CreateUser("qweqweqwe", true, "active");
            auto callOperator = eGenerator.CreateUser("qweqweqweOperator", true, "active");
            TVector<TDBTag> newTags;
            {
                TInstantGuard ig(addTagTime);
                auto session = driveApi.BuildTx<NSQL::Writable>();
                TUserRole userRole;
                userRole.SetRoleId("support_assignee").SetUserId(callOperator).SetActive(true);
                UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, callOperator, session));
                THolder<TSupportPhoneCallTag> tag(new TSupportPhoneCallTag("cc_audiotele_incoming"));
                tag->SetInternalCallId("123");
                auto optionalAddedTags = driveApi.GetTagsManager().GetUserTags().AddTag(tag.Release(), USER_ROOT_DEFAULT, userWithCall, server.Get(), session);
                UNIT_ASSERT(optionalAddedTags);
                newTags = *optionalAddedTags;
                UNIT_ASSERT(session.Commit());
            }
            UNIT_ASSERT(newTags.size() == 1);
            TInstant startTagTime = Now() - TDuration::Minutes(5);
            TInstant endTagTime = startTagTime + TDuration::Minutes(2);
            {
                {
                    TInstantGuard ig(startTagTime);
                    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().SetTagsPerformer(newTags, callOperator, false, session, server.Get()));
                    UNIT_ASSERT(session.Commit());
                }
                {
                    TInstantGuard ig(endTagTime);
                    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().DropTagsPerformer(newTags, callOperator, session));
                    UNIT_ASSERT(driveApi.GetTagsManager().GetUserTags().RemoveTags(newTags, userWithCall, server.Get(), session));
                    UNIT_ASSERT(session.Commit());
                }
            }
            {
                auto reply = gServer.GetSupportRequestArchiveByUser(userWithCall);
                UNIT_ASSERT(reply["tags"][0]["call"].IsMap());
                UNIT_ASSERT(reply["tags"][0]["call"]["connect"].IsUInteger());
                TInstant time = TInstant::Seconds(reply["tags"][0]["call"]["connect"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), addTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["enter"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["enter"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), startTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["exit"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["exit"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), endTagTime.Seconds());
            }
            {
                auto reply = gServer.GetSupportRequestArchiveByPerformer(callOperator);
                UNIT_ASSERT(reply["tags"][0]["call"].IsMap());
                UNIT_ASSERT(reply["tags"][0]["call"]["connect"].IsUInteger());
                TInstant time = TInstant::Seconds(reply["tags"][0]["call"]["connect"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), addTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["enter"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["enter"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), startTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["exit"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["exit"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), endTagTime.Seconds());
            }
            {
                auto start = ToString(startTagTime.Seconds() + 10);
                TString cgi = "user_id=" + userWithCall + "&since=" + start;
                auto reply = gServer.GetSupportRequestsWithCgi(cgi);
                UNIT_ASSERT(reply["tags"][0]["call"].IsMap());
                UNIT_ASSERT(reply["tags"][0]["call"]["connect"].IsUInteger());
                TInstant time = TInstant::Seconds(reply["tags"][0]["call"]["connect"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), addTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["enter"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["enter"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), startTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["exit"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["exit"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), endTagTime.Seconds());
            }
            {
                auto end = ToString(endTagTime.Seconds() - 20);
                TString cgi = "user_id=" + userWithCall + "&until=" + end;
                auto reply = gServer.GetSupportRequestsWithCgi(cgi);
                UNIT_ASSERT(reply["tags"][0]["call"].IsMap());
                UNIT_ASSERT(reply["tags"][0]["call"]["connect"].IsUInteger());
                TInstant time = TInstant::Seconds(reply["tags"][0]["call"]["connect"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), addTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["enter"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["enter"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), startTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["exit"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["exit"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), endTagTime.Seconds());
            }
            {
                TString cgi = "user_id=" + userWithCall + "&rev=true";
                auto reply = gServer.GetSupportRequestsWithCgi(cgi);
                UNIT_ASSERT(reply["tags"][0]["call"].IsMap());
                UNIT_ASSERT(reply["tags"][0]["call"]["connect"].IsUInteger());
                TInstant time = TInstant::Seconds(reply["tags"][0]["call"]["connect"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), addTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["enter"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["enter"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), startTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["exit"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["exit"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), endTagTime.Seconds());
            }
            {
                auto end = ToString(endTagTime.Seconds() - 40);
                TString cgi = "user_id=" + userWithCall + "&until=" + end + "&rev=true";
                auto reply = gServer.GetSupportRequestsWithCgi(cgi);
                UNIT_ASSERT(reply["tags"][0]["call"].IsMap());
                UNIT_ASSERT(reply["tags"][0]["call"]["connect"].IsUInteger());
                TInstant time = TInstant::Seconds(reply["tags"][0]["call"]["connect"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), addTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["enter"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["enter"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), startTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["exit"].IsUInteger());
                time = TInstant::Seconds(reply["tags"][0]["call"]["exit"].GetUInteger());
                UNIT_ASSERT_VALUES_EQUAL(time.Seconds(), endTagTime.Seconds());
                UNIT_ASSERT(reply["tags"][0]["call"]["url"].IsString());
                UNIT_ASSERT_EQUAL_C(reply["tags"][0]["call"]["url"].GetString(), "test.com/123", reply["tags"][0]["call"]["url"].GetString());
            }
        }
    }

    Y_UNIT_TEST(ForbiddenTraitsWrite) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", false, "active");

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support", "hello!", {}));
        UNIT_ASSERT(!gServer.CommitChatAction(userId, "support", "bye!", {}, 0, 0, {NDrive::NChat::TMessage::EMessageTraits::StaffOnly}));
        auto messagesJson = gServer.GetChatMessages(userId, "support");
        UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 5);
    }

    Y_UNIT_TEST(ForbiddenTraitsRead) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", false, "active");

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support", "Hello, Yandex.Drive!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support", "Your support chat is really awesome!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support", "Check out this funny cat picture.", {}));
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support", "test note",  {}, 0, 0, {NDrive::NChat::TMessage::EMessageTraits::StaffOnly}, USER_ROOT_DEFAULT));
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 8);
        }
    }

    Y_UNIT_TEST(ThroughNodesClassify) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there222", false, "active");

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern_v2.123", "!cant_bind_bank_card", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern_v2.123", "!cant_bind_bank_card_yes", {}));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern_v2.123", "hello", {}));

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->RefreshCache(Now()));

            auto tagId = dbTags.front().GetTagId();
            auto existing = gServer.GetCategorizations({tagId}, USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["tag_id"].GetString(), tagId);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["items"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["items"].GetArray()[0]["category"]["id"].GetString(), "af9be1bb-2ce6-41e2-8461-bc5809b05e75" );
        }
    }

    Y_UNIT_TEST(CallSupportImmediately) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto urgentUserId = eGenerator.CreateUser("skulik-was-there222", false, "active");
        auto nonUrgentUserId = eGenerator.CreateUser("skulik-was-there333", false, "active");

        UNIT_ASSERT(gServer.CommitChatAction(urgentUserId, "support_modern.123", "!dtp_happened", {}));
        UNIT_ASSERT(gServer.CommitChatAction(nonUrgentUserId, "support_modern.123", "!other_problem_in_riding", {}));

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ urgentUserId }, {}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT(server->GetSupportCenterManager()->GetSupportRequestCategorizer()->RefreshCache(Now()));

            auto tagId = dbTags.front().GetTagId();
            auto existing = gServer.GetCategorizations({tagId}, USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["tag_id"].GetString(), tagId);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["items"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(existing["categorizations"].GetArray()[0]["items"].GetArray()[0]["category"]["id"].GetString(), "044b8e35-0a02-4705-bb2a-5805ed381561" );
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ nonUrgentUserId }, {}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }
    }

    Y_UNIT_TEST(ConditionsPerformTag) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there222", true, "active");

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["message_text"].GetString(), "talk_with_human");
        }

        // now start the order and extra option appears
        auto session = driveApi.BuildTx<NSQL::Writable>();
        TEnvironmentGenerator::TCar newCar = eGenerator.CreateCar("kia_rio");

        TString objectId = newCar.Id;
        TString imei = newCar.IMEI;

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(imei);
        UNIT_ASSERT(gServer.WaitLocation(objectId));
        UNIT_ASSERT(gServer.WaitSensor(objectId, "mileage"));

        emulator->GetContext().SetFuelPercent(50);
        emulator->GetContext().TrySetEngineStarted(true);
        UNIT_ASSERT(gServer.WaitSensor(objectId, "fuel_level", "50"));

        TString offerId;
        {
            THolder<TStandartOffer> offer(new TStandartOffer);
            offer->SetObjectId(objectId).SetUserId(userId).SetDeadline(Now() + TDuration::Minutes(5));
            offer->SetChargableAccounts({ "card", "bonus" });
            offerId = offer->GetOfferId();
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer.Release(), nullptr)}));
        }
        UNIT_ASSERT(gServer.BookOffer(offerId, userId));
        UNIT_ASSERT(gServer.WaitStatus(objectId, "old_state_reservation", *server));

        {
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 4);
        }

        UNIT_ASSERT(gServer.EvolveTag("old_state_acceptance", userId));

        {
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 2);
        }

        UNIT_ASSERT(gServer.WaitStatus(objectId, "old_state_acceptance", *server));

        {
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 2);
        }

        UNIT_ASSERT(gServer.EvolveTag("old_state_riding", userId));

        {
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 1);
        }
    }

    Y_UNIT_TEST(ConditionsHasTag) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there222", true, "active");

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["message_text"].GetString(), "talk_with_human");
        }

        UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["message_text"].GetString(), "talk_with_human");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[1]["message_text"].GetString(), "talk_about_problem");
        }
    }

    Y_UNIT_TEST(ConditionsStatus) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-there222", true, "active");

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["message_text"].GetString(), "talk_with_human");
        }

        {
            auto userData = driveApi.GetUsersData()->FetchInfo(userId).MutableResult().begin()->second;
            userData.SetStatus("onboarding");
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetUsersData()->UpdateUser(userData, "robot-frontend", session) && session.Commit());
        }

        {
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["message_text"].GetString(), "talk_with_human");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[1]["message_text"].GetString(), "im_in_onboarding");
        }
    }

    Y_UNIT_TEST(ConditionCarTag) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        gServer.SetLogLevel(7);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("skulik-was-not-there111", true, "active");

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["message_text"].GetString(), "talk_with_human");
        }

        auto session = driveApi.BuildTx<NSQL::Writable>();
        TEnvironmentGenerator::TCar newCar = eGenerator.CreateCar("kia_rio");

        TString objectId = newCar.Id;
        TString imei = newCar.IMEI;

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(imei);
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(gServer.WaitLocation(objectId));
        UNIT_ASSERT(gServer.WaitSensor(objectId, "mileage"));
        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(false));

        TString offerId;
        {
            THolder<TStandartOffer> offer(new TStandartOffer);
            offer->SetObjectId(objectId).SetUserId(userId).SetDeadline(Now() + TDuration::Minutes(5));
            offer->SetChargableAccounts({ "card", "bonus" });
            offerId = offer->GetOfferId();
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer.Release(), nullptr)}));
        }
        UNIT_ASSERT(gServer.BookOffer(offerId, userId));
        UNIT_ASSERT(gServer.WaitStatus(objectId, "old_state_reservation", *server));
        { // car has old_state_reservation tag
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["text"].GetString(), "Регистрация и авторизация");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[1]["message_text"].GetString(), "talk_with_human");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[2]["message_text"].GetString(), "talk_about_session_no_fueling");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[3]["message_text"].GetString(), "talk_about_session_any");
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            ITag::TPtr tag = ITag::TFactory::Construct("simple_fueling_tag");
            tag->SetName("simple_fueling_tag");
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(tag, USER_ROOT_DEFAULT, newCar.Id, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        { // car has old_state_reservation tag and simple_fueling_tag
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));
            server->GetChatRobot("support")->Refresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["text"].GetString(), "Регистрация и авторизация");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[1]["message_text"].GetString(), "talk_with_human");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[2]["message_text"].GetString(), "talk_about_session_with_fueling");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[3]["message_text"].GetString(), "talk_about_session_any");
        }
    }

    Y_UNIT_TEST(ConditionUserTagFields) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        gServer.SetLogLevel(7);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-not-there222", true, "active");
        {
            TAtomicSharedPtr<TSimpleUserTag> tag = new TSimpleUserTag("simple_user_tag");
            tag->SetComment("comment_1");
            UNIT_ASSERT(gServer.AddTag(tag, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }
        {
            TAtomicSharedPtr<TSimpleUserTag> tag = new TSimpleUserTag("simple_user_tag");
            tag->SetComment("comment_2");
            UNIT_ASSERT(gServer.AddTag(tag, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_condition");
            INFO_LOG << messagesJson << Endl;
            UNIT_ASSERT(messagesJson["schema"].Has("options"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray().size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[0]["message_text"].GetString(), "talk_with_human");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[1]["message_text"].GetString(), "talk_about_tag_has_field_value");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[2]["message_text"].GetString(), "talk_about_tag_with_field");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["schema"]["options"].GetArray()[3]["message_text"].GetString(), "talk_about_tag_has_field_set");
        }
    }

    Y_UNIT_TEST(Stickers) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        size_t initSizeAdmin = 0;
        size_t initSizeClient = 0;
        size_t initSizeHistory = 0;
        {
            auto history = gServer.GetStickersHistory(USER_ROOT_DEFAULT);
            initSizeHistory = history["history"].GetArray().size();
        }
        {
            auto stickerList = gServer.GetChatStickersList(USER_ROOT_DEFAULT, false);
            UNIT_ASSERT(stickerList.Has("stickers"));
            initSizeClient = stickerList["stickers"].GetArray().size();
        }
        {
            auto stickerList = gServer.GetChatStickersList(USER_ROOT_DEFAULT, true);
            UNIT_ASSERT(stickerList.Has("stickers"));
            initSizeAdmin = stickerList["stickers"].GetArray().size();
        }
        UNIT_ASSERT(initSizeClient <= initSizeAdmin);

        TString url = "";
        for (size_t i = 0; i < 15; ++i) {
            url += 'a' + rand() % 26;
        }

        {
            NDrive::NChat::TSticker sticker;
            NJson::TJsonValue payload;

            payload["url"] = url;
            payload["emoji"] = "a";

            UNIT_ASSERT(sticker.ParseFromJson(payload));
            UNIT_ASSERT(gServer.AddChatSticker(sticker, USER_ROOT_DEFAULT));
        }
        UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        {
            auto stickerList = gServer.GetChatStickersList(USER_ROOT_DEFAULT, false);
            UNIT_ASSERT_VALUES_EQUAL(initSizeClient, stickerList["stickers"].GetArray().size());
        }
        TString id = "";
        {
            auto stickerList = gServer.GetChatStickersList(USER_ROOT_DEFAULT, true);
            INFO_LOG << stickerList << Endl;
            UNIT_ASSERT_VALUES_EQUAL(initSizeAdmin + 1, stickerList["stickers"].GetArray().size());
            for (auto&& elem : stickerList["stickers"].GetArray()) {
                if (elem["url"].GetString() == url) {
                    id = elem["id"].GetString();
                    UNIT_ASSERT_VALUES_EQUAL(elem["emoji"], "a");
                }
            }
        }
        UNIT_ASSERT(!!id);
        {
            NDrive::NChat::TSticker sticker;
            NJson::TJsonValue payload;
            payload["id"] = id;
            payload["is_active"] = true;
            payload["url"] = url;
            payload["emoji"] = "a";
            UNIT_ASSERT(sticker.ParseFromJson(payload));
            UNIT_ASSERT(gServer.EditChatSticker(sticker, USER_ROOT_DEFAULT));
        }
        UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        {
            auto stickerList = gServer.GetChatStickersList(USER_ROOT_DEFAULT, false);
            UNIT_ASSERT_VALUES_EQUAL(initSizeClient + 1, stickerList["stickers"].GetArray().size());
        }
        {
            auto stickerList = gServer.GetChatStickersList(USER_ROOT_DEFAULT, true);
            UNIT_ASSERT_VALUES_EQUAL(initSizeAdmin + 1, stickerList["stickers"].GetArray().size());
        }

        // Start dialogue
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        {
            UNIT_ASSERT(gServer.SendStickerToChat(userId, "support.123", "zxzxzx"));
            UNIT_ASSERT(gServer.SendStickerToChat(userId, "support.123", id));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            UNIT_ASSERT(chatsJson["chats"].IsArray());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray()[0]["last_message"]["emoji"], "a");
        }

        // Remove sticker
        {
            UNIT_ASSERT(gServer.RemoveChatSticker(id, USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }
        {
            auto stickerList = gServer.GetChatStickersList(USER_ROOT_DEFAULT, false);
            UNIT_ASSERT_VALUES_EQUAL(initSizeClient, stickerList["stickers"].GetArray().size());
        }
        {
            auto stickerList = gServer.GetChatStickersList(USER_ROOT_DEFAULT, true);
            UNIT_ASSERT_VALUES_EQUAL(initSizeAdmin, stickerList["stickers"].GetArray().size());
        }
        {
            auto history = gServer.GetStickersHistory(USER_ROOT_DEFAULT);
            INFO_LOG << history << Endl;
            UNIT_ASSERT_VALUES_EQUAL(initSizeHistory + 3, history["history"].GetArray().size());
        }
    }

    Y_UNIT_TEST(ResubmitDocumentsViaSupportChat) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        gServer.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());

        // And now check that Yang document verification assignment was created for this user
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = gServer.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["first_name"] = "сергей";
            updateJson["driving_license"]["first_name"] = "сергей";
            updateJson["first_name"] = "";
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(gServer.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentIds.front(), "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentIds.front()).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto userFetchResult = driveApi.GetUsersData()->FetchInfo(userId);
            auto userPtr = userFetchResult.GetResultPtr(userId);
            UNIT_ASSERT(!!userPtr);
            UNIT_ASSERT_VALUES_EQUAL(userPtr->GetStatus(), "active");
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "!new_driver_license", {}));
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license");
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "", {gServer.GetValidImageString(), gServer.GetValidImageString()}));
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["expected_action"].GetStringRobust());
        }

        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = gServer.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);

            eGenerator.CompleteYangAssignment(newAssignmentIds.front(), "iiioo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentIds.front()).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto session = gServer.GetCurrentSession(userId);
            UNIT_ASSERT_VALUES_EQUAL(session["user"]["show_chat"]["id"].GetString(), "support_modern.123");
            UNIT_ASSERT(!session["user"]["show_chat"]["is_closable"].GetBoolean());
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_front");
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "", {gServer.GetValidImageString()}));
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license_back");
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "", {gServer.GetValidImageString()}));
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_biographical");
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "", {gServer.GetValidImageString()}));
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["expected_action"].GetStringRobust());
        }
    }

    Y_UNIT_TEST(ResubmitSoftDontBlock) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        gServer.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());

        // And now check that Yang document verification assignment was created for this user
        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = gServer.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack);
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            updateJson["passport"]["first_name"] = "сергей";
            updateJson["driving_license"]["first_name"] = "сергей";
            updateJson["first_name"] = "";
            updateJson["passport"]["doc_value"] = "HB2352296";
            UNIT_ASSERT(gServer.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));
            driveApi.GetUsersData()->FetchInfo(userId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "ooooo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto userFetchResult = driveApi.GetUsersData()->FetchInfo(userId);
            auto userPtr = userFetchResult.GetResultPtr(userId);
            UNIT_ASSERT(!!userPtr);
            UNIT_ASSERT_VALUES_EQUAL(userPtr->GetStatus(), "active");
        }

        auto carId = eGenerator.CreateCar().Id;
        {
            UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple2"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        }
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({ carId }, {"simple2"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            auto userPermissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
            UNIT_ASSERT(gServer.StartTag(tag.GetTagId(), userId));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RefreshCache(Now()));
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "!new_driver_license", {}));
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "license");
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "", {gServer.GetValidImageString(), gServer.GetValidImageString()}));
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_C(messagesJson["expected_action"].IsNull(), TStringBuilder() << messagesJson["expected_action"].GetStringRobust());
        }

        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            eGenerator.CreateNewYangAssignments(server.Get());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            UNIT_ASSERT(driveApi.HasDocumentPhotosManager());
            const auto& docPhotoManager = driveApi.GetDocumentPhotosManager();
            TVector<TString> newAssignmentIds = gServer.GetNewAssignmentIdsByUserId(docPhotoManager, userId, NUserDocument::EType::LicenseBack, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(newAssignmentIds.size(), 1);
            TString newAssignmentId = newAssignmentIds.front();
            UNIT_ASSERT(!!newAssignmentId);

            eGenerator.CompleteYangAssignment(newAssignmentId, "iiioo", TYangDocumentVerificationAssignment::EFraudStatus::NotFraud);
            auto assignment = docPhotoManager.GetDocumentVerificationAssignments().FetchInfo(newAssignmentId).begin()->second;
            UNIT_ASSERT(server->GetUserRegistrationManager()->HandleVerifiedAssignment(assignment, "unittest", Now()));
        }

        {
            auto session = gServer.GetCurrentSession(userId);
            UNIT_ASSERT_VALUES_EQUAL(session["user"]["show_chat"]["id"].GetString(), "support_modern.123");
            UNIT_ASSERT(session["user"]["show_chat"]["is_closable"].GetBoolean());
        }
    }

    Y_UNIT_TEST(ParametrizedChatNodes) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        TEnvironmentGenerator::TCar newCar = eGenerator.CreateCar("kia_rio");

        TString objectId = newCar.Id;
        TString imei = newCar.IMEI;

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(imei);
        UNIT_ASSERT(gServer.WaitLocation(objectId));
        UNIT_ASSERT(gServer.WaitSensor(objectId, "mileage"));

        emulator->GetContext().SetFuelPercent(50);
        emulator->GetContext().TrySetEngineStarted(true);
        UNIT_ASSERT(gServer.WaitSensor(objectId, "fuel_level", "50"));

        TString offerId;
        {
            THolder<TStandartOffer> offer(new TStandartOffer);
            offer->SetObjectId(objectId).SetUserId(userId).SetDeadline(Now() + TDuration::Minutes(5));
            offer->SetChargableAccounts({ "card", "bonus" });
            offerId = offer->GetOfferId();
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer.Release(), nullptr)}));
        }
        UNIT_ASSERT(gServer.BookOffer(offerId, userId));
        UNIT_ASSERT(gServer.WaitStatus(objectId, "old_state_reservation", *server));

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "!no_carpets_noriding", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "ok", {}));
            auto messagesJson = gServer.GetChatMessages(userId, "support_variables.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "chat_closed");
            auto messages = messagesJson["messages"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(messages.back()["text"].GetString(), "We will find this carpet!");
        }
    }

    Y_UNIT_TEST(ResourceReusage) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support", "", {"cat.jpg"}));
            auto resourceId = gServer.SubmitChatResource(userId, "support", "some picture", "cat.jpg", "image/jpeg");
            server.Get()->GetChatRobotsManager()->GetChatRobotsMediaStorage()->ForceRefresh(Now());
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support");
            auto messagesArray = messagesJson["messages"].GetArray();
            auto lastMessage = messagesArray[messagesArray.size() - 1];
            UNIT_ASSERT_C(lastMessage.Has("content_types") && lastMessage["content_types"].IsArray(), TStringBuilder() << lastMessage.GetStringRobust());
            UNIT_ASSERT_VALUES_EQUAL(lastMessage["content_types"].GetArray()[0].GetString(), "image/jpeg");
            server.Get()->GetChatRobotsManager()->GetChatRobotsMediaStorage()->ForceRefresh(Now());
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support", "", {"cat.jpg"}));
            server.Get()->GetChatRobotsManager()->GetChatRobotsMediaStorage()->ForceRefresh(Now());
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support");
            auto messagesArray = messagesJson["messages"].GetArray();
            auto lastMessage = messagesArray[messagesArray.size() - 1];
            UNIT_ASSERT_C(lastMessage.Has("content_types") && lastMessage["content_types"].IsArray(), TStringBuilder() << lastMessage.GetStringRobust());
            UNIT_ASSERT_VALUES_EQUAL(lastMessage["content_types"].GetArray()[0].GetString(), "image/jpeg");
        }

        // Check that this regular resource is not available for other users
        auto userId2 = eGenerator.CreateUser("zxqfd555-was-there", true, "onboarding");
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId2, "support", "", {"cat.jpg"}));
            auto messagesJson = gServer.GetChatMessages(userId2, "support");
            auto messagesArray = messagesJson["messages"].GetArray();
            auto lastMessage = messagesArray[messagesArray.size() - 1];
            UNIT_ASSERT_C(lastMessage.Has("content_types") && lastMessage["content_types"].IsArray(), TStringBuilder() << lastMessage.GetStringRobust());
            UNIT_ASSERT_C(lastMessage["content_types"].GetArray()[0].GetString() != "image/jpeg", TStringBuilder() << lastMessage);
        }

        // Submit shared resource and check that it is available to different user
        {
            auto resourceId = gServer.SubmitChatResource(USER_ROOT_DEFAULT, "support", "some pictureeeeee", "pusheen.mp4", "video/mp4", true);
            UNIT_ASSERT(gServer.CommitChatAction(userId2, "support", "", {"pusheen.mp4"}));
            server.Get()->GetChatRobotsManager()->GetChatRobotsMediaStorage()->ForceRefresh(Now());
            auto messagesJson = gServer.GetChatMessages(userId2, "support");
            auto messagesArray = messagesJson["messages"].GetArray();
            auto lastMessage = messagesArray[messagesArray.size() - 1];
            UNIT_ASSERT_C(lastMessage.Has("content_types") && lastMessage["content_types"].IsArray(), TStringBuilder() << lastMessage.GetStringRobust());
            UNIT_ASSERT_VALUES_EQUAL(lastMessage["content_types"].GetArray()[0].GetString(), "video/mp4");
        }

    }

    Y_UNIT_TEST(PhotoStatusesHookAction) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        gServer.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "!photos_test", {}));
        }

        {
            auto mapping = driveApi.GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(
                userId,
                {
                    NUserDocument::EType::LicenseFront,
                    NUserDocument::EType::LicenseBack,
                    NUserDocument::EType::PassportBiographical,
                    NUserDocument::EType::PassportRegistration,
                    NUserDocument::EType::PassportSelfie
                }
            );
            UNIT_ASSERT_VALUES_EQUAL(mapping[NUserDocument::EType::LicenseFront].GetVerificationStatus(), NUserDocument::EVerificationStatus::NeedInfo);
            UNIT_ASSERT_VALUES_EQUAL(mapping[NUserDocument::EType::LicenseBack].GetVerificationStatus(), NUserDocument::EVerificationStatus::Unrecognizable);
            UNIT_ASSERT_VALUES_EQUAL(mapping[NUserDocument::EType::PassportBiographical].GetVerificationStatus(), NUserDocument::EVerificationStatus::Foreign);
            UNIT_ASSERT_VALUES_EQUAL(mapping[NUserDocument::EType::PassportRegistration].GetVerificationStatus(), NUserDocument::EVerificationStatus::NonLatin);
            UNIT_ASSERT_VALUES_EQUAL(mapping[NUserDocument::EType::PassportSelfie].GetVerificationStatus(), NUserDocument::EVerificationStatus::VideoError);
        }
    }

    Y_UNIT_TEST(Macroses) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        TInstantGuard ig(TInstant::Seconds(1586034000));
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "!macroses_test", {}));
        }
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_variables.123");
            auto messagesArray = messagesJson["messages"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 4]["text"].GetString(), "skulik-was-there");
            UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 3]["text"].GetString(), "");
            UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 2]["text"].GetString(), "abcd@efgh.ij");
        }
        {
            auto userFetchResult = driveApi.GetUsersData()->FetchInfo(userId);
            auto userPtr = userFetchResult.GetResultPtr(userId);
            UNIT_ASSERT(!!userPtr);
            auto userData = *userPtr;
            userData.SetFirstName("Sergey");
            userData.SetPhone("+79165556677");
            userData.SetEmail("zxzxzx@yandex.ru");
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetUsersData()->UpdateUser(userData, "robot-frontend", session) && session.Commit());
        }
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, { "user_simple1" }, dbTags, session));
                UNIT_ASSERT_EQUAL(dbTags.size(), 1);
                UNIT_ASSERT_STRINGS_EQUAL(dbTags[0]->GetComment(), "04:00\u00A05\u00A0апреля[21:00\u00A011\u00A0апреля 21:00\u00A04\u00A0апреля");
            }
        }
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_variables.123");
            auto messagesArray = messagesJson["messages"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 4]["text"].GetString(), "Sergey");
            UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 3]["text"].GetString(), "+79165556677");
            UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 2]["text"].GetString(), "zxzxzx@yandex.ru");
        }
    }

    Y_UNIT_TEST(TagModifications) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);
        script.Add<NDrive::NTest::TBuildEnv>().SetNeedTelematics(false);

        TGeoCoord from(37.5848674, 55.7352435);
        script.Add<NDrive::NTest::TCreateCar>().SetPosition(from);
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCreateAndBookOffer>().SetOfferName("standart_offer_constructor").SetUserPosition(from);
        script.Add<NDrive::NTest::TAccept>();

        script.Add<NDrive::NTest::TListTagPropositions>();
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT_VALUES_EQUAL(context.GetPropositionIds().size(), 0);
        });

        script.Add<NDrive::NTest::TCheckHasTag>().SetObjectId(USER_ID_DEFAULT).SetEntityType(NEntityTagsManager::EEntityType::User).SetTagName("user_simple1").SetExpectOK(false);

        script.Add<NDrive::NTest::TCommitChatAction>("support_variables.add", "!tag_add");
        script.Add<NDrive::NTest::TCheckHasTag>().SetObjectId(USER_ID_DEFAULT).SetEntityType(NEntityTagsManager::EEntityType::User).SetTagName("user_simple1").SetTargetCount(1);

        script.Add<NDrive::NTest::TCommitChatAction>("support_variables.update", "!tag_update");
        script.Add<NDrive::NTest::TCheckHasTag>().SetObjectId(USER_ID_DEFAULT).SetEntityType(NEntityTagsManager::EEntityType::User).SetTagName("user_simple1").SetTargetCount(1);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(context.GetDriveAPI().GetTagsManager().GetUserTags().RestoreTags({ context.GetUserId() }, { "user_simple1" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT(dbTags.front().HasData());
            UNIT_ASSERT_VALUES_EQUAL(dbTags.front()->GetComment(), "new comment");
        });


        script.Add<NDrive::NTest::TCommitChatAction>("support_variables.evolve", "!tag_evolve");
        script.Add<NDrive::NTest::TCheckHasTag>().SetObjectId(USER_ID_DEFAULT).SetEntityType(NEntityTagsManager::EEntityType::User).SetTagName("user_simple1").SetExpectOK(false);
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(context.GetDriveAPI().GetTagsManager().GetUserTags().RestoreTags({ context.GetUserId() }, { "user_simple2" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT(dbTags.front().HasData());
            UNIT_ASSERT_VALUES_EQUAL(dbTags.front()->GetTagPriority(0), 1000);
        });

        script.Add<NDrive::NTest::TCommitChatAction>("support_variables.delete", "!tag_remove");
        script.Add<NDrive::NTest::TCheckHasTag>().SetObjectId(USER_ID_DEFAULT).SetEntityType(NEntityTagsManager::EEntityType::User).SetTagName("user_simple1").SetExpectOK(false);
        script.Add<NDrive::NTest::TCheckHasTag>().SetObjectId(USER_ID_DEFAULT).SetEntityType(NEntityTagsManager::EEntityType::User).SetTagName("user_simple2").SetExpectOK(false);

        script.Add<NDrive::NTest::TCommitChatAction>("support_variables.propose", "!tag_propose");
        script.Add<NDrive::NTest::TListTagPropositions>();
        script.Add<NDrive::NTest::TCommonChecker>([](NDrive::NTest::TRTContext& context) {
            UNIT_ASSERT_VALUES_EQUAL(context.GetPropositionIds().size(), 1);
        });
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(DefaultEmails) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "!default_emails_test", {}));
        UNIT_ASSERT(server->GetSettings().SetValue("handlers.default.fake_auth.default_emails", "skulik@yandex.by,skulik@yandex.ru,skulik@google.by", USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_variables.123");
            auto messagesArray = messagesJson["messages"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 1]["text"].GetString(), "Выберете email");

            auto optionsArray = messagesJson["schema"]["options"].GetArray();
            UNIT_ASSERT_VALUES_EQUAL(optionsArray.size(), 4);
            UNIT_ASSERT_VALUES_EQUAL(optionsArray[0]["text"].GetString(), "abcd@efgh.ij");
            UNIT_ASSERT_VALUES_EQUAL(optionsArray[1]["text"].GetString(), "skulik@google.by");
            UNIT_ASSERT_VALUES_EQUAL(optionsArray[2]["text"].GetString(), "skulik@yandex.ru");
            UNIT_ASSERT_VALUES_EQUAL(optionsArray[3]["text"].GetString(), "Нет");
        }
    }

    Y_UNIT_TEST(EndpointContext) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        {
            auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
            UNIT_ASSERT(gServer.AddTag(new TSimpleUserTag(TSimpleUserTag::TypeName), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.1", "!remove_tag_action", {}));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            {
                auto messagesJson = gServer.GetChatMessages(userId, "support_variables.1");
                auto messagesArray = messagesJson["messages"].GetArray();
                UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 1]["text"].GetString(), "no permissions");
            }
        }
        {
            auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
            UNIT_ASSERT(gServer.AddTag(new TSimpleUserTag(TSimpleUserTag::TypeName), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));

            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.2", "!remove_tag_action_path", {}));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            {
                auto messagesJson = gServer.GetChatMessages(userId, "support_variables.2");
                auto messagesArray = messagesJson["messages"].GetArray();
                UNIT_ASSERT_VALUES_EQUAL(messagesArray[messagesArray.size() - 1]["text"].GetString(), "no permissions");
            }
        }
    }

    Y_UNIT_TEST(ThroughNodes) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "!no_sts_noriding_past", {}));
        }

        // Check that tag is NOT created
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "zzzz", {}));
        }

        // Check that tag is created
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_lowpriority"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }
    }

    Y_UNIT_TEST(BadState) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "!no_sts_noriding_past", {}));
        }

        {
            auto chatRobot = server->GetChatRobot("support_variables");
            TString stateKey = userId + "-support_variables-123";
            auto serializedState = server->GetChatRobotsManager()->GetChatRobotStateStorage()->GetSerializedState(stateKey, Now());
            UNIT_ASSERT(serializedState);

            NChatRobotState::TChatRobotState state;
            UNIT_ASSERT(state.ParseFromString(Base64Decode(*serializedState)));
            state.SetCurrentStep("this step does not exist hahaha");
            auto session = server->GetChatEngine()->BuildSession();
            UNIT_ASSERT(server->GetChatRobotsManager()->GetChatRobotStateStorage()->SetSerializedState(stateKey, Base64Encode(state.SerializeAsString()), "", session));
            UNIT_ASSERT(session.Commit());
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_variables.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "user_message");
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "abcde", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "abcde", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "abcde", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "abcde", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "abcde", {}));
        }
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "test note",  {}, 0, 0, {NDrive::NChat::TMessage::EMessageTraits::StaffOnly}, USER_ROOT_DEFAULT));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "test note",  {}, 0, 0, {}, USER_ROOT_DEFAULT));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "test note",  {}, 0, 0, {}, USER_ROOT_DEFAULT));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "test note",  {}, 0, 0, {}, USER_ROOT_DEFAULT));
        }
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "abcde", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_variables.123", "abcde", {}));
        }
    }

    Y_UNIT_TEST(ChatAdminFeed) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto clientId = eGenerator.CreateUser("skulik-was-there", true, "active");
        auto clientId2 = eGenerator.CreateUser("zxqfd555-was-there", true, "active");

        TVector<TString> supportChatTagNames;
        {
            auto td = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TSupportChatTag::TypeName);
            for (auto&& i : td) {
                supportChatTagNames.emplace_back(i->GetName());
            }
        }

        // Clear feed
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, supportChatTagNames, dbTags, session));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(dbTags, USER_ROOT_DEFAULT, server.Get(), session, true) && session.Commit());

            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        // Get empty feed
        {
            auto feedJson = gServer.GetChatsFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
        }

        // Create some chat, requiring attention
        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support", "Test #1", {}));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        // Get feed with urgent chat
        {
            {
                auto feedJson = gServer.GetChatsFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("", "0", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_2_line", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_2_line,support_chat_dtp", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_2_line", "0", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_dtp", "0", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_dtp", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
        }

        // Start processing request, but don't respond anything
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_2_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            auto userPermissions = server->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
            UNIT_ASSERT(gServer.StartTag(tag.GetTagId(), USER_ROOT_DEFAULT));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

       // Get feed with urgent chat
        {
            {
                auto feedJson = gServer.GetChatsFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("", "0", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_2_line", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_2_line,support_chat_dtp", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_2_line", "0", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_dtp", "0", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_dtp", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("", USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_2_line", USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
            {
                auto feedJson = gServer.GetChatsFeed("support_chat_dtp", USER_ROOT_DEFAULT, USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
                UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            }
        }

        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support", "test note",  {}, 0, 0, {NDrive::NChat::TMessage::EMessageTraits::StaffOnly}, USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));

        {
            auto feedJson = gServer.GetChatsFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
        }

        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support", "hello",  {}, 0, 0, {}, USER_ROOT_DEFAULT));
        UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        {
            auto feedJson = gServer.GetChatsFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 1);
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId2, "support", "Test #1", {}));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
            Sleep(TDuration::Seconds(1));
        }
        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support", "Test #1", {}));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
            Sleep(TDuration::Seconds(1));
        }
        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId2, "support", "Test #1", {}));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        // Both are urgent and clientId2 is ranked higher than clientId
        {
            auto feedJson = gServer.GetChatsFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray()[0]["originator"].GetString(), clientId2);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray()[1]["originator"].GetString(), clientId);
        }
    }

    bool CheckFeedChatsCount(const NJson::TJsonValue& feedJson, const TString& userId, TSet<TString>& tagIds, ui32 targetCount, TString& error) {
        ui32 count = 0;
        for (auto i : feedJson["container"].GetArray()) {
            if (i["object_id"].GetString() == userId) {
                if (!tagIds.contains(i["tag_id"].GetString())) {
                    error = "bad tag id found";
                    return false;
                }
                ++count;
            }
        }
        if (count != targetCount) {
            error = "wrong user chats count: " + ToString(count);
            return false;
        }
        return true;
    }

    Y_UNIT_TEST(ChatAdminFeedContainer) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-not-there", true, "active");

        TString outgoingTagId;
        UNIT_ASSERT(gServer.AddTag(new TSupportOutgoingCommunicationTag("test_outgoing_communication"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        UNIT_ASSERT(gServer.GetTagId(userId, "test_outgoing_communication", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, outgoingTagId));

        TString dtpTagId;
        {
            TAtomicSharedPtr<TSupportChatTag> tag = new TSupportChatTag("support_chat_dtp");
            tag->SetTopicLink("support.123");
            tag->SetChatTitle("custom chat title");
            UNIT_ASSERT(gServer.AddTag(tag, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }
        UNIT_ASSERT(gServer.GetTagId(userId, "support_chat_dtp", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, dtpTagId));

        TString supportLine2TagId;
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support.234", "Hello, Yandex.Drive!", {}));
        UNIT_ASSERT(gServer.GetTagId(userId, "support_chat_2_line", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, supportLine2TagId));

        TString lowTagId;
        {
            TAtomicSharedPtr<TSupportChatTag> tag = new TSupportChatTag("support_chat_lowpriority");
            tag->SetTopicLink("support.456");
            tag->SetChatTitle("custom chat title");
            UNIT_ASSERT(gServer.AddTag(tag, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }
        UNIT_ASSERT(gServer.GetTagId(userId, "support_chat_lowpriority", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User, lowTagId));

        UNIT_ASSERT(gServer.DeferRequest(dtpTagId, USER_ROOT_DEFAULT, Now() + TDuration::Hours(10)));
        UNIT_ASSERT(gServer.DeferRequest(outgoingTagId, USER_ROOT_DEFAULT, Now() + TDuration::Hours(10)));
        UNIT_ASSERT(gServer.DeferRequest(supportLine2TagId, USER_ROOT_DEFAULT, Now() + TDuration::Hours(10)));
        UNIT_ASSERT(gServer.DeferRequest(lowTagId, USER_ROOT_DEFAULT, Now() + TDuration::Hours(10)));
        {
            TVector<TDBTag> tags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TSupportChatTag::DeferredContainerName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 3);
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({TUserContainerTag::TypeName}), tags, session));
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
        }
        TString error;
        {
            TSet<TString> tagIds = { dtpTagId, outgoingTagId, supportLine2TagId, lowTagId };
            auto feedJson = gServer.GetFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User), {"container"});
            UNIT_ASSERT_C(feedJson["container"].GetArray().size() >= 4, feedJson["container"].GetArray().size());
            UNIT_ASSERT_C(CheckFeedChatsCount(feedJson, userId, tagIds, 4, error), error);
        }
        {
            TSet<TString> tagIds = { outgoingTagId, lowTagId };
            auto feedJson = gServer.GetFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User), {"container"}, "contained_tag_names=support_chat_lowpriority,test_outgoing_communication");
            UNIT_ASSERT_C(feedJson["container"].GetArray().size() >= 2, feedJson["container"].GetArray().size());
            UNIT_ASSERT_C(CheckFeedChatsCount(feedJson, userId, tagIds, 2, error), error);
        }
        {
            TSet<TString> tagIds = { dtpTagId, supportLine2TagId, lowTagId };
            auto feedJson = gServer.GetFeed(TSupportChatTag::DeferredContainerName, "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User), {"container"});
            UNIT_ASSERT_C(feedJson["container"].GetArray().size() >= 3, feedJson["container"].GetArray().size());
            UNIT_ASSERT_C(CheckFeedChatsCount(feedJson, userId, tagIds, 3, error), error);
        }
        {
            TSet<TString> tagIds = { dtpTagId, lowTagId };
            auto feedJson = gServer.GetFeed(TSupportChatTag::DeferredContainerName, "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User), {"container"}, "contained_tag_names=support_chat_lowpriority,support_chat_dtp");
            UNIT_ASSERT_C(feedJson["container"].GetArray().size() >= 2, feedJson["container"].GetArray().size());
            UNIT_ASSERT_C(CheckFeedChatsCount(feedJson, userId, tagIds, 2, error), error);
        }
        {
            TSet<TString> tagIds = { dtpTagId, outgoingTagId, supportLine2TagId, lowTagId };
            auto feedJson = gServer.GetFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User), {"container"}, "contained_tag_names=test_outgoing_communication&contained_tag_types=user_support_chat_tag");
            UNIT_ASSERT_C(feedJson["container"].GetArray().size() >= 4, feedJson["container"].GetArray().size());
            UNIT_ASSERT_C(CheckFeedChatsCount(feedJson, userId, tagIds, 4, error), error);
        }
        {
            TSet<TString> tagIds = { dtpTagId, supportLine2TagId, lowTagId };
            auto feedJson = gServer.GetFeed("", "", USER_ROOT_DEFAULT, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User), {"container"}, "contained_tag_types=user_support_chat_tag");
            UNIT_ASSERT_C(feedJson["container"].GetArray().size() >= 3, feedJson["container"].GetArray().size());
            UNIT_ASSERT_C(CheckFeedChatsCount(feedJson, userId, tagIds, 3, error), error);
        }
    }

    Y_UNIT_TEST(TwoPhaseResubmit) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        gServer.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        UNIT_ASSERT(driveApi.HasDocumentPhotosManager());

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "!got_new_passport", {}));
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport");
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "", {gServer.GetValidImageString(), gServer.GetValidImageString()}));
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["type"].GetString(), "passport_selfie");
        }

        auto mapping = server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(userId, { NUserDocument::EType::PassportSelfie });
        UNIT_ASSERT_VALUES_EQUAL(mapping[NUserDocument::EType::PassportSelfie].GetVerificationStatus(), NUserDocument::EVerificationStatus::NeedInfo);
    }

    Y_UNIT_TEST(SwitchNextStep) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_switch.1", "hello", {}));
            auto messagesJson = gServer.GetChatMessages(userId, "support_switch.1");
            auto messagesArray = messagesJson["messages"].GetArray();
            auto lastMessage = messagesArray[messagesArray.size() - 1];
            UNIT_ASSERT_VALUES_EQUAL(lastMessage["text"].GetString(), "You are onboarding");
        }

        auto userId2 = eGenerator.CreateUser("zxqfd555-was-there", true, "active");
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId2, "support_switch.1", "hello", {}));
            auto messagesJson = gServer.GetChatMessages(userId2, "support_switch.1");
            auto messagesArray = messagesJson["messages"].GetArray();
            auto lastMessage = messagesArray[messagesArray.size() - 1];
            UNIT_ASSERT_VALUES_EQUAL(lastMessage["text"].GetString(), "You are active");
        }

        auto userId3 = eGenerator.CreateUser("zxqfd-was-there", true, "screening");
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId3, "support_switch.1", "hello", {}));
            auto messagesJson = gServer.GetChatMessages(userId3, "support_switch.1");
            auto messagesArray = messagesJson["messages"].GetArray();
            auto lastMessage = messagesArray[messagesArray.size() - 1];
            UNIT_ASSERT_VALUES_EQUAL(lastMessage["text"].GetString(), "You are neither onboarding nor active");
        }
    }

    Y_UNIT_TEST(PromoEligibility) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "!do_i_participate", {}, 55.75222, 37.61556));
            auto messagesJson = gServer.GetChatMessages(userId, "promo");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["text"], "Эх");
            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "", {}, 55.75222, 37.61556));
        }

        {
            auto messagesJson = gServer.GetChatMessages(userId, "promo");
            size_t messagesBeforeCnt = messagesJson["messages"].GetArray().size();

            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "!whats_my_balance", {}));
            messagesJson = gServer.GetChatMessages(userId, "promo");

            size_t messagesAfterCnt = messagesJson["messages"].GetArray().size();
            UNIT_ASSERT_VALUES_EQUAL(messagesBeforeCnt + 2, messagesAfterCnt);
        }

        auto userId2 = eGenerator.CreateUser("skulik-was-there2", true, "active");
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId2, "promo", "!do_i_participate", {}, 55.75222, 37.61556));
            auto messagesJson = gServer.GetChatMessages(userId2, "promo");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["expected_action"]["text"], "Ура");
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "", {}, 55.75222, 37.61556));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "!want_to_opt_out", {}, 55.75222, 37.61556));
            {
                TVector<TDBTag> tags;
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>({userId}), TVector<TString>({"documents_stuck"}), tags, session));
                UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            }
        }
    }

    Y_UNIT_TEST(DisableNotifications) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "!disable_notifications", {}, 55.75222, 37.61556));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "", {}, 55.75222, 37.61556));
        }

        size_t requiredUnreads;
        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            requiredUnreads = chatsJson["total_unread"].GetUInteger();
            UNIT_ASSERT(gServer.SendArbitraryMessage("promo", userId, "Hello, skulik-was-there!"));
            UNIT_ASSERT(gServer.SendArbitraryMessage("promo", userId, "It seems, you haven't sent any cat picture to me yet :("));
        }

        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            size_t unreads = chatsJson["total_unread"].GetUInteger();
            UNIT_ASSERT_VALUES_EQUAL(unreads, requiredUnreads);
        }
    }

    Y_UNIT_TEST(HideFromList) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");

        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }

        size_t requiredUnreads;
        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            requiredUnreads = chatsJson["total_unread"].GetUInteger();
            UNIT_ASSERT(gServer.SendArbitraryMessage("promo", userId, "Hello, skulik-was-there!"));
            UNIT_ASSERT(gServer.SendArbitraryMessage("promo", userId, "It seems, you haven't sent any cat picture to me yet :("));
        }

        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "!hide_from_list", {}, 55.75222, 37.61556));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "promo", "", {}, 55.75222, 37.61556));
        }

        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            size_t unreads = chatsJson["total_unread"].GetUInteger();
            UNIT_ASSERT_VALUES_EQUAL(unreads, requiredUnreads);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }

        {
            UNIT_ASSERT(gServer.AlterChatFlag(userId, "promo", NDrive::NChat::TChat::EChatFlags::Hidden, false));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            size_t unreads = chatsJson["total_unread"].GetUInteger();
            UNIT_ASSERT_VALUES_EQUAL(unreads, requiredUnreads);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 1);
        }

        {
            UNIT_ASSERT(gServer.AlterChatFlag(userId, "promo", NDrive::NChat::TChat::EChatFlags::Hidden, true));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            auto chatsJson = gServer.GetChatsList(userId, server.Get());
            size_t unreads = chatsJson["total_unread"].GetUInteger();
            UNIT_ASSERT_VALUES_EQUAL(unreads, requiredUnreads);
            UNIT_ASSERT_VALUES_EQUAL(chatsJson["chats"].GetArray().size(), 0);
        }
    }

    Y_UNIT_TEST(MutedChats) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        auto operatorId = eGenerator.CreateUser("operator-was-there", true, "active");
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(operatorId).SetActive(true);
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, operatorId, session) && session.Commit());
        }

        // Clear feed
        TVector<TString> supportChatTagNames;
        {
            auto td = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TSupportChatTag::TypeName);
            for (auto&& i : td) {
                supportChatTagNames.emplace_back(i->GetName());
            }
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, supportChatTagNames, dbTags, session));
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(dbTags, USER_ROOT_DEFAULT, server.Get(), session, true) && session.Commit());

            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support.1", "hello", {}));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            auto feedJson = gServer.GetChatsFeed("", "", operatorId, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            auto userPermissions = server->GetDriveAPI()->GetUserPermissions(operatorId, TUserPermissionsFeatures());
            UNIT_ASSERT(gServer.StartTag(tag.GetTagId(), operatorId));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(TSupportChatTag::ChangeMutedStatus(tag, true, operatorId, server.Get(), session) && session.Commit());

            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        // Muted are not visible in urgent section
        {
            auto feedJson = gServer.GetChatsFeed("", "", operatorId, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 1);
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            UNIT_ASSERT_VALUES_EQUAL(dbTags[0]->GetPerformer(), operatorId);
        }

        // Operator messages does't change muted status
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support.1", "hello", {}, 0, 0, {}, operatorId));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));

            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }
        {
            auto feedJson = gServer.GetChatsFeed("", "", operatorId, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 0);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 1);
        }
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            UNIT_ASSERT(tag.GetTagAs<TSupportChatTag>()->IsMuted());
        }

        // User's messages change muted status
        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support.1", "hello", {}));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }
        {
            auto feedJson = gServer.GetChatsFeed("", "", operatorId, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["urgent"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(feedJson["chats"]["answered"].GetArray().size(), 0);
        }
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            UNIT_ASSERT(!tag.GetTagAs<TSupportChatTag>()->IsMuted());
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(TSupportChatTag::ChangeMutedStatus(tag, true, operatorId, server.Get(), session) && session.Commit());
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        // Evolution drops muted status
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto permissions = server->GetDriveAPI()->GetUserPermissions(operatorId, TUserPermissionsFeatures());
            auto tag = dbTags.front();
            auto newTag = tag.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
            newTag->SetName("support_chat_1_line");
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().EvolveTag(tag, newTag.GetData(), *permissions, server.Get(), session) && session.Commit());
        }
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_1_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            UNIT_ASSERT(!tag.GetTagAs<TSupportChatTag>()->IsMuted());
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(TSupportChatTag::ChangeMutedStatus(tag, true, operatorId, server.Get(), session) && session.Commit());
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        // Drop performer drops muted property
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_1_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto permissions = server->GetDriveAPI()->GetUserPermissions(operatorId, TUserPermissionsFeatures());
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().DropPerformer({}, {dbTags[0].GetTagId()}, *permissions, server.Get(), session) && session.Commit());
        }
        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_1_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            UNIT_ASSERT(!tag.GetTagAs<TSupportChatTag>()->IsMuted());
        }
    }

    Y_UNIT_TEST(OutgoingChats) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        // No tags and no chats initially
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, { "support_chat_dtp", "support_chat_2_line" }, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 0);

            auto chatsList = gServer.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsList["chats"].GetArray().size(), 0);
        }

        // Add tag with communication
        {
            TAtomicSharedPtr<TSupportChatTag> tag = new TSupportChatTag("support_chat_dtp");
            tag->SetTopicLink("support_modern.123");
            tag->SetInitialNode("no_fuel_noriding_no_button");
            tag->SetComment("hello world!");
            tag->SetChatTitle("custom chat title");
            UNIT_ASSERT(gServer.AddTag(tag, userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        // Check that chat appeared
        {
            auto chatsList = gServer.GetChatsList(userId, server.Get());
            INFO_LOG << chatsList.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(chatsList["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsList["chats"].GetArray()[0]["id"].GetString(), "support_modern.123");
            UNIT_ASSERT_VALUES_EQUAL(chatsList["chats"].GetArray()[0]["name"].GetString(), "custom chat title");
        }

        // Check hidden comment
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            size_t msgUser = messagesJson["messages"].GetArray().size();
            messagesJson = gServer.GetChatMessages(userId, "support_modern.123", USER_ROOT_DEFAULT);
            size_t msgAdmin = messagesJson["messages"].GetArray().size();
            UNIT_ASSERT_VALUES_EQUAL(msgUser + 1, msgAdmin);
        }

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "hello", {}));

        // Check that still one tag and one chat
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_dtp", "support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            auto chatsList = gServer.GetChatsList(userId, server.Get());
            UNIT_ASSERT_VALUES_EQUAL(chatsList["chats"].GetArray().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(chatsList["chats"].GetArray()[0]["id"].GetString(), "support_modern.123");
        }

        // Check hidden comment is still hidden
        {
            auto messagesJson = gServer.GetChatMessages(userId, "support_modern.123");
            size_t msgUser = messagesJson["messages"].GetArray().size();
            messagesJson = gServer.GetChatMessages(userId, "support_modern.123", USER_ROOT_DEFAULT);
            size_t msgAdmin = messagesJson["messages"].GetArray().size();
            UNIT_ASSERT_VALUES_EQUAL(msgUser + 1, msgAdmin);
        }
    }

    Y_UNIT_TEST(ForceApproveByChat) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");

        UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.123", "!approve_me", {}));

        {
            auto userFR = server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
            UNIT_ASSERT_VALUES_EQUAL(userFR.size(), 1);
            auto user = std::move(userFR.begin()->second);
            UNIT_ASSERT_VALUES_EQUAL(user.GetStatus(), "active");
        }
    }

    Y_UNIT_TEST(ChatRobotCarModelMacros) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto chatId = "support_modern.chatrobotcarmodeltest123";
        auto userId = eGenerator.CreateUser("user_chatrobotcarmodel_test", true, "active");

        auto session = driveApi.BuildTx<NSQL::Writable>();
        TEnvironmentGenerator::TCar newCar = eGenerator.CreateCar("kia_rio");

        TString objectId = newCar.Id;

        TString imei = newCar.IMEI;
        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        auto emulator = tmBuilder.BuildEmulator(imei);
        UNIT_ASSERT(gServer.WaitLocation(objectId));
        UNIT_ASSERT(gServer.WaitSensor(objectId, "mileage"));

        emulator->GetContext().SetFuelPercent(50);
        emulator->GetContext().TrySetEngineStarted(true);
        UNIT_ASSERT(gServer.WaitSensor(objectId, "fuel_level", "50"));

        TString offerId;
        {
            THolder<TStandartOffer> offer(new TStandartOffer);
            offer->SetObjectId(objectId).SetUserId(userId).SetDeadline(Now() + TDuration::Minutes(5));
            offer->SetChargableAccounts({ "card", "bonus" });
            offerId = offer->GetOfferId();
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer.Release(), nullptr)}));
        }
        UNIT_ASSERT(gServer.BookOffer(offerId, userId));
        UNIT_ASSERT(gServer.WaitStatus(objectId, "old_state_reservation", *server));

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, chatId, "!show_car_info", {}));
        }

        {
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));
            auto messagesJson = gServer.GetChatMessages(userId, chatId);
            auto messages = messagesJson["messages"].GetArray();
            UNIT_ASSERT_EQUAL(messages.back()["text"].GetString(), "kia_rio - KIA Rio - " + newCar.Number);
        }

        UNIT_ASSERT(gServer.EvolveTag("old_state_acceptance", userId));
        UNIT_ASSERT(gServer.WaitStatus(objectId, "old_state_acceptance", *server));
        {
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RefreshCache(Now()));
            auto messagesJson = gServer.GetChatMessages(userId, chatId);
            auto messages = messagesJson["messages"].GetArray();
            UNIT_ASSERT_EQUAL(messages.back()["text"].GetString(), "kia_rio - KIA Rio - " + newCar.Number);
        }
    }

    Y_UNIT_TEST(ChatTagEvolutionMeta) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-not-there", true, "active");
        NJson::TJsonValue meta = NJson::JSON_MAP;
        meta["field1"] = "value1";
        meta["field2"] = "value2";
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            THolder<TSupportChatTag> tagData(new TSupportChatTag());
            tagData->SetName("support_chat_2_line");
            tagData->SetMeta(meta);
            tagData->SetExternalChatInfo(TSupportChatTag::TExternalChatInfo("id1", "provider_name"));
            tagData->SetTopicLink("support.182839327718292393");
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tagData.Release(), USER_ROOT_DEFAULT, userId, server.Get(), session) && session.Commit());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));

            auto permissions = server->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());

            auto tag = dbTags.front();
            THolder<TSupportChatTag> newTag(new TSupportChatTag());
            newTag->SetName("support_chat_1_line");

            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().EvolveTag(tag, newTag.Release(), *permissions, server.Get(), session) && session.Commit());
        }
        {
            TVector<TDBTag> dbTags;
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_1_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto tagImpl = dbTags[0].GetTagAs<TSupportChatTag>();
            UNIT_ASSERT(tagImpl);
            UNIT_ASSERT_VALUES_EQUAL(tagImpl->GetMeta(), meta);
            UNIT_ASSERT_VALUES_EQUAL(tagImpl->GetExternalChatInfo().GetId(), "id1");
            UNIT_ASSERT_VALUES_EQUAL(tagImpl->GetExternalChatInfo().GetProvider(), "provider_name");
        }
    }

    Y_UNIT_TEST(AuthorName) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        UNIT_ASSERT(server->GetSettings().SetValue("support.report_author_name", "true", USER_ROOT_DEFAULT));
        auto userId = eGenerator.CreateUser("skulik-was-there", true, "active");
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(userId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, userId, session) && session.Commit());
        }

        auto clientId = eGenerator.CreateUser("zxqfd555-was-there", true, "active");
        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.zcvvczzxcvzvczx", "Hello, Yandex.Drive!", {}));
        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.zcvvczzxcvzvczx", "Hello, client!",  {}, 0, 0, {}, USER_ROOT_DEFAULT));
        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_2_line" });
        }

        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.zcvvczzxcvzvczx", "What's up?", {}));
        UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.zcvvczzxcvzvczx", "I'm fine thank you",  {}, 0, 0, {}, USER_ROOT_DEFAULT));

        {
            auto messagesJson = gServer.GetChatMessages(clientId, "support.zcvvczzxcvzvczx");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[0]["author_name"], "rbt");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[1]["author_name"], "rbt");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[2]["author_name"], "rbt");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[3]["author_name"], "rbt");
            UNIT_ASSERT(!messagesJson["messages"].GetArray()[4].Has("author_name"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 9);
            UNIT_ASSERT(!messagesJson["messages"].GetArray()[5].Has("author_name"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[6]["author_name"], "Мария");
            UNIT_ASSERT(!messagesJson["messages"].GetArray()[7].Has("author_name"));
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[8]["author_name"], "Мария");
        }
    }

    Y_UNIT_TEST(MessageRating) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-not-there", true, "active");
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Hello, Yandex.Drive!", {}));
        UNIT_ASSERT(configGenerator.CommitChatAction(userId, "support", "Your support chat is really awesome!", {}));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 6);
            UNIT_ASSERT(configGenerator.SendArbitraryMessage("support", userId, "Hello, skulik-was-not-there!"));
        }

        TVector<i64> orderedMessageIds;
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support", USER_ROOT_DEFAULT);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            for (auto&& message : messagesJson["messages"].GetArray()) {
                orderedMessageIds.push_back(message["id"].GetInteger());
            }
        }
        UNIT_ASSERT(configGenerator.RateChatMessage(userId, "support", orderedMessageIds[2], "awesome"));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[2]["rating"], "awesome");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[2]["author"], "robot-frontend");
        }
        UNIT_ASSERT(configGenerator.RateChatMessage(userId, "support", orderedMessageIds[2], "not awesome anymore"));
        UNIT_ASSERT(configGenerator.RateChatMessage(userId, "support", orderedMessageIds[5], "good"));
        {
            auto messagesJson = configGenerator.GetChatMessages(userId, "support");
            UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray().size(), 7);
            for (size_t i = 0; i < messagesJson["messages"].GetArray().size(); ++i) {
                if (i == 2) {
                    UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[i]["rating"], "not awesome anymore");
                    UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[i]["author"], "robot-frontend");
                } else if (i == 5) {
                    UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[i]["rating"], "good");
                    UNIT_ASSERT_VALUES_EQUAL(messagesJson["messages"].GetArray()[i]["author"], userId);
                } else {
                    UNIT_ASSERT(!messagesJson["messages"].GetArray()[i].Has("rating"));
                }
            }
        }
    }

    Y_UNIT_TEST(PerformAfterUndefer) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("steap-was-here", true, "active");
        auto operatorId = eGenerator.CreateUser("operator-was-here", true, "active");

        // 'support_chat_performed' doesn't drop performer on defer
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("support_chat_performed", Now()).size(), 0);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(operatorId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, operatorId, session) && session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees("support_chat_performed", Now()).size(), 1);

        UNIT_ASSERT(gServer.SendOrderToChat(userId, "support.123", "zxzxzx"));

        TString chatTagId = "";
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            chatTagId = dbTags.begin()->GetTagId();

            auto tagOriginal = dbTags.front();
            auto tagDestination = tagOriginal.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
            tagDestination->SetName("support_chat_performed");
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().DirectEvolveTag("robot-frontend", tagOriginal, tagDestination.GetData(), session) && session.Commit());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_performed"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        }

        // Start performing
        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_performed" });

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_performed"}, dbTags, session));

            auto tag = dbTags.front();
            UNIT_ASSERT(tag->GetPerformer() != "");
        }

        // Defer
        {
            UNIT_ASSERT(gServer.DeferRequest(chatTagId, operatorId));
        }

        // Yet one deferred, with performer
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_deferred"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            auto tag = dbTags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), operatorId);
        }

        UNIT_ASSERT(gServer.SendArbitraryMessage("support.123", userId, "Hello!"));
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support.123", "xzxzxzx", {}));

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_performed"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            auto tag = dbTags.front();
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), operatorId);
        }
    }

    Y_UNIT_TEST(DisableAnswered) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        {
            UNIT_ASSERT(server->GetSettings().SetValue("handlers.api/staff/chats/feed.disable_answered", "true", USER_ROOT_DEFAULT));
        }

        ui32 urgentCount = 0;
        auto userId = eGenerator.CreateUser("steap-was-here", true, "onboarding");
        auto operatorId = eGenerator.CreateUser("operator-was-here", true, "active");
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(operatorId).SetActive(true);
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, operatorId, session) && session.Commit());
            auto feedJson = gServer.GetChatsFeed("", "", operatorId, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            urgentCount = feedJson["chats"]["urgent"].GetArray().size();
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support", "hello", {}));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            auto feedJson = gServer.GetChatsFeed("", "", operatorId, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(!feedJson["chats"]["urgent"].GetArray().empty());
            UNIT_ASSERT(feedJson["chats"]["answered"].GetArray().empty());
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            auto userPermissions = server->GetDriveAPI()->GetUserPermissions(operatorId, TUserPermissionsFeatures());
            UNIT_ASSERT(gServer.StartTag(tag.GetTagId(), operatorId));
            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(TSupportChatTag::ChangeMutedStatus(tag, true, operatorId, server.Get(), session) && session.Commit());

            UNIT_ASSERT(server->GetChatEngine()->RefreshCache(Now()));
        }

        // All chats are urgent when answered is disabled, even muted one
        {
            auto feedJson = gServer.GetChatsFeed("", "", operatorId, server->GetDriveAPI()->GetTagsSearch(NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(!feedJson["chats"]["urgent"].GetArray().empty());
            UNIT_ASSERT(feedJson["chats"]["answered"].GetArray().empty());
        }
    }

    Y_UNIT_TEST(OffDutyDropPerformer) {
        TStringStream ss;
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        gServer.SetNeedBackground(0);
        gServer.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("steap-was-here", true, "active");
        auto operatorId = eGenerator.CreateUser("operator-was-here", true, "active");

        // 'support_chat_performed' doesn't drop performer on defer
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 0);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(operatorId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, operatorId, session) && session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 1);

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(gServer.AddTag(new TSimpleUserTag("duty"), operatorId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(session.Commit());
        }

        TAtomicSharedPtr<TRTDropUserTagPerformer> actor = MakeAtomicShared<TRTDropUserTagPerformer>();
        actor->SetPeriod(TDuration::Seconds(0.5));
        actor->SetEnabled(true);
        actor->SetTagNames({"support_chat_2_line"});
        actor->SetPerformerTagNames({"duty_end"});
        TRTBackgroundProcessContainer container(actor);
        container.SetName("support_duty_manager");

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.dsfjhsdgf", "Hello, Yandex.Drive!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.dsfjhsdgf", "Your support chat is really awesome!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.dsfjhsdgf", "Check out this funny cat picture.", {}));
        }

        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_2_line" });
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags[0]->GetPerformer(), operatorId);
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ operatorId }, {"duty"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);

            auto tagOriginal = dbTags.front();
            auto tagDestination = tagOriginal.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
            tagDestination->SetName("duty_end");
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().DirectEvolveTag(operatorId, tagOriginal, tagDestination.GetData(), session) && session.Commit());
        }

        {
            UNIT_ASSERT(container->Execute(nullptr, {*server}));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags[0]->GetPerformer(), "");
        }
    }

    Y_UNIT_TEST(InactiveDropPerformer) {
        TStringStream ss;
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        gServer.SetNeedBackground(0);
        gServer.ToString(ss);
        TServerConfigConstructorParams params(ss.Str().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eGenerator.CreateUser("steap-was-here", true, "active");
        auto operatorId = eGenerator.CreateUser("operator-was-here", true, "active");

        // 'support_chat_performed' doesn't drop performer on defer
        UNIT_ASSERT_VALUES_EQUAL(driveApi.GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 0);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(operatorId).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, operatorId, session) && session.Commit());
        }
        UNIT_ASSERT_VALUES_EQUAL(server->GetDriveAPI()->GetRolesManager()->GetPotentialTagAssignees("support_chat_2_line", Now()).size(), 1);

        {
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.dsfjhsdgf", "Hello, Yandex.Drive!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.dsfjhsdgf", "Your support chat is really awesome!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(userId, "support_modern.dsfjhsdgf", "Check out this funny cat picture.", {}));
        }

        {
            TSupportRequestsDistributonManager processor(server.Get(), "");
            processor.ResolveSupportRequests({ "support_chat_2_line" });
        }

        TAtomicSharedPtr<TRTDropUserTagPerformer> actor(new TRTDropUserTagPerformer);
        actor->SetPeriod(TDuration::Seconds(0.01));
        actor->SetEnabled(true);
        actor->SetTagNames({"support_chat_2_line"});
        actor->SetActivityPeriod(TDuration::Minutes(10));
        TRTBackgroundProcessContainer container(actor);
        container.SetName("support_duty_manager");

        {

            UNIT_ASSERT(container->Execute(nullptr, {*server}));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags[0]->GetPerformer(), operatorId);
        }

        {
            actor->SetActivityPeriod(TDuration::Zero());
            UNIT_ASSERT(container->Execute(nullptr, {*server}));
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(dbTags[0]->GetPerformer(), "");
        }
    }

    Y_UNIT_TEST(Localization) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        auto userId = env.GetEnvironmentGenerator().CreateUser("LocalizationTester", true, "active");
        UNIT_ASSERT(env->CommitChatAction(userId, "support_variables.123", "!localization", {}));
        auto chatMessagesReport = env->GetChatMessages(userId, "support_variables.123");
        auto messages = chatMessagesReport["messages"].GetArray();
        UNIT_ASSERT(!messages.empty());
        auto lastMessage = messages.back();
        UNIT_ASSERT_VALUES_EQUAL(lastMessage["text"].GetStringRobust(), "foo test42, bar test42");
    }
}
