#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/background/car_markers/config.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/tags/tag.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

Y_UNIT_TEST_SUITE(SwitchOffer) {
    using namespace NDrive::NTest;
    Y_UNIT_TEST(Simple) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor");
        script.Add<TAccept>();
        script.AddMove<TRide>(TDuration::Seconds(5));
        script.Add<TSwitchOffer>();
        script.AddMove<TDrop>(TDuration::Seconds(20));
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (json["segment"]["multi_bill"]["bills"].GetArraySafe().size() != 2) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"].GetArraySafe()[0].GetArraySafe()[1]["type"] != "old_state_riding") {
                return false;
            }
            return true;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(DoubleSwitch) {
        TTestEnvironment env;
        env.Execute(TBuildEnv());
        env.Execute(TCreateCar());
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        env.Execute(TCreateAndBookOffer().SetOfferName("standart_offer_constructor"));
        env.Execute(TAccept(Now()));
        env.Execute(TRide(Now()));

        auto offerId = env->CreateOffer(env.GetContext().GetCar().Id, USER_ID_DEFAULT, nullptr, 0, {}, {}, nullptr, "standart_offer_constructor");
        env.Execute(TSwitchOffer().SetOfferId(offerId));
        env.Execute(TSwitchOffer().SetOfferId(offerId).SetExpectOK(false));
    }

    Y_UNIT_TEST(PackSwitch) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TCreateAndBookOffer>().SetOfferName("return_pack_offer_constructor");
        script.Add<TCreateAndSwitchOffer>().SetOfferName("return_pack_offer_constructor");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Seconds(15));
        script.Add<TCreateAndSwitchOffer>().SetOfferName("return_pack_offer_constructor");
        script.Add<TInitSessionId>();
        script.Add<TDrop>(TDuration::Seconds(25));
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            const ui32 ext = json["segment"]["session"]["specials"]["current_offer"]["extension"].GetUIntegerSafe();
            UNIT_ASSERT(ext <= 10800 * 2 && ext > 10800 * 2 - 30);
            if (json["segment"]["multi_bill"]["bills"].GetArraySafe().size() != 3) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][0].GetArraySafe().size() != 3) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][1].GetArraySafe().size() != 3) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][2].GetArraySafe().size() != 3) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][0].GetArraySafe()[2]["cost"].GetUIntegerSafe() != 66) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][1].GetArraySafe()[2]["cost"].GetUIntegerSafe() != 100000) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][2].GetArraySafe()[2]["cost"].GetUIntegerSafe() != 100000) {
                return false;
            }
            if (json["segment"]["multi_bill"]["records"][0]["cost"].GetUInteger() != 200066) {
                return false;
            }
            return true;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        const auto sessionHistoryChecker = [](const TRTContext& context, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(json["sessions"].GetArraySafe().size(), 3);
            UNIT_ASSERT_VALUES_EQUAL(json["sessions"].GetArraySafe()[0]["offer_proto"]["OfferId"].GetStringRobust(), context.GetCurrentSessionIdUnsafe());
            return true;
        };
        script.Add<TCheckSessionHistoryWaiting>().SetChecker(sessionHistoryChecker);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(SwitchStandartInMinimalZone) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);

        script.Add<TBuildEnv>();

        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        {
            THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
            TAreaDropPolicyBuilder odp;
            odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(10000000).SetPolicy(EDropOfferPolicy::FeesMax).SetAlertsCount(0);
            doTag->MutableDropPolicies().emplace_back(odp);

            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
            script.Add<TDropCache>();
        }

        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Seconds(15)).SetCarPosition(to);
        script.Add<TCreateAndSwitchOffer>().SetOfferName("standart_offer_constructor");
        script.Add<TDrop>(TDuration::Seconds(25)).SetCarPosition(from);
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (json["segment"]["multi_bill"]["bills"].GetArraySafe().size() != 2) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][0].GetArraySafe()[2]["cost"].GetUIntegerSafe() != 66) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][1].GetArraySafe()[2]["cost"].GetUIntegerSafe() != 100) {
                return false;
            }
            if (json["segment"]["multi_bill"]["records"][0]["cost"].GetUInteger() != 166) {
                return false;
            }
            return true;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(SwitchStandartToFixInMinimalZone) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        const TGeoCoord toAlter(37.578623258813465, 55.73902070432059);

        script.Add<TBuildEnv>();

        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        {
            THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
            TAreaDropPolicyBuilder odp;
            odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(10000000).SetPolicy(EDropOfferPolicy::FeesMax).SetAlertsCount(0);
            doTag->MutableDropPolicies().emplace_back(odp);

            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
            script.Add<TDropCache>();
        }

        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Seconds(15)).SetCarPosition(to);
        script.Add<TCreateAndSwitchOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(toAlter).SetUserPosition(from);
        script.Add<TDrop>(TDuration::Seconds(25)).SetCarPosition(toAlter);
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (json["segment"]["multi_bill"]["bills"].GetArraySafe().size() != 2) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][0].GetArraySafe()[2]["cost"].GetUIntegerSafe() != 10000000) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][1].GetArraySafe()[2]["cost"].GetUIntegerSafe() != 100) {
                return false;
            }
            if (json["segment"]["multi_bill"]["records"][0]["cost"].GetUInteger() != 10000100) {
                return false;
            }
            return true;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(SwitchStandartToFixInMinimalZoneIncorrectFinish) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        const TGeoCoord from(37.62643984041714, 55.740277103506685);
        const TGeoCoord to(37.57248636458738, 55.73354854993565);
        const TGeoCoord toAlter(37.578623258813465, 55.73902070432059);

        script.Add<TBuildEnv>();

        const TString coordsDenyArea = "37.56836649154051 55.73318532658823 37.56843086455687 55.73215617532038 37.58654113982519 55.735945738857616 37.578902208550815 55.73988019829806 37.56836649154051 55.73318532658823";
        script.Add<TCreateCar>(TDuration::Zero()).SetPosition(from);
        script.Add<TCreateArea>().SetId("test-fix-only-1").SetCoords(coordsDenyArea).SetAreaTags({"allow_drop_car"}).SetActionUserId(USER_ROOT_DEFAULT);
        script.Add<TDropCache>();

        {
            THolder<TDropAreaFeatures> doTag(new TDropAreaFeatures(TDropAreaFeatures::TypeName));
            TAreaDropPolicyBuilder odp;
            odp.SetOfferAttributesFilter(TTagsFilter::BuildFromString("fix,standart")).SetFee(10000000).SetPolicy(EDropOfferPolicy::FeesMax).SetAlertsCount(0);
            doTag->MutableDropPolicies().emplace_back(odp);

            script.Add<TAddTag>(doTag.Release()).SetObjectId("test-fix-only-1").SetEntityType(NEntityTagsManager::EEntityType::Area);
            script.Add<TDropCache>();
        }

        script.Add<TCreateAndBookOffer>().SetOfferName("standart_offer_constructor");
        script.Add<TAccept>();
        script.Add<TRide>(TDuration::Seconds(15)).SetCarPosition(to);
        script.Add<TCreateAndSwitchOffer>().SetOfferName("fixpoint_offer_constructor").SetUserDestination(toAlter).SetUserPosition(from);
        script.Add<TDrop>(TDuration::Seconds(25)).SetCarPosition(to);
        const auto sessionChecker = [](const TRTContext& /*context*/, const NJson::TJsonValue& json)->bool {
            INFO_LOG << json.GetStringRobust() << Endl;
            if (json["segment"]["multi_bill"]["bills"].GetArraySafe().size() != 2) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][0].GetArraySafe()[4]["cost"].GetUIntegerSafe() != 10000000) {
                return false;
            }
            if (json["segment"]["multi_bill"]["bills"][1].GetArraySafe()[2]["cost"].GetUIntegerSafe() != 100) {
                return false;
            }
            if (json["segment"]["multi_bill"]["records"][0]["cost"].GetUInteger() != 10000100) {
                return false;
            }
            return true;
        };
        script.Add<TCheckCurrentSessionWaiting>().SetChecker(sessionChecker);

        UNIT_ASSERT(script.Execute());
    }
}
