#include <drive/backend/ut/library/helper2.h>

struct TTagHistoryStat {
    TInstant MinTimestamp = TInstant::Max();
    TInstant MaxTimestamp = TInstant::Zero();
    ui64 MinEventId = Max<ui64>();
    ui64 MaxEventId = 0;
};

Y_UNIT_TEST_SUITE(TagsManagerSuite) {
    Y_UNIT_TEST(GetEvents) {
        TTestEnvironment env;
        auto check = [] (const IEntityTagsManager& manager) {
            auto session = manager.BuildSession();
            ui64 limit = 10000;
            auto allEvents = manager.GetEventsSince(0, session, limit);
            UNIT_ASSERT(allEvents);
            UNIT_ASSERT(!allEvents->empty());
            INFO_LOG << allEvents->size() << Endl;
            UNIT_ASSERT(allEvents->size() <= limit);

            TMap<TString, TTagHistoryStat> tagIds;
            TMap<TString, TTagHistoryStat> objectIds;
            for (auto&& ev : *allEvents) {
                const auto& tagId = ev.GetTagId();
                const auto& objectId = ev.GetObjectId();
                const auto eventId = ev.GetHistoryEventId();
                TInstant timestamp = ev.GetHistoryTimestamp();
                UNIT_ASSERT(eventId);
                UNIT_ASSERT(timestamp);
                UNIT_ASSERT(tagId);
                UNIT_ASSERT(objectId);
                auto& tagIdInfo = tagIds[tagId];
                auto& objectIdInfo = objectIds[objectId];
                tagIdInfo.MinTimestamp = std::min(tagIdInfo.MinTimestamp, timestamp);
                tagIdInfo.MaxTimestamp = std::max(tagIdInfo.MaxTimestamp, timestamp);
                tagIdInfo.MinEventId = std::min(tagIdInfo.MinEventId, eventId);
                tagIdInfo.MaxEventId = std::max(tagIdInfo.MaxEventId, eventId);
                objectIdInfo.MinTimestamp = std::min(objectIdInfo.MinTimestamp, timestamp);
                objectIdInfo.MaxTimestamp = std::max(objectIdInfo.MaxTimestamp, timestamp);
                objectIdInfo.MinEventId = std::min(objectIdInfo.MinEventId, eventId);
                objectIdInfo.MaxEventId = std::max(objectIdInfo.MaxEventId, eventId);
            }
            size_t crop = 42;
            if (tagIds.size() > crop) {
                auto end = tagIds.begin();
                for (size_t i = 0; i < crop; ++i) {
                    end++;
                }
                TMap<TString, TTagHistoryStat> tagIdsCropped = { tagIds.begin(), end };
                tagIds = std::move(tagIdsCropped);
            }
            if (objectIds.size() > crop) {
                auto end = objectIds.begin();
                for (size_t i = 0; i < crop; ++i) {
                    end++;
                }
                TMap<TString, TTagHistoryStat> objectIdsCropped = { objectIds.begin(), end };
                objectIds = std::move(objectIdsCropped);
            }

            for (auto&& [tagId, info] : tagIds) {
                UNIT_ASSERT(info.MaxEventId >= info.MinEventId);
                UNIT_ASSERT(info.MaxTimestamp >= info.MinTimestamp);
                ui64 sinceEventId = info.MinEventId + 0.5 * (info.MaxEventId - info.MinEventId);
                auto sinceTimestamp = info.MinTimestamp + 0.5 * (info.MaxTimestamp - info.MinTimestamp);
                auto events = manager.GetEventsByTag(tagId, session, sinceEventId, sinceTimestamp);
                UNIT_ASSERT_C(events, session.GetStringReport());
                UNIT_ASSERT(!events->empty());
                for (auto&& ev : *events) {
                    UNIT_ASSERT(ev.GetHistoryEventId() >= sinceEventId);
                    UNIT_ASSERT(ev.GetHistoryTimestamp() >= sinceTimestamp);
                    UNIT_ASSERT_VALUES_EQUAL(ev.GetTagId(), tagId);
                }
            }

            for (auto&& [objectId, info] : objectIds) {
                UNIT_ASSERT(info.MaxEventId >= info.MinEventId);
                UNIT_ASSERT(info.MaxTimestamp >= info.MinTimestamp);
                ui64 sinceEventId = info.MinEventId + 0.5 * (info.MaxEventId - info.MinEventId);
                auto sinceTimestamp = info.MinTimestamp + 0.5 * (info.MaxTimestamp - info.MinTimestamp);
                auto events = manager.GetEventsByObject(objectId, session, sinceEventId, sinceTimestamp);
                UNIT_ASSERT_C(events, session.GetStringReport());
                UNIT_ASSERT(!events->empty());
                for (auto&& ev : *events) {
                    UNIT_ASSERT(ev.GetHistoryEventId() >= sinceEventId);
                    UNIT_ASSERT(ev.GetHistoryTimestamp() >= sinceTimestamp);
                    UNIT_ASSERT_VALUES_EQUAL(ev.GetObjectId(), objectId);
                }
            }
        };

        const auto& tagManager = env.GetServer()->GetDriveAPI()->GetTagsManager();
        check(tagManager.GetDeviceTags());
        check(tagManager.GetUserTags());
        check(tagManager.GetTraceTags());
    }

    Y_UNIT_TEST(QueryOptions) {
        TTestEnvironment env;
        auto check = [] (const IEntityTagsManager& manager) {
            auto session = manager.BuildSession();
            ui64 limit = 100;
            ui64 offset = 142;
            auto allEvents = manager.GetEvents(0, session, { limit, true, offset });
            UNIT_ASSERT(allEvents);
            UNIT_ASSERT(!allEvents->empty());
            for (auto&& ev : *allEvents) {
                auto queryOptions = IEntityTagsManager::TQueryOptions()
                    .SetActions({ ev.GetHistoryAction() })
                    .SetTags({ ev->GetName() })
                    .SetTagIds({ ev.GetTagId() })
                    .SetObjectIds({ ev.GetObjectId() })
                    .SetUserIds({ ev.GetHistoryUserId() })
                ;
                if (ev.GetHistoryOriginatorId()) {
                    queryOptions.SetOriginatorIds({ ev.GetHistoryOriginatorId() });
                }
                if (ev->GetPerformer()) {
                    queryOptions.SetPerformers({ ev->GetPerformer() });
                }
                auto refetch = manager.GetEvents({ ev.GetHistoryEventId(), ev.GetHistoryEventId() + 1 }, session, queryOptions);
                UNIT_ASSERT(refetch);
                UNIT_ASSERT_VALUES_EQUAL(refetch->size(), 1);
                UNIT_ASSERT_VALUES_EQUAL(refetch->front().GetHistoryEventId(), ev.GetHistoryEventId());
            }
        };

        const auto& tagManager = env.GetServer()->GetDriveAPI()->GetTagsManager();
        check(tagManager.GetDeviceTags());
        check(tagManager.GetUserTags());
        check(tagManager.GetTraceTags());
    }
}
