#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/helper2.h>
#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>

#include <drive/backend/actions/evolution.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/account_tags.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/coloring.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/data/feedback.h>
#include <drive/backend/data/markers.h>
#include <drive/backend/data/service_session.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/config.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/offers/manager.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/tags/tags_search.h>
#include <drive/backend/user_options/user_setting.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

#include <util/system/env.h>

Y_UNIT_TEST_SUITE(DriveTags) {
    class TTagTest: public NDrive::ITag {
    protected:
        EUniquePolicy GetUniquePolicy() const override {
            return EUniquePolicy::Rewrite;
        }

        void SerializeSpecialDataToJson(NJson::TJsonValue& /*json*/) const override {

        }

        TBlob DoSerializeSpecialData(NDrive::NProto::TTagHeader& /*h*/) const override {
            return TBlob();
        }

        bool DoDeserializeSpecialData(const NDrive::NProto::TTagHeader& /*h*/, const TBlob& /*data*/) override {
            return true;
        }

        TSet<NEntityTagsManager::EEntityType> GetObjectType() const override {
            return{ NEntityTagsManager::EEntityType::Car };
        }

        TString GetHRDescription() const override {
            return "";
        }
    public:
        static TFactory::TRegistrator<TTagTest> Registrator;
    };
    TTagTest::TFactory::TRegistrator<TTagTest> TTagTest::Registrator("aaa");

    Y_UNIT_TEST(TagDescriptionsSimple) {
        TDriveAPIConfigGenerator generator;
        auto config = generator.Generate();
        auto database = generator.CreateDatabase();
        TDriveTagsManager tagsManager(database, *config);
        TDriveTagsManager tagsManagerSlave(database, *config);
        {
            auto session = tagsManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(tagsManager.GetTagsMeta().UnregisterTag("aaa", USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport());
            UNIT_ASSERT_C(tagsManager.GetTagsMeta().UnregisterTag("bbb", USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport());
            UNIT_ASSERT(session.Commit());
        }
        for (ui32 i = 0; i < 100; ++i) {
            if (!tagsManager.GetTagsMeta().GetDescriptionByName("bbb") && !tagsManagerSlave.GetTagsMeta().GetDescriptionByName("bbb") &&
                !tagsManager.GetTagsMeta().GetDescriptionByName("aaa") && !tagsManagerSlave.GetTagsMeta().GetDescriptionByName("aaa")
                ) {
                break;
            } else {
                INFO_LOG << i << "-s attemption for empty" << Endl;
            }
            Sleep(TDuration::Seconds(1));
            UNIT_ASSERT(i != 100 - 1);
        }
        {
            auto session = tagsManager.BuildTx<NSQL::Writable>();
            TTagDescription tag;
            tag.SetName("bbb").SetType("aaa");
            UNIT_ASSERT_C(tagsManager.GetTagsMeta().RegisterTag(new TTagDescription(tag), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport());
            UNIT_ASSERT(session.Commit());
        }
        {
            THolder<TPropertiesPatchTag::TDescription> patch(new TPropertiesPatchTag::TDescription);
            patch->SetType(TPropertiesPatchTag::TypeName);
            TString name = "test" + ToString(RandomNumber<ui32>());
            patch->SetName(name);
            patch->AddPatch("1", "2");
            auto session = tagsManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(tagsManager.GetTagsMeta().RegisterTag(patch.Release(), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport());
            UNIT_ASSERT(session.Commit());
            bool found = false;
            for (ui32 i = 0; i < 100; ++i) {
                if (!!tagsManager.GetTagsMeta().GetDescriptionByName(name)) {
                    found = true;
                    auto td = tagsManager.GetTagsMeta().GetDescriptionByName(name);
                    auto tdPatch = dynamic_cast<const TPropertiesPatchTag::TDescription*>(td.Get());
                    UNIT_ASSERT(tdPatch);
                    UNIT_ASSERT_VALUES_EQUAL(tdPatch->GetPatch("1"), "2");
                    break;
                }
                Sleep(TDuration::Seconds(1));
            }
            UNIT_ASSERT(found);
        }
        {
            auto session = tagsManager.BuildTx<NSQL::Writable>();
            TTagDescription tag;
            tag.SetName("bbb" + ToString(RandomNumber<ui32>())).SetType("aaa");
            UNIT_ASSERT(!tagsManager.GetTagsMeta().GetDescriptionByName(tag.GetName()));
            UNIT_ASSERT_C(tagsManager.GetTagsMeta().RegisterTag(new TTagDescription(tag), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport());
            UNIT_ASSERT(session.Commit());
            bool found = false;
            for (ui32 i = 0; i < 100; ++i) {
                if (!!tagsManager.GetTagsMeta().GetDescriptionByName(tag.GetName())) {
                    found = true;
                    UNIT_ASSERT(tagsManager.GetTagsMeta().GetDescriptionByName(tag.GetName())->GetIndex() != Max<ui32>());
                    break;
                }
                Sleep(TDuration::Seconds(1));
            }
            UNIT_ASSERT(found);
        }
        for (ui32 i = 0; i < 100; ++i) {
            if (!!tagsManager.GetTagsMeta().GetDescriptionByName("bbb") && !!tagsManagerSlave.GetTagsMeta().GetDescriptionByName("bbb") &&
                !tagsManager.GetTagsMeta().GetDescriptionByName("aaa") && !tagsManagerSlave.GetTagsMeta().GetDescriptionByName("aaa")
                ) {
                break;
            } else {
                INFO_LOG << i << "-s attemption for empty" << Endl;
            }
            Sleep(TDuration::Seconds(1));
            UNIT_ASSERT(i != 100 - 1);
        }
        {
            auto session = tagsManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(tagsManager.GetTagsMeta().UnregisterTag("aaa", USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport());
            UNIT_ASSERT_C(tagsManager.GetTagsMeta().UnregisterTag("bbb", USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport());
            UNIT_ASSERT(session.Commit());
        }
        for (ui32 i = 0; i < 100; ++i) {
            TTagDescription description;
            if (!tagsManager.GetTagsMeta().GetDescriptionByName("bbb") && !tagsManagerSlave.GetTagsMeta().GetDescriptionByName("bbb") &&
                !tagsManager.GetTagsMeta().GetDescriptionByName("aaa") && !tagsManagerSlave.GetTagsMeta().GetDescriptionByName("aaa")
                ) {
                break;
            } else {
                INFO_LOG << i << "-s attemption for empty" << Endl;
            }
            Sleep(TDuration::Seconds(1));
            UNIT_ASSERT(i != 100 - 1);
        }
    }

    Y_UNIT_TEST(FetchData) {
        TFakeHistoryContext context;
        TUsersDB userDB(context);
        auto result = userDB.FetchInfo(USER_ID_DEFAULT);
        auto user = result.GetResultPtr(USER_ID_DEFAULT);
        UNIT_ASSERT(user);
        UNIT_ASSERT_VALUES_EQUAL(user->GetPhone(), "+79006468203");
    }

    Y_UNIT_TEST(ImagesSnapshot) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        const auto api = server->GetDriveAPI();
        UNIT_ASSERT(api);
        const IDriveTagsManager& manager = api->GetTagsManager();

        const TString objectId = OBJECT_ID_DEFAULT;
        auto emulator = tmBuilder.BuildEmulator(server->GetDriveAPI()->GetIMEI(objectId));

        {
            auto userFetchResult = server->GetDriveAPI()->GetUsersData()->FetchInfo(USER_ID_TECH);
            UNIT_ASSERT_VALUES_EQUAL(userFetchResult.size(), 1);
            auto userEntry = userFetchResult.begin()->second;
            userEntry.SetStatus("active");
            auto session = api->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->UpdateUser(userEntry, "robot-frontend", session) && session.Commit());
        }

        tmBuilder.SetSensorValue<ui64>(*emulator, CAN_FUEL_LEVEL_P, 10);
        UNIT_ASSERT(gServer.WaitSensor(objectId, "fuel_level", "10"));
        {
            auto session = api->template BuildTx<NSQL::Writable>();
            TTagDescription tag;
            tag.SetName("fueling").SetType("simple_fueling_tag").SetEnableSessions(true);
            UNIT_ASSERT_C(manager.GetTagsMeta().RegisterTag(new TTagDescription(tag), USER_ID_DEFAULT, session), TStringBuilder() << session.GetReport());

            UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(objectId, USER_ID_TECH, { "fueling" }, session, false));
            NJson::TJsonValue tagInfo;
            tagInfo["tag_name"] = "fueling";
            tagInfo["comment"] = "test fueling tag";
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(IJsonSerializableTag::BuildFromJson(manager, tagInfo), USER_ID_DEFAULT, objectId, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }

        TInstant testStart = Now();
        const auto& historyManager = manager.GetDeviceTags().GetHistoryManager();
        {
            TVector<TDBTag> tagsAll;
            {
                auto session = api->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(objectId, { "fueling" }, tagsAll, session));
                UNIT_ASSERT_VALUES_EQUAL(tagsAll.size(), 1);
            }
            TDBTag& testTag = tagsAll.front();

            auto userPermissions = server->GetDriveAPI()->GetUserPermissions(USER_ID_TECH, TUserPermissionsFeatures());
            {
                auto session = api->template BuildTx<NSQL::Writable>();
                TMap<TString, TVector<TDBTag>> performObjects;
                UNIT_ASSERT(manager.GetDeviceTags().GetPerformObjects({ USER_ID_TECH }, performObjects, session));

                for (auto&& object : performObjects) {
                    UNIT_ASSERT(manager.GetDeviceTags().DropTagsPerformer(object.second, USER_ID_TECH, session, true));
                }

                UNIT_ASSERT(manager.GetDeviceTags().InitPerformer({ testTag }, *userPermissions, server.Get(), session));
                {
                    THolder<TImagesSnapshot> imageSnapshot(new TImagesSnapshot);
                    TImagesSnapshot::TImage image;
                    image.ExternalId = "1";
                    image.Path = "test";
                    image.Marker = "marker";
                    image.ImageId = 42;
                    imageSnapshot->AddImage(std::move(image));
                    UNIT_ASSERT(manager.GetDeviceTags().AddSnapshot(testTag, std::move(imageSnapshot), USER_ID_TECH, session));
                }
                {
                    THolder<TImagesSnapshot> imageSnapshot(new TImagesSnapshot);
                    TImagesSnapshot::TImage image;
                    image.ExternalId = "2";
                    image.Path = "test";
                    image.Marker = "marker";
                    image.ImageId = 42;
                    imageSnapshot->AddImage(std::move(image));
                    UNIT_ASSERT(manager.GetDeviceTags().AddSnapshot(testTag, std::move(imageSnapshot), USER_ID_TECH, session));
                }
                UNIT_ASSERT(session.Commit());
            }

            UNIT_ASSERT(tmBuilder.SetSensorValue<ui64>(*emulator, CAN_FUEL_LEVEL_P, 50));
            UNIT_ASSERT(gServer.WaitSensor(objectId, "fuel_level", "50"));
            {
                auto session = api->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(manager.GetDeviceTags().DropPerformer({ testTag.GetTagId() }, {}, *userPermissions, server.Get(), session));
                UNIT_ASSERT(session.Commit());
            }
        }

        auto builder = historyManager.GetSessionsBuilder("service", Now());
        UNIT_ASSERT(builder);

        auto objectSessions = builder->GetSessionsActualByObjects(testStart, Now(), { objectId });
        UNIT_ASSERT_VALUES_EQUAL(objectSessions.size(), 1);
        UNIT_ASSERT_VALUES_EQUAL(objectSessions.begin()->second.size(), 1);
        for (auto&& session : objectSessions.begin()->second) {
            UNIT_ASSERT(session->Compile());
            TMaybe<TServiceSession::TCompilation> compilation = session->GetCompilationAs<TServiceSession::TCompilation>();
            UNIT_ASSERT(compilation);
            INFO_LOG << compilation->GetReport(DefaultLocale, nullptr, nullptr).GetStringRobust() << Endl;
            INFO_LOG << session->GetEventsReport().GetStringRobust() << Endl;
            UNIT_ASSERT_VALUES_EQUAL(compilation->GetImages().size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(compilation->GetUserId(), USER_ID_TECH);
            UNIT_ASSERT(compilation->GetImages().contains("1"));
            UNIT_ASSERT_VALUES_EQUAL(compilation->GetFuelLevelBefore(), 10);
            UNIT_ASSERT_VALUES_EQUAL(compilation->GetFuelLevelAfter(), 50);
        }
    }

    Y_UNIT_TEST(Create) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();
        auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "low_voltage" }, session, false));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagLowVoltage(1), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagNoSignal(), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
        {
            TVector<TDBTag> tagsAll;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {}, tagsAll, session));
            bool lowVoltageFlag = false;
            bool noSignalFlag = false;
            for (auto&& testTag : tagsAll) {
                lowVoltageFlag |= (testTag->GetName() == "low_voltage");
                noSignalFlag |= (testTag->GetName() == "no_signal");
            }
            UNIT_ASSERT_C(noSignalFlag && lowVoltageFlag, TStringBuilder() << noSignalFlag << lowVoltageFlag << Endl);
        }
        TVector<TDBTag> tagsLV;
        UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "low_voltage" }, tagsLV, session));
        {
            bool lowVoltageFlag = false;
            bool noSignalFlag = false;
            for (auto&& testTag : tagsLV) {
                lowVoltageFlag |= (testTag->GetName() == "low_voltage");
                noSignalFlag |= (testTag->GetName() == "no_signal");
            }
            UNIT_ASSERT_C(!noSignalFlag && lowVoltageFlag, TStringBuilder() << noSignalFlag << lowVoltageFlag << Endl);
        }
        TVector<TDBTag> tagsNS;
        UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "no_signal" }, tagsNS, session));
        {
            bool lowVoltageFlag = false;
            bool noSignalFlag = false;
            for (auto&& testTag : tagsNS) {
                lowVoltageFlag |= (testTag->GetName() == "low_voltage");
                noSignalFlag |= (testTag->GetName() == "no_signal");
            }
            UNIT_ASSERT_C(noSignalFlag && !lowVoltageFlag, TStringBuilder() << noSignalFlag << lowVoltageFlag << Endl);
        }
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple({ tagsLV.front().GetTagId() }, USER_ID_DEFAULT, session, false));
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {}, tags, session));
            {
                bool lowVoltageFlag = false;
                bool noSignalFlag = false;
                for (auto&& testTag : tags) {
                    lowVoltageFlag |= (testTag->GetName() == "low_voltage");
                    noSignalFlag |= (testTag->GetName() == "no_signal");
                }
                UNIT_ASSERT_C(noSignalFlag && !lowVoltageFlag, TStringBuilder() << noSignalFlag << lowVoltageFlag << Endl);
            }
        }
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "no_signal" }, session, false));
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {}, tags, session));
            {
                bool lowVoltageFlag = false;
                bool noSignalFlag = false;
                for (auto&& testTag : tags) {
                    lowVoltageFlag |= (testTag->GetName() == "low_voltage");
                    noSignalFlag |= (testTag->GetName() == "no_signal");
                }
                UNIT_ASSERT_C(!noSignalFlag && !lowVoltageFlag, TStringBuilder() << noSignalFlag << lowVoltageFlag << Endl);
            }
        }
    }

    Y_UNIT_TEST(Performer) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();
        auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
        {
            TVector<TDBTag> tagsAll;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {}, tagsAll, session));
            UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(tagsAll, USER_ID_DEFAULT, session, true));
        }
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagLowVoltage(1), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
        {
            TVector<TDBTag> tagsLV;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "low_voltage" }, tagsLV, session));
            UNIT_ASSERT_VALUES_EQUAL(tagsLV.size(), 1);
            UNIT_ASSERT(manager.GetDeviceTags().SetTagPerformer(tagsLV.front(), "a966c353-2c34-4d0c-82d0-18af589c9ff0", false, session, &*server));
            UNIT_ASSERT(manager.GetDeviceTags().SetTagPerformer(tagsLV.front(), "a966c353-2c34-4d0c-82d0-18af589c9ff0", false, session, &*server));
            UNIT_ASSERT(!manager.GetDeviceTags().SetTagPerformer(tagsLV.front(), "0acb00f1-4342-412a-9425-f7dbfb8a1a87", false, session, &*server));
            session.ClearErrors();
            UNIT_ASSERT(manager.GetDeviceTags().SetTagPerformer(tagsLV.front(), "0acb00f1-4342-412a-9425-f7dbfb8a1a87", true, session, &*server));
            UNIT_ASSERT(manager.GetDeviceTags().SetTagPerformer(tagsLV.front(), "a966c353-2c34-4d0c-82d0-18af589c9ff0", true, session, &*server));
        }

        {
            TVector<TDBTag> tagsNS;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "low_voltage" }, tagsNS, session));
            UNIT_ASSERT_VALUES_EQUAL(tagsNS.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tagsNS.front()->GetPerformer(), "a966c353-2c34-4d0c-82d0-18af589c9ff0");
            UNIT_ASSERT(!manager.GetDeviceTags().DropTagPerformer(tagsNS.front(), "incorrect", session));
            session.ClearErrors();
            UNIT_ASSERT(manager.GetDeviceTags().DropTagPerformer(tagsNS.front(), "a966c353-2c34-4d0c-82d0-18af589c9ff0", session));
            UNIT_ASSERT(manager.GetDeviceTags().DropTagPerformer(tagsNS.front(), "a966c353-2c34-4d0c-82d0-18af589c9ff0", session));
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, { "low_voltage" }, tagsNS, session));
            UNIT_ASSERT_VALUES_EQUAL(tagsNS.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(tagsNS.front()->GetPerformer(), "");
        }
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "low_voltage" }, session, false));
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {}, tags, session));
            {
                bool lowVoltageFlag = false;
                for (auto&& testTag : tags) {
                    lowVoltageFlag |= (testTag->GetName() == "low_voltage");
                }
                UNIT_ASSERT_C(!lowVoltageFlag, TStringBuilder() << lowVoltageFlag << Endl);
            }
        }
    }

    Y_UNIT_TEST(List) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();

        auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "low_voltage" }, session, false));
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple("57194c39-6f36-4110-8838-429064e13954", USER_ID_DEFAULT, { "low_voltage", "no_signal" }, session, false));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagLowVoltage(1), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagLowVoltage(2), USER_ID_DEFAULT, "57194c39-6f36-4110-8838-429064e13954", server.Get(), session));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagLowVoltage(3), USER_ID_DEFAULT, "57194c39-6f36-4110-8838-429064e13954", server.Get(), session));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagLowVoltage(4), USER_ID_DEFAULT, "57194c39-6f36-4110-8838-429064e13954", server.Get(), session));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagLowVoltage(5), USER_ID_DEFAULT, "57194c39-6f36-4110-8838-429064e13954", server.Get(), session));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagNoSignal(), USER_ID_DEFAULT, "57194c39-6f36-4110-8838-429064e13954", server.Get(), session));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagNoSignal(), USER_ID_DEFAULT, "57194c39-6f36-4110-8838-429064e13954", server.Get(), session));
        UNIT_ASSERT(manager.GetDeviceTags().AddTag(new TTagNoSignal(), USER_ID_DEFAULT, "57194c39-6f36-4110-8838-429064e13954", server.Get(), session));

        {
            TVector<TTaggedDevice> devices;
            UNIT_ASSERT(manager.GetDeviceTags().GetObjects({ "low_voltage" }, false, devices, session));
            UNIT_ASSERT_VALUES_EQUAL(devices.size(), 2);
            for (auto&& i : devices) {
                if (i.GetId() == OBJECT_ID_DEFAULT) {
                    UNIT_ASSERT_VALUES_EQUAL(i.GetTags().size(), 1);
                } else {
                    UNIT_ASSERT_VALUES_EQUAL(i.GetId(), "57194c39-6f36-4110-8838-429064e13954");
                    UNIT_ASSERT_VALUES_EQUAL(i.GetTags().size(), 1);
                }
                for (auto&& tag : i.GetTags()) {
                    INFO_LOG << tag->GetHRDescription() << Endl;
                }
            }
        }
        {
            TVector<TTaggedDevice> devices;
            UNIT_ASSERT(manager.GetDeviceTags().GetObjects({ "low_voltage", "no_signal" }, false, devices, session));
            UNIT_ASSERT_VALUES_EQUAL(devices.size(), 2);
            for (auto&& i : devices) {
                if (i.GetId() == OBJECT_ID_DEFAULT) {
                    UNIT_ASSERT_VALUES_EQUAL(i.GetTags().size(), 1);
                } else {
                    UNIT_ASSERT_VALUES_EQUAL(i.GetId(), "57194c39-6f36-4110-8838-429064e13954");
                    UNIT_ASSERT_VALUES_EQUAL(i.GetTags().size(), 2);
                }
                for (auto&& tag : i.GetTags()) {
                    INFO_LOG << tag->GetHRDescription() << Endl;
                }
            }
        }
        {
            TVector<TTaggedDevice> devices;
            UNIT_ASSERT(manager.GetDeviceTags().GetObjects({ "low_voltage" }, true, devices, session));
            UNIT_ASSERT_VALUES_EQUAL(devices.size(), 2);
            for (auto&& i : devices) {
                if (i.GetId() == OBJECT_ID_DEFAULT) {
                    bool lowVoltageFlag = false;
                    bool noSignalFlag = false;
                    for (auto&& testTag : i.GetTags()) {
                        lowVoltageFlag |= (testTag->GetName() == "low_voltage");
                        noSignalFlag |= (testTag->GetName() == "no_signal");
                    }
                    UNIT_ASSERT_C(!noSignalFlag && lowVoltageFlag, TStringBuilder() << noSignalFlag << lowVoltageFlag << Endl);
                } else {
                    UNIT_ASSERT_VALUES_EQUAL(i.GetId(), "57194c39-6f36-4110-8838-429064e13954");
                    bool lowVoltageFlag = false;
                    bool noSignalFlag = false;
                    for (auto&& testTag : i.GetTags()) {
                        if (testTag->GetName() == "low_voltage") {
                            UNIT_ASSERT_VALUES_EQUAL(VerifyDynamicCast<const TTagLowVoltage*>(testTag.GetData().Get())->GetVoltage(), 5);
                        }
                        lowVoltageFlag |= (testTag->GetName() == "low_voltage");
                        noSignalFlag |= (testTag->GetName() == "no_signal");
                    }
                    UNIT_ASSERT_C(noSignalFlag && lowVoltageFlag, TStringBuilder() << noSignalFlag << lowVoltageFlag << Endl);
                }
                for (auto&& tag : i.GetTags()) {
                    INFO_LOG << tag->GetHRDescription() << Endl;
                }
            }
        }
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "low_voltage" }, session, false));
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple("57194c39-6f36-4110-8838-429064e13954", USER_ID_DEFAULT, { "low_voltage", "no_signal" }, session, false));
    }

    Y_UNIT_TEST(Restore) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const auto& manager = server->GetDriveAPI()->GetTagsManager();
        const auto& deviceTagsManager = manager.GetDeviceTags();

        ui32 count = 42;
        TSet<TString> tags = {
            "old_state_reservation",
            "old_state_acceptance",
            "class_base",
        };
        auto tx = deviceTagsManager.BuildTx<NSQL::ReadOnly>();
        TTagEventsManager::TQueryOptions queryOptions(count);
        queryOptions.SetTags(tags);
        queryOptions.SetOrderBy({ "object_id", "tag_id" });
        auto optionalTags = deviceTagsManager.RestoreTags(tx, std::move(queryOptions));
        UNIT_ASSERT(optionalTags);
        UNIT_ASSERT_VALUES_EQUAL(optionalTags->size(), count);
        for (auto&& tag : *optionalTags) {
            UNIT_ASSERT(tag);
            UNIT_ASSERT(tags.contains(tag->GetName()));
        }
        bool sorted = std::is_sorted(optionalTags->begin(), optionalTags->end(), [](const TDBTag& left, const TDBTag& right) {
            return std::tie(left.GetObjectId(), left.GetTagId()) < std::tie(right.GetObjectId(), right.GetTagId());
        });
        UNIT_ASSERT(sorted);
    }

    Y_UNIT_TEST(Update) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const auto& manager = server->GetDriveAPI()->GetTagsManager();
        const auto& deviceTagsManager = manager.GetDeviceTags();

        i32 priority = 42;
        auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
        UNIT_ASSERT(deviceTagsManager.RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "low_voltage" }, session, false));
        {
            auto tag = MakeAtomicShared<TTagLowVoltage>(1.f);
            UNIT_ASSERT(deviceTagsManager.AddTag(tag, USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
        }
        TString tagId;
        {
            auto optionalTags = deviceTagsManager.RestoreTags(TVector<TString>{ OBJECT_ID_DEFAULT }, { "low_voltage" }, session);
            UNIT_ASSERT(optionalTags);
            UNIT_ASSERT_VALUES_EQUAL(optionalTags->size(), 1);
            auto tag = std::move(optionalTags->at(0));
            UNIT_ASSERT(tag);
            tagId = tag.GetTagId();
            tag->SetTagPriority(priority);
            UNIT_ASSERT(deviceTagsManager.UpdateTagData(tag, USER_ID_DEFAULT, session));
        }
        {
            auto optionalTag = deviceTagsManager.RestoreTag(tagId, session);
            UNIT_ASSERT(optionalTag);
            auto tag = std::move(*optionalTag);
            UNIT_ASSERT(tag);
            UNIT_ASSERT(tag->HasTagPriority());
            UNIT_ASSERT_VALUES_EQUAL(tag->GetTagPriorityRef(), priority);
        }
    }

    Y_UNIT_TEST(TagRegistration) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        auto name = "new_tag";
        UNIT_ASSERT(gServer.RegisterTag(MakeAtomicShared<TTagDescription>(name, "simple_car_tag"), USER_ROOT_DEFAULT));
        UNIT_ASSERT(!gServer.RegisterTag(MakeAtomicShared<TTagDescription>(name, "generic_signal_car_tag"), USER_ROOT_DEFAULT));
        UNIT_ASSERT(gServer.RegisterTag(MakeAtomicShared<TTagDescription>(name, "generic_signal_car_tag"), USER_ROOT_DEFAULT, /*force=*/true));
    }

    Y_UNIT_TEST(TagDescriptionProposition) {
        auto newTagDesc = R"({
            "name": "simple_car_propose42",
            "display_name": "",
            "comment": "",
            "default_priority": "0",
            "meta": {
                "enable_report": false,
                "deprecated": false,
                "transferred_to_double": false,
                "enable_sessions": false,
                "actions": []
            },
            "type": "simple_car_tag"
        })";

        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        const auto& tagsManager = server->GetDriveAPI()->GetRolesManager();

        {
            const TDriveAPI& driveApi = *server->GetDriveAPI();
            auto session = tagsManager->BuildTx<NSQL::Writable>();
            {
                TUserRole userRole;
                userRole.SetRoleId("adm_permissions_root").SetUserId(USER_ID_DEFAULT1);
                UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            }
            {
                TUserRole userRole;
                userRole.SetRoleId("full_tag_access").SetUserId(USER_ID_DEFAULT1);
                UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            }
            UNIT_ASSERT(session.Commit());
        }


        NJson::TJsonValue newTagDescJson;
        UNIT_ASSERT(NJson::ReadJsonFastTree(newTagDesc, &newTagDescJson));

        NStorage::TTableRecord tRecord;
        UNIT_ASSERT(tRecord.DeserializeFromJson(newTagDescJson));
        TTagDescription::TPtr tDescription = TTagDescription::ConstructFromTableRecord(tRecord);
        UNIT_ASSERT(tDescription);

        NJson::TJsonValue response;
        NJson::TJsonValue propositions;
        { // propose-confirm
            UNIT_ASSERT(gServer.ProposeTagDesc(tDescription, USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto reply = gServer.ListTagDesc(USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(reply.IsSuccessReply(), reply.GetDebugReply());


            UNIT_ASSERT(NJson::ReadJsonFastTree(reply.Content(), &response));
            UNIT_ASSERT(response.GetValueByPath("propositions", propositions));
            UNIT_ASSERT(propositions.IsArray());
            UNIT_ASSERT(propositions.GetArray().size() > 0);
            UNIT_ASSERT_C(propositions.GetArraySafe()[0]["name"].GetString() == "simple_car_propose42", "Cannot get propositions");
            auto proposeId = propositions.GetArraySafe()[0]["proposition_id"].GetString();

            UNIT_ASSERT(gServer.ConfirmTagDesc({proposeId}, USER_ID_DEFAULT1));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            reply = gServer.ListTagDesc(USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(reply.IsSuccessReply(), reply.GetDebugReply());

            UNIT_ASSERT(NJson::ReadJsonFastTree(reply.Content(), &response));
            UNIT_ASSERT(response.GetValueByPath("propositions", propositions));
            UNIT_ASSERT_C(propositions.GetArraySafe().size() == 0, "Cannot confirm propositions");
            NJson::TJsonValue records;
            UNIT_ASSERT(response.GetValueByPath("records", records));
            bool created = false;
            for (const auto& tagDesc : records.GetArraySafe()) {
                if (tagDesc["name"].GetString() == "simple_car_propose42") {
                    created = true;
                    break;
                }
            }
            UNIT_ASSERT_C(created, "new tag description doesn't created");
        }

        { // propose-reject
            tDescription->SetName("simple_car_propose43");
            UNIT_ASSERT(gServer.ProposeTagDesc(tDescription, USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto reply = gServer.ListTagDesc(USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(reply.IsSuccessReply(), reply.GetDebugReply());

            UNIT_ASSERT(NJson::ReadJsonFastTree(reply.Content(), &response));
            UNIT_ASSERT(response.GetValueByPath("propositions", propositions));
            UNIT_ASSERT_C(propositions.GetArraySafe()[0]["name"].GetString() == "simple_car_propose43", "Cannot get propositions");
            auto proposeId = propositions.GetArraySafe()[0]["proposition_id"].GetString();

            UNIT_ASSERT(gServer.RejectTagDesc({proposeId}, USER_ID_DEFAULT1));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            reply = gServer.ListTagDesc(USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(reply.IsSuccessReply(), reply.GetDebugReply());

            UNIT_ASSERT(NJson::ReadJsonFastTree(reply.Content(), &response));
            UNIT_ASSERT(response.GetValueByPath("propositions", propositions));
            UNIT_ASSERT_C(propositions.GetArraySafe().size() == 0, "Cannot confirm propositions");
            NJson::TJsonValue records;
            UNIT_ASSERT(response.GetValueByPath("records", records));
            bool created = false;
            for (const auto& tagDesc : records.GetArraySafe()) {
                if (tagDesc["name"].GetString() == "simple_car_propose43") {
                    created = true;
                    break;
                }
            }
            UNIT_ASSERT_C(!created, "new tag description must be rejected");
        }

        {
            const TDriveAPI& driveApi = *server->GetDriveAPI();
            auto session = tagsManager->BuildTx<NSQL::Writable>();
            const auto& rm = driveApi.GetRolesManager();
            {
                UNIT_ASSERT(rm->RemoveRoleForUser(USER_ID_DEFAULT1, "adm_permissions_root", USER_ROOT_DEFAULT, session));
                UNIT_ASSERT(rm->RemoveRoleForUser(USER_ID_DEFAULT1, "full_tag_access", USER_ROOT_DEFAULT, session));
            }
            UNIT_ASSERT(session.Commit());
        }
    }

    Y_UNIT_TEST(FilterByTags) {
        InitGlobalLog2Console(TLOG_DEBUG);
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        NJson::TJsonValue requestData;

        auto objectIds = TVector<TString>({ OBJECT_ID_DEFAULT, "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6" });
        for (auto objectId : objectIds) {
            NJson::TJsonValue report = gServer.ListTags(objectId, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);

            NJson::TJsonValue::TArray arr;
            if (!report["records"].GetArray(&arr)) {
                continue;
            }
            for (auto&& i : arr) {
                UNIT_ASSERT(i.IsMap());
                if (i["tag"].GetString() == "simple1" || i["tag"].GetString() == "simple2" || i["tag"] == "priority_simple1") {
                    UNIT_ASSERT(gServer.RemoveTag({ i["tag_id"].GetString() }, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
                }
            }
        }

        {
            NJson::TJsonValue report = gServer.ListTags(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);
            UNIT_ASSERT_C(report.IsNull() || report["records"].GetArraySafe().empty(), TStringBuilder() << report.GetStringRobust());
        }

        {
            UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
            NJson::TJsonValue result;
            NJson::TJsonValue requestResult;
            for (ui32 i = 0; i < 100; ++i) {
                NDrive::TServerConfigGenerator::TDisableLogging disableLogging(gServer);
                requestResult = gServer.GetCachedServiceCarsList({}, USER_ROOT_DEFAULT, "simple1");
                UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
                if (TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT)) {
                    break;
                }
                INFO_LOG << "Check iteration " << i << Endl;
                UNIT_ASSERT_C(i != 99, TStringBuilder() << result << Endl);
                Sleep(TDuration::Seconds(1));
            }

            requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "simple2");
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6"), TStringBuilder() << result << Endl);

            INFO_LOG << "1!!!!!!!!!!" << Endl;
            UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple2"), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
            INFO_LOG << "2!!!!!!!!!!" << Endl;

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "simple2");
            INFO_LOG << "3!!!!!!!!!!" << Endl;
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6"), TStringBuilder() << result << Endl);

            requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "simple1,simple2");
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT), TStringBuilder() << result << Endl);
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6"), TStringBuilder() << result << Endl);

            requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "simple1*simple2");
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT), TStringBuilder() << result << Endl);
            UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6"), TStringBuilder() << result << Endl);

            UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple2"), OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
            requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "simple1*simple2");
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT), TStringBuilder() << result << Endl);

            {
                TGeoCoord c(55, 55);
                {
                    requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "", &c, 1);
                    UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
                    ui32 count = 0;
                    for (auto&& i : result.GetArray()) {
                        if (i["location"] != NJson::JSON_NULL) {
                            ++count;
                        }
                    }
                    UNIT_ASSERT_VALUES_EQUAL(count, 1);
                    UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT), TStringBuilder() << result << Endl);
                }
                {
                    requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "simple1*simple2", &c, 1);
                    UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
                    ui32 count = 0;
                    for (auto&& i : result.GetArray()) {
                        if (i["location"] != NJson::JSON_NULL) {
                            ++count;
                        }
                    }
                    UNIT_ASSERT_VALUES_EQUAL(count, 1);
                    UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT), TStringBuilder() << result << Endl);
                }
            }

            UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("priority_simple1"), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
            requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "simple1*simple2*priority_simple1");
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT), TStringBuilder() << result << Endl);
            UNIT_ASSERT_C(!TValidationHelpers::ResponseListHasObject(result.GetArray(), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6"), TStringBuilder() << result << Endl);

            requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "#simple1*#simple2,+simple2*#priority_simple1");
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT), TStringBuilder() << result << Endl);
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6"), TStringBuilder() << result << Endl);

            requestResult = gServer.GetServiceCarsList({}, USER_ROOT_DEFAULT, "simple2*(simple1,priority_simple1)");
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), OBJECT_ID_DEFAULT), TStringBuilder() << result << Endl);
            UNIT_ASSERT_C(TValidationHelpers::ResponseListHasObject(result.GetArray(), "50c0c78d-f546-4a26-b1ea-ae577d7b0bf6"), TStringBuilder() << result << Endl);
        }

        for (auto objectId : objectIds) {
            NJson::TJsonValue report = gServer.ListTags(objectId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car);

            NJson::TJsonValue::TArray arr;
            if (!report["records"].GetArray(&arr)) {
                continue;
            }
            for (auto&& i : arr) {
                UNIT_ASSERT(i.IsMap());
                if (i["tag"].GetString() == "simple1" || i["tag"].GetString() == "simple2" || i["tag"] == "priority_simple1") {
                    UNIT_ASSERT(gServer.RemoveTag({ i["tag_id"].GetString() }, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
                }
            }
        }
    }

    Y_UNIT_TEST(SimpleRollbackTagsState) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        const auto& tagsManager = server->GetDriveAPI()->GetTagsManager().GetDeviceTags();
        auto session = tagsManager.BuildTx<NSQL::Writable>();

        auto instant = ModelingNow();

        TInstantGuard ig(instant - TDuration::Minutes(10));
        UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
        {
            auto tags = tagsManager.SimpleRollbackTagsState(OBJECT_ID_DEFAULT, instant - TDuration::Minutes(20), {}, session);
            UNIT_ASSERT_C(tags, session.GetStringReport());
            UNIT_ASSERT(!tags->contains("simple1"));
        }
        {
            auto tags = tagsManager.SimpleRollbackTagsState(OBJECT_ID_DEFAULT, instant, { "simple1" }, session);
            UNIT_ASSERT_C(tags, session.GetStringReport());
            UNIT_ASSERT(tags->contains("simple1"));
        }
        {
            auto tags = tagsManager.SimpleRollbackTagsState(OBJECT_ID_DEFAULT, instant, {}, session);
            UNIT_ASSERT_C(tags, session.GetStringReport());
            UNIT_ASSERT(tags->contains("simple1"));
        }
        {
            auto tags = tagsManager.SimpleRollbackTagsState(OBJECT_ID_DEFAULT, instant - TDuration::Minutes(2), {}, session);
            UNIT_ASSERT_C(tags, session.GetStringReport());
            UNIT_ASSERT(tags->contains("simple1"));
        }
        {
            auto tags = tagsManager.SimpleRollbackTagsState(OBJECT_ID_DEFAULT, instant, { "simple2" }, session);
            UNIT_ASSERT_C(tags, session.GetStringReport());
            UNIT_ASSERT(!tags->contains("simple1"));
        }
        {
            ig.Set(instant - TDuration::Minutes(5));
            UNIT_ASSERT_C(tagsManager.RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "simple1" }, session, true), session.GetStringReport());
        }
        {
            auto tags = tagsManager.SimpleRollbackTagsState(OBJECT_ID_DEFAULT, instant - TDuration::Minutes(8), {}, session);
            UNIT_ASSERT_C(tags, session.GetStringReport());
            UNIT_ASSERT(tags->contains("simple1"));
        }
        {
            auto tags = tagsManager.SimpleRollbackTagsState(OBJECT_ID_DEFAULT, instant, {}, session);
            UNIT_ASSERT_C(tags, session.GetStringReport());
            UNIT_ASSERT(!tags->contains("simple1"));
        }
        {
            auto tags = tagsManager.SimpleRollbackTagsState(OBJECT_ID_DEFAULT, instant - TDuration::Minutes(2), {}, session);
            UNIT_ASSERT_C(tags, session.GetStringReport());
            UNIT_ASSERT(!tags->contains("simple1"));
        }
    }

    Y_UNIT_TEST(FiltersParsing) {
        {
            TTagsFilter filter;
            UNIT_ASSERT(filter.DeserializeFromString("(dd*(bb,ff*(gg,hh)))"));
            UNIT_ASSERT_VALUES_EQUAL(filter.ToLinearString(), "bb*dd,dd*ff*gg,dd*ff*hh");
            INFO_LOG << filter.ToLinearString() << Endl;
        }
        {
            TTagsFilter filter;
            UNIT_ASSERT(filter.DeserializeFromString("&a*(&b, &c)*('d,   ?e),?d*(#b),(n)*(m),(dd*(bb,ff*(gg,hh))),aa"));
            UNIT_ASSERT_VALUES_EQUAL(filter.ToLinearString(), "aa,&a*&b*'d,&a*&b*?e,&a*&c*'d,&a*&c*?e,#b*?d,m*n,bb*dd,dd*ff*gg,dd*ff*hh");
        }
    }

    Y_UNIT_TEST(NamedFilters) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        TNamedFilter namedFilter;
        NJson::TJsonValue filterJs;
        {
            TString strFilter = R"({
            "object":
                {
                    "display_name": "firstFilter",
                    "description": "big description",
                    "type": "car",
                    "filter": {"conditions": "iter1,iter2,iter3"},
                    "revision": 10
                }
            })";
            UNIT_ASSERT(NJson::ReadJsonFastTree(strFilter, &filterJs));
            TMessagesCollector errors;
            UNIT_ASSERT_C(TBaseDecoder::DeserializeFromJsonVerbose(namedFilter, filterJs["object"], errors), "incorrect object json: " << errors.GetStringReport());
            namedFilter.SetId(ToString(NUtil::CreateUUID()));

            auto tx = server->GetDriveAPI()->BuildTx<NSQL::Writable>();
            NStorage::TObjectRecordsSet<TNamedFilter> records;
            UNIT_ASSERT(server->GetDriveAPI()->GetNamedFiltersDB()->AddObjects({namedFilter}, USER_ID_DEFAULT, tx, &records));
            UNIT_ASSERT_C(!records.empty(), "cannot add namedFilter to table");
            UNIT_ASSERT(tx.Commit());
        }

        {
            auto tx = server.Get()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto action = MakeHolder<TTagAction>("root_iter");
            action->SetTagName("$iter")
                .AddTagAction(TTagAction::ETagAction::Add)
                .AddTagAction(TTagAction::ETagAction::Propose)
                .AddTagAction(TTagAction::ETagAction::Confirm)
                .AddTagAction(TTagAction::ETagAction::Reject)
                .AddTagAction(TTagAction::ETagAction::Perform)
                .AddTagAction(TTagAction::ETagAction::ForcePerform)
                .AddTagAction(TTagAction::ETagAction::Observe)
                .AddTagAction(TTagAction::ETagAction::ObserveObject)
                .AddTagAction(TTagAction::ETagAction::DropPerform)
                .AddTagAction(TTagAction::ETagAction::Remove)
                .AddTagAction(TTagAction::ETagAction::RemovePerform);
            UNIT_ASSERT(server.Get()->GetDriveAPI()->GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, tx));

            auto roleHeader = TDriveRoleHeader{"root_iter_role"};
            UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesDB().Upsert(roleHeader, USER_ID_DEFAULT, tx), tx.GetStringReport());

            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("root_iter").SetRoleId("root_iter_role");
            UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ID_DEFAULT, tx), tx.GetStringReport());

            TUserRole userRole;
            userRole.SetRoleId("root_iter_role").SetUserId(USER_ID_DEFAULT);
            UNIT_ASSERT(server.Get()->GetDriveAPI()->GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, tx));
            UNIT_ASSERT(tx.Commit());

        }

        TVector<TEnvironmentGenerator::TCar> newCars;
        uint carsCount = 20;
        newCars.resize(carsCount);
        {
            uint cnt = 0;
            auto tx = server->GetDriveAPI()->BuildTx<NSQL::Writable>();
            for (auto& car : newCars) {
                car = eGenerator.CreateCar(tx, "porsche_carrera", "");
                UNIT_ASSERT(car.Id);

                TTagDescription description;
                description.SetName("iter" + ToString(cnt++)).SetType("simple_car_tag").SetDefaultPriority(0);
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(MakeAtomicShared<TTagDescription>(description), USER_ID_DEFAULT, tx));
            }
            UNIT_ASSERT(tx.Commit());

            for (auto& car : newCars) {
                UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), car.Id, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
                UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("iter" + ToString(--cnt)), car.Id, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
            }
        }

        TCarsFilter carsFilter;
        {
            carsFilter.SetIncludeTagsFilter(TTagsFilter::BuildFromString("simple1"));
            TSet<TString> filteredCars;
            UNIT_ASSERT(carsFilter.GetAllowedCarIds(filteredCars, server.Get(), TInstant::Now()));
            UNIT_ASSERT_VALUES_EQUAL(filteredCars.size(), carsCount);
        }
        {
            carsFilter.SetIncludeDynamicFilters({namedFilter.GetId()});
            TSet<TString> filteredCars;
            UNIT_ASSERT(carsFilter.GetAllowedCarIds(filteredCars, server.Get(), TInstant::Now()));
            UNIT_ASSERT_VALUES_EQUAL(filteredCars.size(), SplitString(filterJs["object"]["filter"]["conditions"].GetString(), ",").size());
        }

        auto tx = server->GetDriveAPI()->BuildTx<NSQL::Writable>();
        UNIT_ASSERT_C(server.Get()->GetDriveAPI()->GetRolesManager()->GetRolesInfoDB().UnlinkAllFromRole("root_iter_role", USER_ID_DEFAULT, tx), tx.GetStringReport());
        UNIT_ASSERT(tx.Commit());

    }

    Y_UNIT_TEST(TagsPerformLimit) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetNeedBackground(0);
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        auto car1 = eGenerator.CreateCar();
        auto car2 = eGenerator.CreateCar();
        auto car3 = eGenerator.CreateCar();

        auto emulator0 = tmBuilder.BuildEmulator(car1.IMEI);
        auto emulator1 = tmBuilder.BuildEmulator(car2.IMEI);
        auto emulator2 = tmBuilder.BuildEmulator(car3.IMEI);
        UNIT_ASSERT(gServer.WaitCar(car1.Id));
        UNIT_ASSERT(gServer.WaitCar(car2.Id));
        UNIT_ASSERT(gServer.WaitCar(car3.Id));
        {
            UNIT_ASSERT(gServer.ActualizeTags({ new TUniqueTag("simple1") }, {}, car1.Id, USER_ROOT_DEFAULT));
            UNIT_ASSERT(gServer.ActualizeTags({ new TUniqueTag("simple1") }, {}, car2.Id, USER_ROOT_DEFAULT));
            UNIT_ASSERT(gServer.ActualizeTags({ new TUniqueTag("simple1") }, {}, car3.Id, USER_ROOT_DEFAULT));

            TString tagId0;
            UNIT_ASSERT(gServer.GetTagId(car1.Id, "simple1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, tagId0));
            TString tagId1;
            UNIT_ASSERT(gServer.GetTagId(car2.Id, "simple1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, tagId1));
            TString tagId2;
            UNIT_ASSERT(gServer.GetTagId(car3.Id, "simple1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, tagId2));

            UNIT_ASSERT(gServer.StartTag(tagId0, USER_ID_DEFAULT));
            UNIT_ASSERT(gServer.StartTag(tagId1, USER_ID_DEFAULT));
            UNIT_ASSERT(!gServer.StartTag(tagId2, USER_ID_DEFAULT));

            UNIT_ASSERT(gServer.FinishTag(tagId0, USER_ID_DEFAULT, false));
            UNIT_ASSERT(gServer.FinishTag(tagId1, USER_ID_DEFAULT, false));

            UNIT_ASSERT(gServer.StartTag(tagId0, USER_ID_DEFAULT2));
            UNIT_ASSERT(gServer.StartTag(tagId1, USER_ID_DEFAULT2));
            UNIT_ASSERT(gServer.StartTag(tagId2, USER_ID_DEFAULT2));

            UNIT_ASSERT(gServer.FinishTag(tagId0, USER_ID_DEFAULT2, true));
            UNIT_ASSERT(gServer.FinishTag(tagId1, USER_ID_DEFAULT2, true));
            UNIT_ASSERT(gServer.FinishTag(tagId2, USER_ID_DEFAULT2, true));
        }
    }

    Y_UNIT_TEST(TagsPerformOverwrite) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();

        const IDriveTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager();
        const TDeviceTagsManager& deviceTagsManager = tagsManager.GetDeviceTags();
        auto tag = MakeAtomicShared<TUniqueTagRewrite>(TUniqueTagRewrite::TypeName);
        auto commentOriginal = TString("original");
        tag->SetComment(commentOriginal);
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(deviceTagsManager.AddTag(tag, USER_ROOT_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        auto commentReplacement = TString("replacement");
        tag->SetComment(commentReplacement);
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();

            auto dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            UNIT_ASSERT(deviceTagsManager.SetTagPerformer(dbTags->at(0), USER_ROOT_DEFAULT, false, session, &*server));

            const TString& commentBefore = dbTags->at(0)->GetComment();
            UNIT_ASSERT_VALUES_EQUAL(commentBefore, commentOriginal);
            UNIT_ASSERT(deviceTagsManager.AddTag(tag, USER_ROOT_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));

            dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            const TString& commentAfter = dbTags->at(0)->GetComment();
            UNIT_ASSERT_VALUES_EQUAL(commentAfter, commentOriginal);
        }
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            auto dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            UNIT_ASSERT(deviceTagsManager.SetTagPerformer(dbTags->at(0), USER_ROOT_DEFAULT, false, session, &*server));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();

            auto dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            const TString& commentBefore = dbTags->at(0)->GetComment();
            UNIT_ASSERT_VALUES_EQUAL(commentBefore, commentOriginal);
            UNIT_ASSERT(deviceTagsManager.AddTag(tag, USER_ROOT_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));

            dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            const TString& commentAfter = dbTags->at(0)->GetComment();

            UNIT_ASSERT_VALUES_EQUAL(commentAfter, commentOriginal);
        }
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();

            auto dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            UNIT_ASSERT(deviceTagsManager.DropTagPerformer(dbTags->at(0), USER_ROOT_DEFAULT, session));

            const TString& commentBefore = dbTags->at(0)->GetComment();
            UNIT_ASSERT_VALUES_EQUAL(commentBefore, commentOriginal);
            UNIT_ASSERT(deviceTagsManager.AddTag(tag, USER_ROOT_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));

            dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            const TString& commentAfter = dbTags->at(0)->GetComment();
            UNIT_ASSERT_VALUES_EQUAL(commentAfter, commentReplacement);
            UNIT_ASSERT(session.Rollback());
        }
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            auto dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            UNIT_ASSERT(deviceTagsManager.DropTagPerformer(dbTags->at(0), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            auto dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            const TString& commentBefore = dbTags->at(0)->GetComment();
            UNIT_ASSERT_VALUES_EQUAL(commentBefore, commentOriginal);
            UNIT_ASSERT(deviceTagsManager.AddTag(tag, USER_ROOT_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));

            dbTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { tag->GetName() }, session);
            UNIT_ASSERT(dbTags);
            UNIT_ASSERT(!dbTags->empty());
            const TString& commentAfter = dbTags->at(0)->GetComment();
            UNIT_ASSERT_VALUES_EQUAL(commentAfter, commentReplacement);
            UNIT_ASSERT(session.Commit());
        }
    }

    Y_UNIT_TEST(TagsActualization) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment();
        {
            UNIT_ASSERT(gServer.ActualizeTags({ new TUniqueTag("simple1") }, { "priority_simple1", "priority_simple2", "simple2" }, OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT));

            NJson::TJsonValue report = gServer.ListTags(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);
            INFO_LOG << report << Endl;
            bool check1 = false;
            for (auto&& i : report["records"].GetArraySafe()) {
                UNIT_ASSERT(i.IsMap());
                if (i["tag"].GetString() == "simple1") {
                    check1 = true;
                }
                UNIT_ASSERT(i["tag"].GetString() != "simple2" && i["tag"] != "priority_simple1" && i["tag"] != "priority_simple2");
            }
            UNIT_ASSERT(check1);
        }

        {
            UNIT_ASSERT(gServer.ActualizeTags({ new TUniqueTag("simple2") }, { "simple1" }, OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT));

            NJson::TJsonValue report = gServer.ListTags(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);
            INFO_LOG << report << Endl;
            bool check1 = false;
            for (auto&& i : report["records"].GetArraySafe()) {
                UNIT_ASSERT(i.IsMap());
                if (i["tag"].GetString() == "simple2") {
                    check1 = true;
                }
                UNIT_ASSERT(i["tag"].GetString() != "simple1" && i["tag"] != "priority_simple1" && i["tag"] != "priority_simple2");
            }
            UNIT_ASSERT(check1);
        }

    }

    Y_UNIT_TEST(CarVisibilityBasedOnPerformedTags) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment();
        NJson::TJsonValue requestData;

        auto car1 = eGenerator.CreateCar();
        auto car2 = eGenerator.CreateCar();

        auto objectIds = TVector<TString>({car1.Id, car2.Id});

        {
            UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), car1.Id, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
            UNIT_ASSERT(gServer.AddTag(new TDeviceTagRecord("simple1"), car2.Id, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
            NJson::TJsonValue result;
            {
                bool found = false;
                const TInstant start = Now();
                while (Now() - start < TDuration::Minutes(2)) {
                    auto requestResult = gServer.GetServiceCarsList({}, USER_ID_DEFAULT, "simple1");
                    INFO_LOG << requestResult << Endl;
                    UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
                    if (result.GetArray().size() == 2) {
                        found = true;
                        break;
                    }
                }
                UNIT_ASSERT_C(found, TStringBuilder() << result << Endl);
            }

            NJson::TJsonValue report = gServer.ListTags(car1.Id, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);
            NJson::TJsonValue::TArray arr;
            UNIT_ASSERT(report["records"].GetArray(&arr));
            TString tagId = arr[0]["tag_id"].GetString();
            UNIT_ASSERT(gServer.StartTag(tagId, USER_ID_DEFAULT));

            {
                bool found = false;
                const TInstant start = Now();
                while (Now() - start < TDuration::Minutes(2)) {
                    auto requestResult = gServer.GetServiceCarsList({}, USER_ID_DEFAULT, "simple1");
                    UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
                    if (result.GetArray().size() == 1) {
                        found = true;
                        break;
                    }
                }
                UNIT_ASSERT_C(found, TStringBuilder() << result << Endl);
            }
        }

        for (auto objectId : objectIds) {
            NJson::TJsonValue report = gServer.ListTags(objectId, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car);

            NJson::TJsonValue::TArray arr;
            if (!report["records"].GetArray(&arr)) {
                continue;
            }
            for (auto&& i : arr) {
                UNIT_ASSERT(i.IsMap());
                if (i["tag"].GetString() == "simple1" || i["tag"].GetString() == "simple2" || i["tag"] == "priority_simple1") {
                    UNIT_ASSERT(gServer.RemoveTag({ i["tag_id"].GetString() }, USER_ID_DEFAULT, NEntityTagsManager::EEntityType::Car));
                }
            }
        }
    }

    Y_UNIT_TEST(NotUnique) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();
        auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();

        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "simple_car_tag" }, session, false));
        ui32 devicesOriginalCount = 0;
        {
            TVector<TTaggedDevice> devices;
            UNIT_ASSERT(manager.GetDeviceTags().GetObjects({"simple_car_tag"}, false, devices, session));
            devicesOriginalCount = devices.size();
        }
        {
            NJson::TJsonValue tagInfo;
            tagInfo["tag_name"] = "simple_car_tag";
            tagInfo["comment"] = "nsofya was here";

            UNIT_ASSERT(manager.GetDeviceTags().AddTag(IJsonSerializableTag::BuildFromJson(manager, tagInfo), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(IJsonSerializableTag::BuildFromJson(manager, tagInfo), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(IJsonSerializableTag::BuildFromJson(manager, tagInfo), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(IJsonSerializableTag::BuildFromJson(manager, tagInfo), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(IJsonSerializableTag::BuildFromJson(manager, tagInfo), USER_ID_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session));
        }

        {
            TVector<TTaggedDevice> devices;
            UNIT_ASSERT(manager.GetDeviceTags().GetObjects({ "simple_car_tag" }, false, devices, session));
            UNIT_ASSERT_VALUES_EQUAL(devices.size(), devicesOriginalCount + 1);
            for (auto&& i : devices) {
                if (i.GetId() == OBJECT_ID_DEFAULT) {
                    UNIT_ASSERT_VALUES_EQUAL(i.GetTags().size(), 5);
                }
                for (auto&& tag : i.GetTags()) {
                    INFO_LOG << tag->GetHRDescription() << Endl;
                }
            }
        }
        UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(OBJECT_ID_DEFAULT, USER_ID_DEFAULT, { "simple_car_tag" }, session, false));
        {
            TVector<TDBTag> tags;
            UNIT_ASSERT(manager.GetDeviceTags().RestoreEntityTags(OBJECT_ID_DEFAULT, {}, tags, session));
            for (auto&& tag : tags) {
                UNIT_ASSERT(tag->GetName() != "simple_car_tag");
            }
        }

    }

    Y_UNIT_TEST(OldStatus) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();
        ITag::TPtr tagService = new TServiceTagRecord("evacuation");
        tagService->SetTagPriority(10);
        ITag::TPtr tagCleaning = new TServiceTagRecord("urgent_cleaning");
        tagCleaning->SetTagPriority(100);
        auto car = eg.CreateCar();
        const TString carId = car.Id;

        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            auto transaction = session.GetTransaction();
            auto carsTable = transaction->GetDatabase().GetTable("\"car\"");
            carsTable->UpdateRow("id='" + carId + "'", "status='available'", transaction);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(tagService, USER_ID_DEFAULT, carId, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(gServer.WaitStatus(carId, "service", *server));

        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            UNIT_ASSERT(manager.GetDeviceTags().RemoveTagsSimple(carId, USER_ID_DEFAULT, { tagService->GetName() }, session, false));
            auto transaction = session.GetTransaction();
            auto carsTable = transaction->GetDatabase().GetTable("\"car\"");
            carsTable->UpdateRow("id='" + carId + "'", "status='available'", transaction);
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(tagCleaning, USER_ID_DEFAULT, carId, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        UNIT_ASSERT(gServer.WaitStatus(carId, "cleaning", *server));
    }

    Y_UNIT_TEST(FiltersForVisibilityAction) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();

        auto car1 = eg.CreateCar();
        auto car2 = eg.CreateCar();
        auto car3 = eg.CreateCar();

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(&(new TDeviceTagRecord("simple1"))->SetTagPriority(1), USER_ROOT_DEFAULT, car1.Id, server.Get(), session));
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(&(new TDeviceTagRecord("priority_simple2"))->SetTagPriority(1), USER_ROOT_DEFAULT, car1.Id, server.Get(), session));

            UNIT_ASSERT(manager.GetDeviceTags().AddTag(&(new TDeviceTagRecord("simple1"))->SetTagPriority(1), USER_ROOT_DEFAULT, car2.Id, server.Get(), session));
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(&(new TDeviceTagRecord("simple2"))->SetTagPriority(1), USER_ROOT_DEFAULT, car2.Id, server.Get(), session));
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(&(new TDeviceTagRecord("uniq_simple1"))->SetTagPriority(1), USER_ROOT_DEFAULT, car2.Id, server.Get(), session));

            UNIT_ASSERT(manager.GetDeviceTags().AddTag(&(new TDeviceTagRecord("uniq_simple1"))->SetTagPriority(1), USER_ROOT_DEFAULT, car3.Id, server.Get(), session));
            UNIT_ASSERT(manager.GetDeviceTags().AddTag(&(new TDeviceTagRecord("priority_simple2"))->SetTagPriority(1), USER_ROOT_DEFAULT, car3.Id, server.Get(), session));
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }


        auto userId = eg.CreateUser("nsofya-was-here", false);
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        NJson::TJsonValue requestResult = gServer.GetServiceCarsList({}, userId);
        NJson::TJsonValue result;
        UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
        UNIT_ASSERT(!TValidationHelpers::ResponseListHasObject(result.GetArray(), car1.Id));
        UNIT_ASSERT(!TValidationHelpers::ResponseListHasObject(result.GetArray(), car2.Id));
        UNIT_ASSERT(!TValidationHelpers::ResponseListHasObject(result.GetArray(), car3.Id));

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TDriveRoleHeader defaultUserRole("test_user");
            UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesDB().Upsert(defaultUserRole, USER_ROOT_DEFAULT, session));

            TUserRole userRole;
            userRole.SetRoleId("test_user").SetUserId(userId);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            THolder<TTagAction> action(new TTagAction("uniq_simple1_observe"));
            action->SetTagName("uniq_simple1").AddTagAction(TTagAction::ETagAction::ObserveObject);
            UNIT_ASSERT(driveApi.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            THolder<TObjectAccessAction> action(new TObjectAccessAction("simple1+priority_simple2"));
            TTagsFilter filter;
            UNIT_ASSERT(filter.DeserializeFromString("simple1*priority_simple2"));
            action->SetFilter(filter).MutableActions().emplace(TObjectAccessAction::EAction::Observe);
            UNIT_ASSERT(driveApi.GetRolesManager()->GetActionsDB().ForceUpsert(action.Release(), USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("uniq_simple1_observe").SetRoleId("test_user");
            UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));

            actionHeader.SetSlaveObjectId("view_permissions_action").SetRoleId("test_user");
            UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }

        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            requestResult = gServer.GetServiceCarsList({}, userId);
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT(!TValidationHelpers::ResponseListHasObject(result.GetArray(), car1.Id));
            UNIT_ASSERT(TValidationHelpers::ResponseListHasObject(result.GetArray(), car2.Id));
            UNIT_ASSERT(TValidationHelpers::ResponseListHasObject(result.GetArray(), car3.Id));
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TLinkedRoleActionHeader actionHeader;
            actionHeader.SetSlaveObjectId("simple1+priority_simple2").SetRoleId("test_user");
            UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Link(actionHeader, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }

        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            requestResult = gServer.GetServiceCarsList({}, userId);
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT(TValidationHelpers::ResponseListHasObject(result.GetArray(), car1.Id));
            UNIT_ASSERT(TValidationHelpers::ResponseListHasObject(result.GetArray(), car2.Id));
            UNIT_ASSERT(TValidationHelpers::ResponseListHasObject(result.GetArray(), car3.Id));
        }

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(driveApi.GetRolesManager()->GetRolesInfoDB().GetRoleActions().Unlink("uniq_simple1_observe", "test_user", USER_ROOT_DEFAULT, session));
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetStringReport() << Endl);
        }

        {
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            requestResult = gServer.GetServiceCarsList({}, userId);
            UNIT_ASSERT(requestResult.GetValueByPath("cars", result));
            UNIT_ASSERT(TValidationHelpers::ResponseListHasObject(result.GetArray(), car1.Id));
            UNIT_ASSERT(!TValidationHelpers::ResponseListHasObject(result.GetArray(), car2.Id));
            UNIT_ASSERT(!TValidationHelpers::ResponseListHasObject(result.GetArray(), car3.Id));
        }
    }

    Y_UNIT_TEST(MTRuntimeVisibility) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();

        auto car = eg.CreateCar();
        TGeoCoord someCoord(12.58643489401084, 45.73761541977004);
        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        UNIT_ASSERT(gServer.WaitCar(car.Id));
        NDrive::TCarDriver driver(tmBuilder.GetAPI());
        driver.RelocateCar(car.IMEI, someCoord);
        UNIT_ASSERT(gServer.WaitLocation(car.Id, someCoord));

        auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
        UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(new TDeviceAdditionalFeature("manual_gearbox"), USER_ROOT_DEFAULT, car.Id, server.Get(), session));
        UNIT_ASSERT(session.Commit());

        auto newUserId = eg.CreateUser("zxqfd-was-here");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            auto user = driveApi.GetUsersData()->FetchInfo(newUserId).GetResult().begin()->second;
            user.SetHasATMark(true);
            driveApi.GetUsersData()->UpdateUser(user, "robot-frontend", session);
            UNIT_ASSERT(session.Commit());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            auto carsList = gServer.GetUserCarsList({}, newUserId);
            NJson::TJsonValue result;
            UNIT_ASSERT_C(carsList.GetValueByPath("cars", result), TStringBuilder() << carsList);
            UNIT_ASSERT(!TValidationHelpers::ResponseListHasObject(result.GetArray(), "", car.Number));
        }
        INFO_LOG << "absence ok" << Endl;
        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            auto user = driveApi.GetUsersData()->FetchInfo(newUserId).GetResult().begin()->second;
            user.SetHasATMark(false);
            driveApi.GetUsersData()->UpdateUser(user, "robot-frontend", session);
            UNIT_ASSERT(session.Commit());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            user = driveApi.GetUsersData()->FetchInfo(newUserId).GetResult().begin()->second;
            INFO_LOG << "user.IsMTAllowed=" << user.IsMTAllowed() << Endl;
            auto carsList = gServer.GetUserCarsList({}, newUserId);
            NJson::TJsonValue result;
            UNIT_ASSERT_C(carsList.GetValueByPath("cars", result), TStringBuilder() << carsList);
            UNIT_ASSERT(TValidationHelpers::ResponseListHasObject(result.GetArray(), "", car.Number));
        }
    }

    Y_UNIT_TEST(TelematicsTag) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetUsersData()->GetRoles().Unlink(USER_ID_DEFAULT1, "default_user", USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }

        TString titleKey = "flash_cooldown_title";
        TString messageKey = "flash_cooldown_message";
        TString title = "hold on";
        TString message = "not so fast";
        {
            NLocalization::TResource resource;
            resource.SetId(titleKey);
            {
                TVector<NLocalization::TResource> result;
                UNIT_ASSERT(gServer.GetLocalizations(USER_ROOT_DEFAULT, result, {titleKey}));
                if (!result.empty()) {
                    resource.SetRevision(result.front().GetRevision());
                }
            }
            resource.MutableLocalizations().emplace_back(::ToString(DefaultLocale), title);
            UNIT_ASSERT(gServer.UpsertLocalization(resource, USER_ROOT_DEFAULT));
        }
        {
            NLocalization::TResource resource;
            resource.SetId(messageKey);
            {
                TVector<NLocalization::TResource> result;
                UNIT_ASSERT(gServer.GetLocalizations(USER_ROOT_DEFAULT, result, {messageKey}));
                if (!result.empty()) {
                    resource.SetRevision(result.front().GetRevision());
                }
            }
            resource.MutableLocalizations().emplace_back(::ToString(DefaultLocale), message);
            UNIT_ASSERT(gServer.UpsertLocalization(resource, USER_ROOT_DEFAULT));
        }

        const IDriveTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager();
        const auto& deviceTagsManager = tagsManager.GetDeviceTags();

        UNIT_ASSERT(server->GetSettings().SetValue("telematics.command.HORN.cooldown", "10000", USER_ID_TECH));
        UNIT_ASSERT(server->GetSettings().SetValue("telematics.command.HORN.cooldown_localization_message_key", messageKey, USER_ID_TECH));
        UNIT_ASSERT(server->GetSettings().SetValue("telematics.command.HORN.cooldown_localization_title_key", titleKey, USER_ID_TECH));
        UNIT_ASSERT(server->GetSettings().SetValue("telematics.command.HORN.duration", "42", USER_ID_TECH));
        UNIT_ASSERT(server->GetSettings().SetValue("telematics.command.YADRIVE_FORCED_END_OF_LEASE.limit", "0", USER_ID_TECH));
        TMaybe<double> limit;
        for (size_t i = 0; i < 100; ++i) {
            limit = server->GetSettings().GetValue<double>("telematics.command.YADRIVE_FORCED_END_OF_LEASE.limit");
            if (limit) {
                break;
            } else {
                Sleep(TDuration::Seconds(1));
            }
        }
        UNIT_ASSERT(limit);

        auto car = eg.CreateCar();
        TGeoCoord someCoord(12.58643489401084, 45.73761541977004);
        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        UNIT_ASSERT(gServer.WaitCar(car.Id));

        auto userPermissions = server->GetDriveAPI()->GetUserPermissions(USER_ID_TECH, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            auto response = TTelematicsCommandTag::Command(car.Id, NDrive::NVega::ECommandCode::YADRIVE_FORCED_END_OF_LEASE, TDuration::Seconds(10), *userPermissions, server.Get(), session);
            UNIT_ASSERT(!response.Initialized());
        }
        {
            auto firmwareInfo = NDrive::NVega::ParseFirmwareInfo("VEGA MT-32K LTE V2 0.10b rc37 'Kaptur'");
            auto firmwareTag = MakeAtomicShared<TTelematicsFirmwareTag>();
            firmwareTag->SetFirmware(firmwareInfo, "fake_handler");
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(tagsManager.GetDeviceTags().AddTag(firmwareTag, USER_ID_TECH, car.Id, server.Get(), session));
            UNIT_ASSERT(session.Commit());
            auto tag = tagsManager.GetDeviceTags().GetTagFromCache(car.Id, TTelematicsFirmwareTag::Type(), Now());
            UNIT_ASSERT(tag);
        }
        {
            auto command = TTelematicsCommandTag::ParseCommand("horn", car.Id, *server);
            UNIT_ASSERT(command);
            UNIT_ASSERT_VALUES_EQUAL(command->Code, NDrive::NVega::ECommandCode::YADRIVE_PANIC);
            UNIT_ASSERT(!command->Argument.IsNull());
            auto panic = command->Argument.Get<NDrive::NVega::TCommandRequest::TPanic>();
            UNIT_ASSERT_VALUES_EQUAL(panic.Type, panic.HORN);
            UNIT_ASSERT_VALUES_EQUAL(panic.Time, 42);
        }
        userPermissions = server->GetDriveAPI()->GetUserPermissions(USER_ID_DEFAULT1, TUserPermissionsFeatures());
        UNIT_ASSERT(userPermissions);
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            auto response = TTelematicsCommandTag::Command(car.Id, NDrive::NVega::ECommandCode::HORN, TDuration::Seconds(10), *userPermissions, server.Get(), session);
            UNIT_ASSERT(response.Initialized());
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            auto response = TTelematicsCommandTag::Command(car.Id, NDrive::NVega::ECommandCode::HORN, TDuration::Seconds(10), *userPermissions, server.Get(), session);
            UNIT_ASSERT(!response.Initialized());
            try {
                THttpStatusManagerConfig dummy;
                session.DoExceptionOnFail(dummy, server->GetLocalization());
            } catch (const TCodedException& e) {
                UNIT_ASSERT_VALUES_EQUAL(e.GetLocalizedTitle(), title);
                UNIT_ASSERT_VALUES_EQUAL(e.GetLocalizedMessage(), message);
            }
        }
    }

    Y_UNIT_TEST(FuelingTag) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eg(*server);;
        eg.BuildEnvironment();

        const TDriveAPI& driveApi = *server->GetDriveAPI();
        const IDriveTagsManager& manager = server->GetDriveAPI()->GetTagsManager();

        auto car = eg.CreateCar();
        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        UNIT_ASSERT(gServer.WaitCar(car.Id));
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(false));
        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            ITag::TPtr chargable = MakeAtomicShared<TChargableTag>("old_state_reservation");
            ITag::TPtr tag = ITag::TFactory::Construct("simple_fueling_tag");
            tag->SetName("simple_fueling_tag");
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(tag, USER_ROOT_DEFAULT, car.Id, server.Get(), session));
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(chargable, USER_ROOT_DEFAULT, car.Id, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            auto expectedTags = driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(car.Id, { "simple_fueling_tag" }, session);
            UNIT_ASSERT(expectedTags);
            UNIT_ASSERT_VALUES_EQUAL(expectedTags->size(), 1);
            auto tag = std::move(expectedTags->at(0));
            UNIT_ASSERT(tag);
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RemoveTag(tag, USER_ROOT_DEFAULT, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        {
            bool running = false;
            for (size_t i = 0; i < 42; i++) {
                running = emulator->GetContext().GetEngineStarted();
                if (running) {
                    break;
                } else {
                    Sleep(TDuration::Seconds(1));
                }
            }
            UNIT_ASSERT(running);
        }

        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(false));
        {
            auto offer = MakeAtomicShared<TStandartOfferReport>(new TStandartOffer, nullptr);
            offer->GetOfferPtrAs<IOffer>()->SetObjectId(car.Id).SetUserId(USER_ID_DEFAULT).SetDeadline(Now() + TDuration::Minutes(5));
            offer->GetOffer()->SetChargableAccounts({ "card", "bonus" });
            UNIT_ASSERT(server->GetOffersStorage()->StoreOffers({ offer }));
            UNIT_ASSERT(gServer.BookOffer(offer->GetOffer()->GetOfferId(), USER_ID_DEFAULT));
        }
        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            ITag::TPtr tag = ITag::TFactory::Construct("simple_fueling_tag");
            tag->SetName("simple_fueling_tag");
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().AddTag(tag, USER_ROOT_DEFAULT, car.Id, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            auto expectedTags = driveApi.GetTagsManager().GetDeviceTags().RestoreEntityTags(car.Id, { "simple_fueling_tag" }, session);
            UNIT_ASSERT(expectedTags);
            UNIT_ASSERT_VALUES_EQUAL(expectedTags->size(), 1);
            auto tag = std::move(expectedTags->at(0));
            UNIT_ASSERT(tag);
            UNIT_ASSERT(driveApi.GetTagsManager().GetDeviceTags().RemoveTag(tag, USER_ROOT_DEFAULT, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        {
            bool running = false;
            for (size_t i = 0; i < 42; i++) {
                running = emulator->GetContext().GetEngineStarted();
                if (running) {
                    break;
                } else {
                    Sleep(TDuration::Seconds(1));
                }
            }
            UNIT_ASSERT(!running);
        }
    }

    Y_UNIT_TEST(EvolvedTagsSearch) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eg.CreateUser("zxqfd-was-here");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"user_problem_tag_minor"}, dbTags, session));
            auto permissions = server->GetDriveAPI()->GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());

            auto tag = dbTags.front();
            auto newTag = tag.Clone(server->GetDriveAPI()->GetTagsHistoryContext());
            newTag->SetName("user_problem_tag_major");
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().EvolveTag(tag, newTag.GetData(), *permissions, server.Get(), session) && session.Commit());
        }

        {
            TTagsSearchRequest req({"user_problem_tag_major"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
        }
    }

    Y_UNIT_TEST(TagsSearch) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eg.CreateUser("zxqfd-was-here");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
            UNIT_ASSERT(searchResult.HasTotalMatched());
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {"user_problem_tag_major"}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            TTagsSearchRequest req({"user_problem_tag_major"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
            UNIT_ASSERT(searchResult.HasTotalMatched());
        }

        {
            TTagsSearchRequest req({"user_problem_tag_major", "user_problem_tag_minor"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
        }

        {
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_major"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {"user_problem_tag_major"}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
        }

        {
            TTagsSearchRequest req({"user_problem_tag_major"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
            UNIT_ASSERT(searchResult.HasTotalMatched());
        }

        {
            TTagsSearchRequest req({"user_problem_tag_major", "user_problem_tag_minor"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            auto searchResult = gServer.DoTagsSearch({"user_problem_tag_major"}, {});
            INFO_LOG << searchResult << Endl;
            UNIT_ASSERT(!searchResult.IsNull());
        }
        {
            auto registeredTags = driveApi.GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User);
            auto registeredTagNames = MakeVector(NContainer::Keys(registeredTags));
            INFO_LOG << "registered " << registeredTagNames.size() << " tags" << Endl;
            registeredTagNames.resize(100);
            auto searcher = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User);
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            for (auto&& left : registeredTagNames) {
                {
                    auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
                    auto count = searcher->GetCachedCount(left, tx);
                    if (!count) {
                        INFO_LOG << "skip " << left << Endl;
                        continue;
                    }
                }
                for (auto&& right : registeredTagNames) {
                    if (left == right) {
                        continue;
                    }
                    {
                        auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
                        auto count = searcher->GetCachedCount(right, tx);
                        if (!count) {
                            INFO_LOG << "skip " << right << Endl;
                            continue;
                        }
                    }
                    {
                        TTagsSearchRequest request({ left, right }, {}, {});
                        auto result = searcher->Search(request);
                        for (auto&& objectId : result.GetMatchedIds()) {
                            auto optionalObject = driveApi.GetTagsManager().GetUserTags().RestoreObject(objectId, tx);
                            UNIT_ASSERT(optionalObject);
                            UNIT_ASSERT(optionalObject->HasTag(left));
                            UNIT_ASSERT(optionalObject->HasTag(right));
                        }
                    }
                    {
                        TTagsSearchRequest request({ left }, { right }, {});
                        auto result = searcher->Search(request);
                        for (auto&& objectId : result.GetMatchedIds()) {
                            auto optionalObject = driveApi.GetTagsManager().GetUserTags().RestoreObject(objectId, tx);
                            UNIT_ASSERT(optionalObject);
                            UNIT_ASSERT(optionalObject->HasTag(left));
                            UNIT_ASSERT(!optionalObject->HasTag(right));
                        }
                    }
                    {
                        TTagsSearchRequest request({ left }, {}, { right });
                        auto result = searcher->Search(request);
                        for (auto&& objectId : result.GetMatchedIds()) {
                            auto optionalObject = driveApi.GetTagsManager().GetUserTags().RestoreObject(objectId, tx);
                            UNIT_ASSERT(optionalObject);
                            UNIT_ASSERT(optionalObject->HasTag(left));
                            UNIT_ASSERT(optionalObject->HasTag(right));
                        }
                    }
                }
            }
        }
    }

    Y_UNIT_TEST(TagsSearchUpdateData) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eg.CreateUser("zxqfd-was-here");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"user_problem_tag_minor"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            auto userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
            UNIT_ASSERT(gServer.StartTag(tag.GetTagId(), USER_ROOT_DEFAULT));
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            req.SetPerformer(USER_ROOT_DEFAULT);
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"user_problem_tag_minor"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0].Clone(server->GetDriveAPI()->GetTagsManager().GetContext());
            tag.MutableTagAs<TUserProblemTag>()->SetCarNumber("zx666qfd");
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(tag, "robot-frontend", session) && session.Commit());
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            req.SetPerformer(USER_ROOT_DEFAULT);
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }
    }

    Y_UNIT_TEST(TagsSearchOneOf) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eg.CreateUser("zxqfd-was-here");
        auto userId2 = eg.CreateUser("zxqfd2-was-here");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_major"), userId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TTagsSearchRequest req({}, {}, {"user_problem_tag_minor", "user_problem_tag_major"});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 2);
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor", "user_problem_tag_major"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
        }
    }

    Y_UNIT_TEST(TagsSearchResolveClass) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eg.CreateUser("zxqfd-was-here");
        auto userId2 = eg.CreateUser("zxqfd2-was-here");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_major"), userId2, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TTagsSearchRequest req({}, {}, {"@user_problem_tag"}, 300);
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            auto oks = 0;
            for (auto&& id : searchResult.GetMatchedIds()) {
                if (id == userId || id == userId2) {
                    ++oks;
                }
            }
            UNIT_ASSERT_VALUES_EQUAL(oks, 2);
        }

        {
            TTagsSearchRequest req({"@user_problem_tag"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
        }
    }

    Y_UNIT_TEST(PerformedTagsSearch) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto userId = eg.CreateUser("zxqfd-was-here");
        auto operatorId = eg.CreateUser("zxqfd2-was-here");
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        {
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            req.SetPerformer(userId);
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"user_problem_tag_minor"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            auto userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
            UNIT_ASSERT(gServer.StartTag(tag.GetTagId(), USER_ROOT_DEFAULT));
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            req.SetPerformer(operatorId);
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            req.SetPerformer(USER_ROOT_DEFAULT);
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, {"user_problem_tag_minor"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            auto tag = dbTags[0];
            auto userPermissions = driveApi.GetUserPermissions(USER_ROOT_DEFAULT, TUserPermissionsFeatures());
            UNIT_ASSERT(gServer.FinishTag(tag.GetTagId(), USER_ROOT_DEFAULT, false));
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            req.SetPerformer(USER_ROOT_DEFAULT);
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 0);
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            req.SetPerformer("");  // look for tags with no performer
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }

        {
            TTagsSearchRequest req({"user_problem_tag_minor"}, {}, {});
            auto searchResult = driveApi.GetTagsSearch(NEntityTagsManager::EEntityType::User)->Search(req);
            UNIT_ASSERT_VALUES_EQUAL(searchResult.GetMatchedIds().size(), 1);
        }
    }

    Y_UNIT_TEST(LockOnCertainTagsPerform) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto clientId = eg.CreateUser("client-was-there");
        auto supportId = USER_ROOT_DEFAULT;

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("lock_support").SetUserId(supportId);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, supportId, session) && session.Commit());
        }
        Sleep(TDuration::Seconds(10));

        {
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), clientId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_minor"), clientId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_major"), clientId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("user_problem_tag_major"), clientId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            UNIT_ASSERT(gServer.AddTag(new TUserProblemTag("blocked_old_license"), clientId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"user_problem_tag_minor", "user_problem_tag_major"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 4);
            }
            for (size_t i = 0; i < 3; ++i) {
                UNIT_ASSERT_C(gServer.StartTag(dbTags[i].GetTagId(), supportId), TStringBuilder() << i);
            }
            UNIT_ASSERT(!gServer.StartTag(dbTags[3].GetTagId(), supportId));
        }

        {
            TVector<TDBTag> dbTags;
            {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"blocked_old_license"}, dbTags, session));
                UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            }
            UNIT_ASSERT(gServer.StartTag(dbTags[0].GetTagId(), supportId));
        }
    }

    Y_UNIT_TEST(OneUserManyPerformers) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const TDriveAPI& driveApi = *server->GetDriveAPI();

        auto support1 = eg.CreateUser("skulik1-was-there", true, "active");
        auto support2 = eg.CreateUser("skulik2-was-there", true, "active");

        auto clientId = eg.CreateUser("client-was-there");

        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(support1).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, support1, session) && session.Commit());
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(support2).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, support2, session) && session.Commit());
        }

        {
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.topic1", "Hello, topic 1!", {}));
            UNIT_ASSERT(gServer.CommitChatAction(clientId, "support.topic2", "Hello, topic 2!", {}));
        }

        TString tag1id;
        TString tag2id;
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TVector<TDBTag> dbTags;
            UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ clientId }, {"support_chat_2_line"}, dbTags, session));
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 2);

            tag1id = dbTags[0].GetTagId();
            tag2id = dbTags[1].GetTagId();
        }

        UNIT_ASSERT(gServer.StartTag(tag1id, support1));
        UNIT_ASSERT(!gServer.StartTag(tag1id, support2));
        UNIT_ASSERT(gServer.StartTag(tag1id, USER_ROOT_DEFAULT));
        UNIT_ASSERT(gServer.StartTag(tag2id, support2));
    }

    Y_UNIT_TEST(FeedbackBonuses) {
        NDrive::TServerConfigGenerator gServer;
        NDrive::NTest::TScript script(gServer, Now() - TDuration::Minutes(2));
        using namespace NDrive::NTest;
        script.Add<TBuildEnv>();
        script.Add<TCreateCar>().SetPosition(TGeoCoord(37.5675511, 55.7323499));
        script.Add<TCreateAndBookOffer>().SetOfferType("standart_offer");
        script.Add<TDrop>(TDuration::Minutes(1)).SetExpectOK(true);
        UNIT_ASSERT(script.Execute());

        const TDriveAPI& driveApi = script.GetDriveAPI();
        const IDriveTagsManager& tagsManager = driveApi.GetTagsManager();
        const ITagsMeta& tagsMeta = tagsManager.GetTagsMeta();

        const TString userId = USER_ID_DEFAULT;
        const auto currentSession = gServer.GetCurrentSession(userId);
        INFO_LOG << currentSession.GetStringRobust() << Endl;
        const TString objectId = currentSession["segment"]["meta"]["object_id"].GetString();
        const TString sessionId = currentSession["segment"]["meta"]["session_id"].GetString();
        UNIT_ASSERT(objectId);
        UNIT_ASSERT(sessionId);
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto description = MakeAtomicShared<TFeedbackTraceTag::TDescription>("ugc_flat_tire", TFeedbackTraceTag::Type());
            description->SetBonusInfo("bb_feedback_flat_tire", 150);
            description->SetBonusLocalization("bonus button", "bonus message");
            description->SetLocalization("button", "message");
            UNIT_ASSERT(tagsMeta.RegisterTag(description, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto description = MakeAtomicShared<TFeedbackTraceTag::TDescription>("ugc_damaged_car", TFeedbackTraceTag::Type());
            description->SetBonusInfo("bb_feedback_damaged_car", 100);
            description->SetBonusLocalization("bonus button", "bonus message");
            description->SetLocalization("button", "message");
            UNIT_ASSERT(tagsMeta.RegisterTag(description, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto description = MakeAtomicShared<TFeedbackTraceTag::TDescription>("ugc_damaged_car_2", TFeedbackTraceTag::Type());
            description->SetBonusInfo("bb_feedback_damaged_car", 100);
            description->SetBonusLocalization("bonus button", "bonus message");
            description->SetLocalization("button", "message");
            UNIT_ASSERT(tagsMeta.RegisterTag(description, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            auto description = MakeAtomicShared<TFeedbackTraceTag::TDescription>("ugc_dirty_interior", TFeedbackTraceTag::Type());
            description->SetBonusInfo("bb_feedback_dirty_interior", 50);
            description->SetBonusLocalization("bonus button", "bonus message");
            description->SetLocalization("button", "message");
            UNIT_ASSERT(tagsMeta.RegisterTag(description, USER_ROOT_DEFAULT, session));
            UNIT_ASSERT(session.Commit());
        }
        TVector<TString> userTagNames = {
            "bb_feedback_damaged_car",
            "bb_feedback_dirty_interior",
            "bb_feedback_flat_tire",
        };
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto flatTire = tagsMeta.CreateTag("ugc_flat_tire");
            auto damagedCar = tagsMeta.CreateTag("ugc_damaged_car_2");
            UNIT_ASSERT(gServer.ActualizeTags({ damagedCar, flatTire }, {}, {}, userId, sessionId));
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> userTags;
            UNIT_ASSERT(tagsManager.GetUserTags().RestoreEntityTags(userId, userTagNames, userTags, session));
            UNIT_ASSERT_VALUES_EQUAL(userTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(userTags.at(0)->GetName(), "bb_feedback_flat_tire");
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto flatTire = tagsMeta.CreateTag("ugc_flat_tire");
            UNIT_ASSERT(gServer.ActualizeTags({ flatTire }, {}, {}, userId, sessionId));
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> userTags;
            UNIT_ASSERT(tagsManager.GetUserTags().RestoreEntityTags(userId, userTagNames, userTags, session));
            UNIT_ASSERT_VALUES_EQUAL(userTags.size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(userTags.at(0)->GetName(), "bb_feedback_flat_tire");
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto damagedCar = tagsMeta.CreateTag("ugc_damaged_car");
            UNIT_ASSERT(gServer.ActualizeTags({ damagedCar }, {}, {}, userId, sessionId));
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> userTags;
            UNIT_ASSERT(tagsManager.GetUserTags().RestoreEntityTags(userId, userTagNames, userTags, session));
            UNIT_ASSERT_VALUES_EQUAL(userTags.size(), 2);
            TSet<TString> names;
            for (auto&& userTag : userTags) {
                names.insert(userTag->GetName());
            }
            UNIT_ASSERT(names.contains("bb_feedback_damaged_car"));
            UNIT_ASSERT(names.contains("bb_feedback_flat_tire"));
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        {
            auto dirtyInterior = tagsMeta.CreateTag("ugc_dirty_interior");
            UNIT_ASSERT(gServer.ActualizeTags({ dirtyInterior }, {}, {}, userId, sessionId));
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TVector<TDBTag> userTags;
            UNIT_ASSERT(tagsManager.GetUserTags().RestoreEntityTags(userId, userTagNames, userTags, session));
            UNIT_ASSERT_VALUES_EQUAL(userTags.size(), 2);
            TSet<TString> names;
            for (auto&& userTag : userTags) {
                names.insert(userTag->GetName());
            }
            UNIT_ASSERT(!names.contains("bb_feedback_dirty_interior"));
        }
    }

    Y_UNIT_TEST(ExplicitZeroPriority) {
        NDrive::TServerConfigGenerator gServer;
        gServer.SetSensorApiName({});
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        const IDriveTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager();
        const TDeviceTagsManager& deviceTagsManager = tagsManager.GetDeviceTags();
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            auto tagDescription = MakeHolder<TTagDescription>();
            tagDescription->SetName("fakefake");
            tagDescription->SetType("aaa");
            tagDescription->SetDefaultPriority(1000);
            UNIT_ASSERT_C(tagsManager.GetTagsMeta().RegisterTag(std::move(tagDescription), USER_ID_DEFAULT, session), session.GetStringReport());
            UNIT_ASSERT(session.Commit());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }
        {
            auto tag = MakeAtomicShared<TTagTest>();
            UNIT_ASSERT(tag);
            tag->SetName("fakefake");
            tag->SetTagPriority(0);
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT_C(deviceTagsManager.AddTag(tag, USER_ROOT_DEFAULT, OBJECT_ID_DEFAULT, server.Get(), session), TStringBuilder() << session.GetMessages().GetStringReport() << Endl);
            UNIT_ASSERT_C(session.Commit(), TStringBuilder() << session.GetMessages().GetStringReport() << Endl);
        }
        {
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            auto expectedTags = deviceTagsManager.RestoreEntityTags(OBJECT_ID_DEFAULT, { "fakefake" }, session);
            UNIT_ASSERT(expectedTags);
            const auto& tags = *expectedTags;
            UNIT_ASSERT_VALUES_EQUAL(tags.size(), 1);
            auto tag = tags.at(0);
            UNIT_ASSERT(tag);
            UNIT_ASSERT(tag->HasTagPriority());
            UNIT_ASSERT_VALUES_EQUAL(tag->GetTagPriorityUnsafe(), 0);
        }
    }

    Y_UNIT_TEST(TelematicsConfiguration) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);

        UNIT_ASSERT(server->GetSettings().SetValue("tag.telematics_configuration.ingore_command_failures", "true", USER_ID_TECH));

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();

        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        auto car = eg.CreateCar();
        auto emulator = tmBuilder.BuildEmulator(car.IMEI);
        UNIT_ASSERT(gServer.WaitCar(car.Id));

        const IDriveTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager();
        const TDeviceTagsManager& deviceTagsManager = tagsManager.GetDeviceTags();
        {
            auto tag = MakeAtomicShared<TTelematicsConfigurationTag>();
            tag->SetApplyOnAdd(true);
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();
            UNIT_ASSERT(deviceTagsManager.AddTag(tag, USER_ROOT_DEFAULT, car.Id, server.Get(), session));
            UNIT_ASSERT(session.Commit());
        }
        ui64 hash = 0;
        for (ui64 i = 0; i < 5; ++i) {
            auto session = deviceTagsManager.BuildTx<NSQL::ReadOnly>();
            auto expectedTags = deviceTagsManager.RestoreEntityTags(car.Id, { TTelematicsConfigurationTag::Type() }, session);
            auto tag = expectedTags->at(0).MutableTagAs<TTelematicsConfigurationTag>();
            UNIT_ASSERT(tag);
            hash = tag->GetConfigurationHash();
            if (hash != 0) {
                break;
            }
            Sleep(TDuration::Seconds(10));
        }
        UNIT_ASSERT(hash);
    }

    Y_UNIT_TEST(UserLandingTagSubstitute) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();
        const IDriveTagsManager& tagsManager = server->GetDriveAPI()->GetTagsManager();

        {
            auto tagDescription = MakeHolder<TLandingUserTag::TDescription>();
            TLandingUserTag::TParametersMap defaultMap = {{"d1", "default one"}, {"d2", "default two"}, {"d3", "default three"}};
            tagDescription->SetDefaultContextParameters(defaultMap);
            tagDescription->SetLandingId("landing1");
            tagDescription->SetName("landing_with_context");
            tagDescription->SetType(TLandingUserTag::TypeName);
            tagDescription->SetDefaultPriority(0);
            UNIT_ASSERT(gServer.RegisterTag(std::move(tagDescription), USER_ROOT_DEFAULT));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }
        {
            auto tag = MakeHolder<TLandingUserTag>("landing_with_context");
            TLandingUserTag::TParametersMap cMap = {{"one", "eins"}, {"two", "zwei"}, {"three", "drei"}};
            tag->SetContextParameters(cMap);
            UNIT_ASSERT(gServer.AddTag(std::move(tag), USER_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }
        TVector<TDBTag> userTags;
        {
            auto session = tagsManager.GetUserTags().BuildTx<NSQL::Writable>();
            UNIT_ASSERT(tagsManager.GetUserTags().RestoreEntityTags(USER_ID_DEFAULT, {"landing_with_context"}, userTags, session));
            UNIT_ASSERT_VALUES_EQUAL(userTags.size(), 1);
        }

        auto tag = userTags[0].GetTagAs<TLandingUserTag>();
        UNIT_ASSERT(tag);

        UNIT_ASSERT(gServer.ModifyLanding("landing1", "{'text':'[three] [d2][one] [d1] blah [d3] blah'}", true, USER_ROOT_DEFAULT));
        SendGlobalMessage<NDrive::TCacheRefreshMessage>("drive_landings_history");

        TGeoCoord gc(37.5, 55.5);
        {
            NJson::TJsonValue landings = gServer.GetLandings(USER_ID_DEFAULT, &gc);
            INFO_LOG << "LANDEBUG" << landings << Endl;
            UNIT_ASSERT(landings["landings"].Has("landing1"));
            UNIT_ASSERT(landings["landings"]["landing1"]["text"].IsString());
            UNIT_ASSERT_VALUES_EQUAL(landings["landings"]["landing1"]["text"], "drei default twoeins default one blah default three blah");
        }

        auto td = tagsManager.GetTagsMeta().GetDescriptionByName("landing_with_context");
        auto description = dynamic_cast<const TLandingUserTag::TDescription*>(td.Get());
        UNIT_ASSERT(description);

        auto result = tag->SubstituteTemplate(NJson::TJsonValue(""), server.Get(), "[", "$");
        UNIT_ASSERT_VALUES_EQUAL(NJson::TJsonValue(""), result);

        auto textPure = NJson::TJsonValue("one two three");
        result = tag->SubstituteTemplate(textPure, server.Get(), "[[", "$]");
        UNIT_ASSERT_VALUES_EQUAL(textPure, result);

        auto textPure2 = NJson::TJsonValue("two$]");
        result = tag->SubstituteTemplate(textPure2, server.Get(), "[", "]");
        UNIT_ASSERT_VALUES_EQUAL(textPure2, result);

        result = tag->SubstituteTemplate(NJson::TJsonValue("[[[one]"), nullptr, "[", "]");
        UNIT_ASSERT_VALUES_EQUAL(NJson::TJsonValue("[[eins"), result);

        auto textNoDefaults1 = NJson::TJsonValue("[[one$] [[two$] [[three$]");
        auto resultNoDefaults1 = NJson::TJsonValue("eins zwei drei");
        result = tag->SubstituteTemplate(textNoDefaults1, server.Get(), "[[", "$]");
        UNIT_ASSERT_VALUES_EQUAL(resultNoDefaults1, result);

        auto textNoDefaults2 = NJson::TJsonValue("qwe [[one]qwe[[two] [[three] qweqwe");
        auto resultNoDefaults2 = NJson::TJsonValue("qwe einsqwezwei drei qweqwe");
        result = tag->SubstituteTemplate(textNoDefaults2, server.Get(), "[[", "]");
        UNIT_ASSERT_VALUES_EQUAL(resultNoDefaults2, result);

        auto textNoDefaults3 = NJson::TJsonValue("[one$]qwe[two$]qwe[three$]");
        auto resultNoDefaults3 = NJson::TJsonValue("einsqwezweiqwedrei");
        result = tag->SubstituteTemplate(textNoDefaults3, server.Get(), "[", "$]");
        UNIT_ASSERT_VALUES_EQUAL(resultNoDefaults3, result);

        auto textNoDefaults4 = NJson::TJsonValue("[three][two][one]");
        auto resultNoDefaults4 = NJson::TJsonValue("dreizweieins");
        result = tag->SubstituteTemplate(textNoDefaults4, nullptr, "[", "]");
        UNIT_ASSERT(result);
        UNIT_ASSERT_VALUES_EQUAL(resultNoDefaults4, result);

        auto textNoDefaults5 = NJson::TJsonValue("[[two$]");
        auto resultNoDefaults5 = NJson::TJsonValue("[zwei]");
        result = tag->SubstituteTemplate(textNoDefaults5, nullptr, "[", "$");
        UNIT_ASSERT_VALUES_EQUAL(resultNoDefaults5, result);

        auto textNoDefaults6 = NJson::TJsonValue("[one]][");
        auto resultNoDefaults6 = NJson::TJsonValue("eins][");
        result = tag->SubstituteTemplate(textNoDefaults6, nullptr, "[", "]");
        UNIT_ASSERT_VALUES_EQUAL(resultNoDefaults6, result);

        result = tag->SubstituteTemplate(textNoDefaults6, nullptr, "[", "]");
        UNIT_ASSERT(result);
        UNIT_ASSERT_VALUES_EQUAL(resultNoDefaults6, result);

        auto textDefaults1 = NJson::TJsonValue("[[%d2%]] [[%d1%]] [[%d3%]]");
        auto resultDefaults1 = NJson::TJsonValue("default two default one default three");
        result = tag->SubstituteTemplate(textDefaults1, server.Get(), "[[%", "%]]");
        UNIT_ASSERT_VALUES_EQUAL(resultDefaults1, result);

        auto textDefaults3 = NJson::TJsonValue("[d2]] [d1]][d3]][one]] ");
        auto resultDefaults3 = NJson::TJsonValue("default two default onedefault threeeins ");
        result = tag->SubstituteTemplate(textDefaults3, server.Get(), "[", "]]");
        UNIT_ASSERT_VALUES_EQUAL(resultDefaults3, result);

        result = tag->SubstituteTemplate(textDefaults3, nullptr, "[", "]]");
        UNIT_ASSERT_VALUES_EQUAL(NJson::TJsonValue("[d2]] [d1]][d3]]eins "), result);

        auto textSame1 = NJson::TJsonValue("[]");
        result = tag->SubstituteTemplate(textSame1, nullptr, "[", "]");
        UNIT_ASSERT_VALUES_EQUAL(textSame1, result);
        auto textSame2 = NJson::TJsonValue("[one");
        result = tag->SubstituteTemplate(textSame2, nullptr, "[", "]");
        UNIT_ASSERT_VALUES_EQUAL(textSame2, result);

        result = tag->SubstituteTemplate(NJson::TJsonValue("one"), nullptr, "", "");
        UNIT_ASSERT_VALUES_EQUAL(NJson::TJsonValue("eins"), result);
        result = tag->SubstituteTemplate(NJson::TJsonValue(" onetwo "), nullptr, "", "");
        UNIT_ASSERT_VALUES_EQUAL(NJson::TJsonValue(" einszwei "), result);
    }

    void CheckUserSettingTag(NDrive::TServerGuard& server, const TString& userId, const TString& tagName, const TString& field, const TMaybe<TString>& expectedValue = {}) {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        TVector<TDBTag> dbTags;
        UNIT_ASSERT(server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({ userId }, { tagName }, dbTags, session));
        UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
        auto tag = dbTags.front().GetTagAs<TUserDictionaryTag>();
        UNIT_ASSERT(tag);
        auto value = tag->GetField(field);
        if (expectedValue.Defined()) {
            UNIT_ASSERT(value.Defined());
            UNIT_ASSERT_VALUES_EQUAL(value.GetRef(), expectedValue.GetRef());
        } else {
            UNIT_ASSERT(!value.Defined());
        }
    }

    Y_UNIT_TEST(DictionaryTagAccess) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        auto userId = eg.CreateUser("dictionary-user", false);
        TUserSetting userSetting(userId);

        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            TDictionaryTagDescription description1;
            description1.SetName("dictionary_tag_1").SetType(TUserDictionaryTag::TypeName).SetDefaultPriority(0);
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(new TDictionaryTagDescription(description1), USER_ROOT_DEFAULT, session), session.GetStringReport());
            TDictionaryTagDescription description2;
            description2.SetName("dictionary_tag_2").SetType(TUserDictionaryTag::TypeName).SetDefaultPriority(0);
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(new TDictionaryTagDescription(description2), USER_ROOT_DEFAULT, session), session.GetStringReport());

            TString tagDescr = R"(
                {
                    "tag_flow": "",
                    "enable_report": false,
                    "fields": [
                        {
                            "default": "default_value",
                            "name": "def_test_field",
                            "materialize_default_value": false,
                            "type": "string",
                            "id": "def_test_field"
                        }
                    ],
                    "deprecated": false,
                    "transferred_to_double": false,
                    "enable_sessions": false,
                    "tag_flow_priority": 0,
                    "unique_policy": "rewrite",
                    "groupping_attributes": [],
                    "unlimited_history": false
                }
            )";

            NJson::TJsonValue descrJson = NJson::JSON_NULL;
            UNIT_ASSERT(NJson::ReadJsonTree(tagDescr, &descrJson));
            TDictionaryTagDescription description3;
            UNIT_ASSERT(description3.DoDeserializeMetaFromJson(descrJson));
            description3.SetName("dictionary_tag_3").SetType(TUserDictionaryTag::TypeName);
            UNIT_ASSERT_C(server->GetDriveAPI()->GetTagsManager().GetTagsMeta().RegisterTag(new TDictionaryTagDescription(description3), USER_ROOT_DEFAULT, session), session.GetStringReport());

            UNIT_ASSERT(session.Commit());
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();
        }

        // no tag, no field
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("tag_is_a_lie", "some_field", session);
            UNIT_ASSERT(!result);
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache("tag_is_a_lie", "some_field", session);
            UNIT_ASSERT(!result);
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache("tag_is_a_lie", "some_field", session, TInstant::Zero(), true);
            UNIT_ASSERT(!result);
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("tag_is_a_lie", "some_field", session, true);
            UNIT_ASSERT(!result);
        }

        // tag not dictionary
        UNIT_ASSERT(gServer.CommitChatAction(userId, "support.topic39473729kjf", "Hello!", {}));
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("support_chat_2_line", "some_field", session);
            UNIT_ASSERT(!result);
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache("support_chat_2_line", "some_field", session);
            UNIT_ASSERT(!result);
        }

        // add field -- no tag
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            userSetting.SetValue("dictionary_tag_1", "my_little_setting", "value1", "unittest", session);
            UNIT_ASSERT(session.Commit());
        }
        {
            CheckUserSettingTag(server, userId, "dictionary_tag_1", "my_little_setting", "value1");
        }
        SendGlobalMessage<NDrive::TCacheRefreshMessage>();

        // has tag, no field
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("dictionary_tag_1", "some_field", session);
            UNIT_ASSERT(!result);
            UNIT_ASSERT_C(result.GetError() == TUserSettingStatus::NotFound, result.GetError());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("dictionary_tag_1", "some_field", session, true);
            UNIT_ASSERT(!result);
            UNIT_ASSERT_C(result.GetError() == TUserSettingStatus::NotFound, result.GetError());
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache("dictionary_tag_1", "some_field", session);
            UNIT_ASSERT(!result);
            UNIT_ASSERT_C(result.GetError() == TUserSettingStatus::NotFound, result.GetError());
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache("dictionary_tag_1", "some_field", session, TInstant::Zero(), true);
            UNIT_ASSERT(!result);
            UNIT_ASSERT_C(result.GetError() == TUserSettingStatus::NotFound, result.GetError());
        }

        // has tag, has field
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("dictionary_tag_1", "my_little_setting", session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "value1");
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("dictionary_tag_1", "my_little_setting", session, true);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "value1");
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache("dictionary_tag_1", "my_little_setting", session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "value1");
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache("dictionary_tag_1", "my_little_setting", session, TInstant::Zero(), true);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "value1");
        }

        // add field -- has tag, new field
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(userSetting.SetValue("dictionary_tag_1", "my_little_setting2", "value2", "unittest", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            CheckUserSettingTag(server, userId, "dictionary_tag_1", "my_little_setting2", "value2");
        }

        // add field -- has tag, update field
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(userSetting.SetValue("dictionary_tag_1", "my_little_setting2", "value3", "unittest", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            CheckUserSettingTag(server, userId, "dictionary_tag_1", "my_little_setting2", "value3");
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("dictionary_tag_1", "my_little_setting2", session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "value3");
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache("dictionary_tag_1", "my_little_setting2", session, Now());
            UNIT_ASSERT(result );
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "value3");
        }

        // add field -- wrong type tag
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(!userSetting.SetValue("support_chat_2_line", "my_little_setting2", "value3", "unittest", session));
            INFO_LOG << "DictionaryTagAccess " << session.GetStringReport() << Endl;
            UNIT_ASSERT(session.GetStringReport().Contains("support_chat_2_line is not dictionary tag"));
        }

        // regexp
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(userSetting.SetValue("dictionary_tag_2", "my_little_setting", "value_dictionary_tag_2", "unittest", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            NDrive::TInfoEntitySession session;
            auto result = userSetting.GetValueFromCache(".*", "my_little_setting", session, Now(), true);
            UNIT_ASSERT(result);
            UNIT_ASSERT_C(result.GetRef() == "value_dictionary_tag_2" || result.GetRef() == "value1", result.GetRef());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("(^dictionary)", "my_little_setting", session, true);
            UNIT_ASSERT(result);
            UNIT_ASSERT_C(result.GetRef() == "value_dictionary_tag_2" || result.GetRef() == "value1", result.GetRef());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("^support", "my_little_setting", session, true);
            UNIT_ASSERT(!result);
            UNIT_ASSERT_C(result.GetError() == TUserSettingStatus::NotFound, result.GetError());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("(dictionary_tag_2|support_chat_2_line)", "my_little_setting", session, true);
            UNIT_ASSERT(result);
            UNIT_ASSERT_C(result.GetRef() == "value_dictionary_tag_2" || result.GetRef() == "value1", result.GetRef());
        }

        //default value -- no tag
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("dictionary_tag_3", "def_test_field", session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "default_value");
        }

        // default value -- regex
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("^dictionary_tag", "def_test_field", session, true);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "default_value");
        }

        // default value -- bad regex
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("^adictionary_", "def_test_field", session, true);
            UNIT_ASSERT(!result);
            UNIT_ASSERT_C(result.GetError() == TUserSettingStatus::NotFound, result.GetError());
        }

        //default value -- has tag
        {
            UNIT_ASSERT(gServer.AddTag(new TUserDictionaryTag("dictionary_tag_3"), userId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::User));
            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("dictionary_tag_3", "def_test_field", session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "default_value");
        }

        // default value -- overridden
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            UNIT_ASSERT(userSetting.SetValue("dictionary_tag_3", "def_test_field", "real value", "unittest", session));
            UNIT_ASSERT(session.Commit());
        }
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            auto result = userSetting.GetValue("dictionary_tag_3", "def_test_field", session);
            UNIT_ASSERT(result);
            UNIT_ASSERT_VALUES_EQUAL(result.GetRef(), "real value");
        }
    }

    Y_UNIT_TEST(UserOption) {
        NDrive::TServerConfigGenerator gServer;
        TServerConfigConstructorParams params(gServer.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eg(*server);
        eg.BuildEnvironment();

        auto userId = eg.CreateUser("dictionary-user", false);
        TUserSetting userSetting(userId);

        TUserOptionMeta optionMeta;
        optionMeta.SetId("dictionary_tag_test");
        optionMeta.MutableClientInfo().emplace_back("1111", NJson::JSON_NULL);
        auto permissions = server->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());

        TCommonOption userOption(*permissions.Get(), optionMeta);
        auto value = userOption.GetValue(TInstant::Zero());
        UNIT_ASSERT(!value.Defined());
        {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            auto& userTagsManager = server->GetDriveAPI()->GetTagsManager().GetUserTags();

            TDBTag settingsTag = server->GetDriveAPI()->GetUserSettings(userId, session);
            UNIT_ASSERT_C(!!settingsTag.HasData(), "incorrect settings configuration");

            TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
            UNIT_ASSERT_C(!!settingsData, "incorrect settings configuration");
            settingsData->SetField("dictionary_tag_test", "1111");

            UNIT_ASSERT(userTagsManager.AddTag(settingsTag.GetData(), userId, userId, server.Get(), session) && session.Commit());
        }
        auto value2 = userOption.GetValue(TInstant::Now());
        UNIT_ASSERT(value2.Defined());
        UNIT_ASSERT_VALUES_EQUAL(value2.GetRef(), "1111");
    }

    Y_UNIT_TEST(AccountTags) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);

        const TInstant now = Now();
        TString accountName = "limited";
        NDrive::NTest::TScript script(configGenerator, now);
        script.Add<NDrive::NTest::TBuildEnv>(TDuration::Zero());
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Control, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>(accountName);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(accountName).SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TDropCache>();

        ui32 accountId = 0;
        script.Add<NDrive::NTest::TCommonChecker>([&accountId, &accountName](NDrive::NTest::TRTContext& context) {
            auto session = context.GetDriveAPI().GetBillingManager().BuildSession(true);
            auto accounts = context.GetDriveAPI().GetBillingManager().GetAccountsManager().GetUserAccounts(USER_ID_DEFAULT, session);
            UNIT_ASSERT(accounts);
            for (const auto& acc : *accounts) {
                if (acc->GetUniqueName() == accountName) {
                    accountId = acc->GetId();
                }
            }
            UNIT_ASSERT(accountId);
        });

        TString tagName = "new_account_tag";
        TTagDescription::TPtr tag = new TTagDescription(tagName, TSimpleAccountTag::TypeName);
        script.Add<NDrive::NTest::TRegisterTag>(tag);

        script.Add<NDrive::NTest::TAddTag>(new TSimpleAccountTag(tagName)).SetObjectId(ToString(accountId)).SetEntityType(NEntityTagsManager::EEntityType::Account);
        script.Add<NDrive::NTest::TCheckHasTag>().SetTagName(tagName).SetObjectId(ToString(accountId)).SetEntityType(NEntityTagsManager::EEntityType::Account);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(PostEvolution) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());
        {
            auto evolution = MakeAtomicShared<TTagEvolutionAction>();
            evolution->SetName("evolution_with_post");
            evolution->SetTagNameFrom("simple1");
            evolution->SetTagNameTo("simple2");
            evolution->SetPostAction(NTagActions::ETagAction::DropPerform);
            RegisterAction(*env.GetServer(), evolution, "default_user");
        }
        auto api = env.GetServer()->GetDriveAPI();
        auto carId = TString{OBJECT_ID_DEFAULT};
        auto userId = TString{USER_ID_DEFAULT};
        UNIT_ASSERT(env->AddTag(MakeHolder<TDeviceTagRecord>("simple1"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));

        TString tagId;
        UNIT_ASSERT(env->GetTagId(carId, "simple1", USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car, tagId));
        UNIT_ASSERT(tagId);

        UNIT_ASSERT(env->StartTag(tagId, userId));
        {
            auto tx = api->template BuildTx<NSQL::ReadOnly>();
            auto optionalTag = api->GetTagsManager().GetDeviceTags().RestoreTag(tagId, tx);
            UNIT_ASSERT(optionalTag);
            auto tag = *optionalTag;
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), userId);
        }
        UNIT_ASSERT(env->EvolveTag("simple2", userId));
        {
            auto tx = api->template BuildTx<NSQL::ReadOnly>();
            auto optionalTag = api->GetTagsManager().GetDeviceTags().RestoreTag(tagId, tx);
            UNIT_ASSERT(optionalTag);
            auto tag = *optionalTag;
            UNIT_ASSERT(tag);
            UNIT_ASSERT_VALUES_EQUAL(tag->GetPerformer(), TString{});
        }
    }

    Y_UNIT_TEST(NewTag) {
        TTestEnvironment env;
        env.Execute(NDrive::NTest::TBuildEnv());

        const auto server = env.GetServer().Get();
        const auto& tagsManager = server->GetDriveDatabase().GetTagsManager();
        auto tagName = TString{"my_new_fancy_tag"};
        auto tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        {
            auto tag = MakeAtomicShared<TTagDescription>();
            tag->SetName(tagName);
            tag->SetType(TDeviceTagRecord::TypeName);
            UNIT_ASSERT(tagsManager.GetTagsMeta().RegisterTag(tag, USER_ROOT_DEFAULT, tx));
        }
        {
            auto tag = MakeAtomicShared<TDeviceTagRecord>(tagName);
            UNIT_ASSERT(tagsManager.GetDeviceTags().AddTag(tag, USER_ROOT_DEFAULT, OBJECT_ID_DEFAULT, server, tx));
        }
        UNIT_ASSERT(tx.Commit());
    }
}
