#include <drive/backend/ut/library/helper2.h>

#include <drive/backend/drivematics/signals/signal_configurations.h>
#include <drive/telematics/server/back/pusher.h>
#include <drive/telematics/server/library/config.h>

namespace {

const TString assignmentTagName = "assignment";
const TString owningTagName = "owning_car_tag";

template <class TAssertion>
void RepeatAssert(TAssertion&& assertion, size_t tryCount = 10, TDuration sleepDuration = TDuration::Seconds(10)) {
    for (size_t i = 0; i < tryCount; ++i) {
        if (assertion()) {
            return;
        }
        Sleep(sleepDuration);
    }
    UNIT_ASSERT(false);
}

void RegisterEngineSessionMetadata(const NDrive::IServer& server) {
    auto enableEngineSessionsTag = MakeAtomicShared<TTagDescription>();
    enableEngineSessionsTag->SetName("enable_engine_sessions");
    enableEngineSessionsTag->SetType(TDeviceTagRecord::TypeName);
    RegisterTag(server, enableEngineSessionsTag);

    auto performerAssignmentTag = MakeAtomicShared<TTagDescription>();
    performerAssignmentTag->SetName(assignmentTagName);
    performerAssignmentTag->SetType(TDeviceTagRecord::TypeName);
    RegisterTag(server, performerAssignmentTag);

    SetSetting(server, "telematics.engine_session.enable", "true");
    SetSetting(server, "telematics.engine_session.evolve_to_riding", "true");
    SetSetting(server, "telematics.engine_session.tag", enableEngineSessionsTag->GetName());
    SetSetting(server, "telematics.engine_session.offer_name", "engine_session");
    SetSetting(server, "telematics.engine_session.time_to_finish_session", "10s");
    SetSetting(server, "telematics.engine_session.time_to_parking", "10s");
    SetSetting(server, "telematics.engine_session.time_to_riding", "10s");
}

bool CheckSessionStage(TTestEnvironment& env, const size_t secondsToSleepFor, const TString& operatorId, const TString& expectedStage) {
    TString stage;
    RepeatAssert([&]() {
        auto currentSession = env->GetCurrentSession(operatorId);
        UNIT_ASSERT(currentSession.IsDefined());
        stage = currentSession["segment"]["session"]["current_performing"].GetStringRobust();
        return stage == expectedStage;
    }, 10, TDuration::Seconds(secondsToSleepFor));
    return stage == expectedStage;
}

void PrepareEnv(TTestEnvironment& env) {
    auto backendPusherOptions = MakeHolder<NDrive::TBackendPusherOptions>();
    backendPusherOptions->Endpoint = TStringBuilder() << "http://localhost:" << env.GetConfig().GetHttpServerOptions().Port << "/api/telematics/push";
    backendPusherOptions->SensorIds.insert(CAN_ENGINE_IS_ON);
    env.GetTelematicServerBuilder().GetConfig()->AddPusher(std::move(backendPusherOptions));

    env.Execute(NDrive::NTest::TBuildEnv());
    RegisterEngineSessionMetadata(*env.GetServer());
    SendGlobalMessage<NDrive::TCacheRefreshMessage>();

    {
        const auto& server = *env.GetServer();
        const auto& driveApi = *Yensured(server.GetDriveAPI());

        auto tagDescription = MakeAtomicShared<TTagDescription>();
        tagDescription->SetName(owningTagName);
        tagDescription->SetType(TDeviceTagRecord::TypeName);

        auto tx = driveApi.BuildTx<NSQL::Writable>();
        const auto& tagsMeta = driveApi.GetTagsManager().GetTagsMeta();
        UNIT_ASSERT_C(tagsMeta.RegisterTag(tagDescription, USER_ROOT_DEFAULT, tx), tx.GetStringReport());
        UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());
    }
}

}

Y_UNIT_TEST_SUITE(Telematics) {
    Y_UNIT_TEST(EngineSessions) {
        TTestEnvironment env;
        PrepareEnv(env);

        UNIT_ASSERT(env->AddTag(MakeAtomicShared<TDeviceTagRecord>("enable_engine_sessions"), OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        auto emulator = env.GetContext().GetEmulator();
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(true));

        UNIT_ASSERT(CheckSessionStage(env, 10, USER_ID_DEFAULT, TChargableTag::Riding));

        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(false));
        UNIT_ASSERT(CheckSessionStage(env, 10, USER_ID_DEFAULT, TChargableTag::Parking));

        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(true));
        UNIT_ASSERT(CheckSessionStage(env, 10, USER_ID_DEFAULT, TChargableTag::Riding));

        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(false));
        UNIT_ASSERT(CheckSessionStage(env, 10, USER_ID_DEFAULT, TChargableTag::Parking));

        UNIT_ASSERT(CheckSessionStage(env, 10, USER_ID_DEFAULT, "null"));
    }

    Y_UNIT_TEST(AssignedPerformer) {
        TTestEnvironment env;
        PrepareEnv(env);

        const auto& server = *env.GetServer();
        const auto carId = OBJECT_ID_DEFAULT;

        {
            server.GetSettings().SetValue("telematics.settings.linked_operator_tag_name", assignmentTagName, USER_ROOT_DEFAULT);

            auto assignmentTag = MakeAtomicShared<TDeviceTagRecord>(assignmentTagName);
            assignmentTag->SetPerformer(USER_ID_DEFAULT2);

            const auto& deviceTagsManager = Yensured(server.GetDriveAPI())->GetTagsManager().GetDeviceTags();
            auto session = deviceTagsManager.BuildTx<NSQL::Writable>();

            UNIT_ASSERT(deviceTagsManager.AddTag(assignmentTag, USER_ROOT_DEFAULT, carId, &server, session));
            UNIT_ASSERT(session.Commit());
        }

        {
            UNIT_ASSERT(env->AddTag(MakeAtomicShared<TDeviceTagRecord>("enable_engine_sessions"), carId, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
        }

        auto emulator = env.GetContext().GetEmulator();
        UNIT_ASSERT(emulator);
        UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(true));

        UNIT_ASSERT(CheckSessionStage(env, 10, USER_ID_DEFAULT2, TChargableTag::Riding));
    }
}

Y_UNIT_TEST_SUITE(Signals) {
    TRTBackgroundProcessContainer CreateSignal(const NDrive::TServer& server, const NJson::TJsonValue& jsonConfig, bool createTags) {
        auto tx = Yensured(server.GetDriveAPI())->template BuildTx<NSQL::Writable>();

        TMessagesCollector errors;
        auto configuration = NDrivematics::ISignalConfiguration::ConstructConfiguration(jsonConfig, &errors, server);
        UNIT_ASSERT_C(configuration, errors.GetStringReport());

        configuration->SetSignalId("unique_id");
        configuration->SetCarsFilter(TCarsFilter{});
        configuration->MutableCarsFilterRef().SetIncludeTagsFilter(TTagsFilter::BuildFromString(owningTagName));
        if (createTags) {
            configuration->CreateTags(USER_ROOT_DEFAULT, server, tx);
        }
        UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());

        auto vector = configuration->ConstructBackgroundProcesses();
        UNIT_ASSERT(vector.size() == 1);
        return std::move(vector.front());
    }

    void PrepareCar(const TString& carId, const NDrive::TServer& server) {
        const auto& driveApi = *Yensured(server.GetDriveAPI());
        const auto& carTagsManager = driveApi.GetTagsManager().GetDeviceTags();

        auto tx = driveApi.BuildTx<NSQL::Writable>();
        {
            auto tag = MakeHolder<TDeviceTagRecord>();
            tag->SetName(owningTagName);
            UNIT_ASSERT_C(carTagsManager.AddTag(tag.Release(), USER_ROOT_DEFAULT, carId, &server, tx), tx.GetStringReport());
        }
        {
            auto tag = MakeHolder<NDrivematics::THasTelematicsTag>();
            tag->SetName(NDrivematics::THasTelematicsTag::TypeName);
            UNIT_ASSERT_C(carTagsManager.AddTag(tag.Release(), USER_ROOT_DEFAULT, carId, &server, tx), tx.GetStringReport());
        }
        UNIT_ASSERT_C(tx.Commit(), tx.GetStringReport());
    }

    Y_UNIT_TEST(LabourOverdo) {
        TTestEnvironment env;
        PrepareEnv(env);
        const TInstant startTs = Now();
        const auto& server = *env.GetServer();
        const auto& driveApi = *Yensured(server.GetDriveAPI());
        const auto& carTagsManager = driveApi.GetTagsManager().GetDeviceTags();

        {
            UNIT_ASSERT(env->AddTag(MakeAtomicShared<TDeviceTagRecord>("enable_engine_sessions"), OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
            auto emulator = env.GetContext().GetEmulator();
            UNIT_ASSERT(emulator);
            UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(true));
            UNIT_ASSERT(CheckSessionStage(env, 10, USER_ID_DEFAULT, TChargableTag::Riding));
        }

        auto carId = env->GetCurrentSession(USER_ID_DEFAULT)["car"]["id"].GetStringRobust();
        UNIT_ASSERT(carId);
        PrepareCar(carId, server);

        NJson::TJsonValue signalConfig;
        signalConfig["type"] = "labour_overdo_signal";
        signalConfig["display_name"] = "test signal";

        auto tagGetter = [&]() {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            NSQL::TQueryOptions options;
            options.AddGenericCondition("object_id", carId);
            options.SetCustomCondition("object_id = '" + carId + "'");
            options.SetOrderBy({"history_event_id"});
            options.SetDescending(true);
            options.SetLimit(1);
            auto events = carTagsManager.GetHistoryManager().GetEvents({}, {startTs, Now()}, tx, options);
            UNIT_ASSERT_C(events, tx.GetStringReport());
            return
                events->size() > 0 &&
                events->front()->GetName() == "signal_unique_id";
        };

        {
            signalConfig["labour_limit"] = TDuration::Days(1).ToString();
            auto signal = CreateSignal(server, signalConfig, true);
            UNIT_ASSERT(signal.GetProcessSettings()->Execute(nullptr, server));
            UNIT_ASSERT(!tagGetter());
        }

        {
            signalConfig["labour_limit"] = TDuration::Zero().ToString();
            auto signal = CreateSignal(server, signalConfig, true);
            RepeatAssert(
                [&](){
                    UNIT_ASSERT(signal.GetProcessSettings()->Execute(nullptr, server));
                    return !!tagGetter();
                },
                10, TDuration::Seconds(10)
            );
        }

    }

    Y_UNIT_TEST(LittleRest) {
        TTestEnvironment env;
        PrepareEnv(env);
        const TInstant startTs = Now();
        const auto& server = *env.GetServer();
        const auto& driveApi = *Yensured(server.GetDriveAPI());
        const auto& carTagsManager = driveApi.GetTagsManager().GetDeviceTags();

        {
            UNIT_ASSERT(env->AddTag(MakeAtomicShared<TDeviceTagRecord>("enable_engine_sessions"), OBJECT_ID_DEFAULT, USER_ROOT_DEFAULT, NEntityTagsManager::EEntityType::Car));
            auto emulator = env.GetContext().GetEmulator();
            UNIT_ASSERT(emulator);
            UNIT_ASSERT(emulator->GetContext().TrySetEngineStarted(true));
            UNIT_ASSERT(CheckSessionStage(env, 10, USER_ID_DEFAULT, TChargableTag::Riding));
        }

        auto carId = env->GetCurrentSession(USER_ID_DEFAULT)["car"]["id"].GetStringRobust();
        UNIT_ASSERT(carId);
        PrepareCar(carId, server);

        NJson::TJsonValue signalConfig;
        signalConfig["type"] = "little_rest_signal";
        signalConfig["display_name"] = "test signal";

        auto tagGetter = [&]() {
            auto tx = driveApi.BuildTx<NSQL::ReadOnly>();
            NSQL::TQueryOptions options;
            options.AddGenericCondition("object_id", carId);
            options.AddGenericCondition("tag", "signal_unique_id");
            options.SetOrderBy({"history_event_id"});
            options.SetDescending(true);
            options.SetLimit(1);
            auto events = carTagsManager.GetHistoryManager().GetEvents({}, {startTs, Now()}, tx, options);
            UNIT_ASSERT_C(events, tx.GetStringReport());
            return
                events->size() > 0 &&
                events->front()->GetName() == "signal_unique_id";
        };

        {
            signalConfig["rest_duration"] = TDuration::Seconds(1).ToString();
            auto signal = CreateSignal(server, signalConfig, true);
            UNIT_ASSERT(signal.GetProcessSettings()->Execute(nullptr, server));
            UNIT_ASSERT(!tagGetter());
        }

        {
            signalConfig["rest_duration"] = TDuration::Days(365).ToString();
            auto signal = CreateSignal(server, signalConfig, true);
            UNIT_ASSERT(signal.GetProcessSettings()->Execute(nullptr, server));
            RepeatAssert(
                [&](){
                    UNIT_ASSERT(signal.GetProcessSettings()->Execute(nullptr, server));
                    return !!tagGetter();
                },
                10, TDuration::Seconds(10)
            );
        }
    }
}
