#include <drive/backend/abstract/base.h>
#include <drive/backend/billing/interfaces/entities.cpp>
#include <drive/backend/billing/redis_cache.h>
#include <drive/backend/roles/permissions.h>
#include <drive/backend/server/library/server.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/scripts/abstract.h>
#include <drive/backend/ut/library/scripts/common.h>
#include <drive/backend/ut/library/script.h>
#include <drive/backend/ut/library/scripts/billing.h>
#include <drive/backend/ut/library/scripts/session.h>

#include <drive/library/cpp/trust/entity.h>
#include <drive/library/cpp/trust/trust_cache.h>

#include <rtline/library/storage/redis/abstract.h>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/ptr.h>

Y_UNIT_TEST_SUITE(RedisTrust) {
    bool IsSubset(const TVector<NJson::TJsonValue>& p1, const TVector<NJson::TJsonValue>& p2) {
        if (p1.size() > p2.size()) {
            return false;
        }

        for (const auto& p: p1) {
            if (std::find(p2.begin(), p2.end(), p) == p2.end()) {
                return false;
            }
        }

        return true;
    }

    TVector<NJson::TJsonValue> CutJsonInfo(const TVector<NJson::TJsonValue>& data) {
        TVector<NJson::TJsonValue> result;
        for (const auto& json: data) {
            NJson::TJsonValue newJson;
            newJson["request_parameters"] = json["request_parameters"];
            newJson["name"] = json["name"];
            result.push_back(newJson);
        }
        return result;
    }

    TMaybe<TVector<NJson::TJsonValue>> CanonizePaymentMethods(const TMaybe<TVector<NDrive::NTrustClient::TPaymentMethod>>& paymentMethods, NDrive::TServer* server) {
        if (paymentMethods.Empty()) {
            return {};
        }

        auto filteredCards = TBillingManager::GetUserPaymentCards(paymentMethods.Get(), false);

        TVector<NJson::TJsonValue> correctPayments;
        for (auto& cards: *filteredCards) {
            auto json = GetPaymentMethodUserReport(cards, DefaultLocale, *server);
            NJson::TJsonValue cuttedInfo;
            json["request_parameters"].InsertValue("account_id", "card");
            correctPayments.push_back(json);
        };

        correctPayments = CutJsonInfo(correctPayments);

        return {correctPayments};
    }

    TMaybe<TVector<NJson::TJsonValue>> GetPaymentMethods(NDriveRedis::NKVAbstract::TRedisVersionedStorage& storage, const TString& key, NDrive::TServer* server) {
        TString jsonData;
        bool success = storage.GetValue(key, jsonData);

        if (!success || jsonData.Empty()) {
            return {};
        }

        NJson::TJsonValue json = NJson::ReadJsonFastTree(Base64Decode(jsonData));

        NDrive::ITrustStorage::TTimedMethods methods;
        success = methods.FromJson(json);
        if (!success) {
            return {};
        }

        return CanonizePaymentMethods({methods.Methods}, server);
    }

    TMaybe<TVector<NJson::TJsonValue>> GetPaymentMethods(NDrive::TServer* server, const TUserPermissions& permissions, bool cache=true) {
        auto userCards = server->GetDriveAPI()->GetUserPaymentMethods(permissions, *server, cache).GetValueSync();
        UNIT_ASSERT(!userCards.empty());

        return CanonizePaymentMethods(userCards, server);
    }

    TVector<NJson::TJsonValue> PaymentMethodsFromJson(const NJson::TJsonValue& response) {
        const NJson::TJsonValue::TArray& payments = response["user"]["billing"]["payment_methods"].GetArray();
        TVector<NJson::TJsonValue> methods;
        for (const auto& payment: payments) {
            methods.push_back(payment);
        }

        return CutJsonInfo(methods);
    }

    const TString DEFAULT_ACCOUNT = "y.drive";

    Y_UNIT_TEST(SuccessfulLaunch) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);

        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>(DEFAULT_ACCOUNT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(DEFAULT_ACCOUNT).SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TCommonChecker>([] (NDrive::NTest::TRTContext& /*context*/) {
            // Passed
        });

        script.Execute();
    }

    Y_UNIT_TEST(CorrectResponseWithoutSecondaryCache) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);

        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>(DEFAULT_ACCOUNT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(DEFAULT_ACCOUNT).SetUserId(USER_ID_DEFAULT);

        // Don't use secondary cache storage
        script.Add<NDrive::NTest::TSetSetting>().Set("lpm_client_cache_id", "local1");
        script.Add<NDrive::NTest::TSetSetting>().Set("use_lpm_client", "true");

        TVector<NJson::TJsonValue> payMethods;
        auto getPayMethods = [&payMethods](const NJson::TJsonValue& response) {
            // Obtained payments
            payMethods = PaymentMethodsFromJson(response);
        };
        script.Add<NDrive::NTest::TCheckCurrentSession>().SetChecker(getPayMethods).SetUserId(USER_ID_DEFAULT);

        script.Add<NDrive::NTest::TCommonChecker>([&payMethods](NDrive::NTest::TRTContext& context) {
            NDrive::TServer* server = context.GetServer().Get();
            TUserPermissions::TPtr permissions = context.GetDriveAPI().GetUserPermissions(USER_ID_DEFAULT);

            // Correct cards
            auto correctMethods = GetPaymentMethods(server, *permissions.Get());

            if (correctMethods.Empty()) {
                UNIT_ASSERT(payMethods.empty());
            }

            UNIT_ASSERT(IsSubset(*correctMethods, payMethods));
        });

        script.Execute();
    }

    Y_UNIT_TEST(BadSecondaryCacheStorage) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);

        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>(DEFAULT_ACCOUNT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(DEFAULT_ACCOUNT).SetUserId(USER_ID_DEFAULT);

        // Use secondary cache storage
        script.Add<NDrive::NTest::TSetSetting>().Set("lpm_client_cache_id", "redis1");
        script.Add<NDrive::NTest::TSetSetting>().Set("use_lpm_client", "true");

        // Incorrect secondary cache storage
        SetEnv("REDIS_HOSTNAME", "Fake");

        TVector<NJson::TJsonValue> payMethods;
        auto getPayMethods = [&payMethods](const NJson::TJsonValue& response) {
            // Obtained payments
            payMethods = PaymentMethodsFromJson(response);
        };
        script.Add<NDrive::NTest::TCheckCurrentSession>().SetChecker(getPayMethods).SetUserId(USER_ID_DEFAULT);

        script.Add<NDrive::NTest::TCommonChecker>([&payMethods] (NDrive::NTest::TRTContext& context) {
            NDrive::TServer* server = context.GetServer().Get();
            TUserPermissions::TPtr permissions = context.GetDriveAPI().GetUserPermissions(USER_ID_DEFAULT);

            // Correct cards
            auto correctMethods = GetPaymentMethods(server, *permissions.Get());

            if (correctMethods.Empty()) {
                UNIT_ASSERT(payMethods.empty());
            }

            UNIT_ASSERT(IsSubset(*correctMethods, payMethods));
        });

        script.Execute();
    }

    Y_UNIT_TEST(CorrectnessInSecondaryCacheStorage) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);

        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>(DEFAULT_ACCOUNT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(DEFAULT_ACCOUNT).SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TSetSetting>().Set("lpm_client_cache_id", "redis1");
        script.Add<NDrive::NTest::TSetSetting>().Set("use_lpm_client", "true");

        TString port = GetEnv("REDIS_PORT", "26379");
        NDriveRedis::NSession::TSessionConfig redisConfig {GetEnv("REDIS_HOSTNAME", "localhost"), FromStringWithDefault(port, 6379u), GetEnv("REDIS_TOKEN")};
        NDriveRedis::NKVAbstract::TRedisVersionedStorage storage({}, redisConfig);

        script.Add<NDrive::NTest::TCommonChecker>([&storage](NDrive::NTest::TRTContext& context) {
            NDrive::TServer* server = context.GetServer().Get();
            TUserPermissions::TPtr permissions = context.GetDriveAPI().GetUserPermissions(USER_ID_DEFAULT);

            TString userId = permissions->GetPaymethodsUid(server->GetSettings());

            // By default, storage is empty
            auto redisPayMethods = GetPaymentMethods(storage, userId, server);

            UNIT_ASSERT(redisPayMethods.Empty());
        });

        script.Add<NDrive::NTest::TCheckCurrentSession>().SetExpectOK(true);

        script.Add<NDrive::NTest::TCommonChecker>([&storage] (NDrive::NTest::TRTContext& context) {
            NDrive::TServer* server = context.GetServer().Get();
            TUserPermissions::TPtr permissions = context.GetDriveAPI().GetUserPermissions(USER_ID_DEFAULT);

            TString userId = permissions->GetPaymethodsUid(server->GetSettings());

            // True payment methods
            auto payMethods = GetPaymentMethods(server, *permissions.Get(), false);

            // True values should be in the cache
            auto redisPayMethods = GetPaymentMethods(storage, userId, server);
            UNIT_ASSERT(!redisPayMethods.Empty());

            UNIT_ASSERT(IsSubset(*redisPayMethods, *payMethods));

            // Waiting while values will be removed from cache
            Sleep(TDuration::Seconds(5));
            redisPayMethods = GetPaymentMethods(storage, userId, server);
            UNIT_ASSERT(redisPayMethods.Empty());
        });

        script.Execute();
    }

    Y_UNIT_TEST(Refreshness) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetNeedBackground(0);
        configGenerator.SetLogLevel(6);

        NDrive::NTest::TScript script(configGenerator);

        script.Add<NDrive::NTest::TBuildEnv>();
        script.Add<NDrive::NTest::TSetScriptUser>(USER_ROOT_DEFAULT);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Add, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TAddAdmActions>(TAdministrativeAction::EAction::Modify, TAdministrativeAction::EEntity::Wallet);
        script.Add<NDrive::NTest::TCreateAccount>(DEFAULT_ACCOUNT);
        script.Add<NDrive::NTest::TDropCache>();
        script.Add<NDrive::NTest::TLinkAccount>(DEFAULT_ACCOUNT).SetUserId(USER_ID_DEFAULT);
        script.Add<NDrive::NTest::TSetSetting>().Set("lpm_client_cache_id", "redis1");
        script.Add<NDrive::NTest::TSetSetting>().Set("use_lpm_client", "true");

        TString port = GetEnv("REDIS_PORT", "26379");
        NDriveRedis::NSession::TSessionConfig redisConfig {GetEnv("REDIS_HOSTNAME", "localhost"), (uint32_t)std::atoi(port.c_str()), GetEnv("REDIS_TOKEN")};
        NDriveRedis::NKVAbstract::TRedisVersionedStorage storage({}, redisConfig);

        TVector<NJson::TJsonValue> redisPayMethods;
        script.Add<NDrive::NTest::TCheckCurrentSession>().SetChecker([&redisPayMethods](const NJson::TJsonValue& response) {
            // Obtain old values
            redisPayMethods = PaymentMethodsFromJson(response);
            UNIT_ASSERT(!redisPayMethods.empty());
        }).SetUserId(USER_ID_DEFAULT);

        script.Add<NDrive::NTest::TCommonChecker>([] (NDrive::NTest::TRTContext& context) {
            /*  Refresh data */
            context.GetEGenerator().GetBillingMock().SetReply(R"(
                {
                    "status": "success",
                    "bound_payment_methods": [{
                        "region_id": 225,
                        "payment_method": "card",
                        "expiration_month": "07",
                        "binding_ts": "1536324485.656",
                        "id": "card-xxx",
                        "expired": false,
                        "card_bank": "TINKOFF BANK",
                        "system": "VISA",
                        "account": "510000****0257",
                        "expiration_year": "2021"
                    },
                    {
                        "region_id": 225,
                        "payment_method": "card",
                        "expiration_month": "07",
                        "binding_ts": "1536324485.656",
                        "id": "card-x12",
                        "expired": false,
                        "card_bank": "TINKOFF BANK",
                        "system": "VISA",
                        "account": "511100****0257",
                        "expiration_year": "2021"
                    }
                ]
                }
            )");
        });

        TMaybe<TVector<NJson::TJsonValue>> payMethods;
        script.Add<NDrive::NTest::TCommonChecker>([&redisPayMethods, &payMethods](NDrive::NTest::TRTContext& context) {
            NDrive::TServer *server = context.GetServer().Get();
            TUserPermissions::TPtr permissions = context.GetDriveAPI().GetUserPermissions(USER_ID_DEFAULT);

            TString userId = permissions->GetPaymethodsUid(server->GetSettings());

            // True payment methods
            payMethods = GetPaymentMethods(server, *permissions.Get(), false);

            /* Now payment methods are not equal */
            UNIT_ASSERT(!IsSubset(*payMethods, redisPayMethods));

            // Waiting while values will be removed from cache
            Sleep(TDuration::Seconds(5));
        });

        script.Add<NDrive::NTest::TCheckCurrentSession>().SetChecker([&redisPayMethods, &payMethods](const NJson::TJsonValue& response) {
            // Obtain new values
            redisPayMethods = PaymentMethodsFromJson(response);
            UNIT_ASSERT(!redisPayMethods.empty());

            /* Now payment methods are equal */
            UNIT_ASSERT(IsSubset(*payMethods, redisPayMethods));
        }).SetUserId(USER_ID_DEFAULT);

        script.Execute();
    }
}
