
#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/script.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/data/event_tag.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>

/*
    TString phone = GetString(requestData, "phone", true);
    TVector<TString> allowedStatuses = GetStrings(requestData, "allowed_statuses", false);
    TVector<TString> forbiddenStatuses = GetStrings(requestData, "forbidden_statuses", false);
    bool preferNoDebt = GetValue<bool>(requestData, "prefer_no_debt", false).GetOrElse(true);
    auto phoneVerified = GetValue<bool>(requestData, "phone_verified", false);
    bool lastPhoneVerification = GetValue<bool>(requestData, "last_phone_verification", false).GetOrElse(false);
*/

namespace NDrive::NTest {
    class TGetUserByPhone: public TAPIAction {
    public:
        using TChecker = std::function<void(const NJson::TJsonValue& response, TRTContext& context)>;
    private:
        using TBase = TAPIAction;
        static TFactory::TRegistrator<TGetUserByPhone> Registrator;

        R_FIELD(TString, Phone);
        R_OPTIONAL(TSet<TString>, AllowedStatuses);
        R_OPTIONAL(bool, PreferNoDebt);
        R_OPTIONAL(bool, PhoneVerified);
        R_OPTIONAL(bool, LastPhoneVerification);
        R_OPTIONAL(TChecker, Checker);

    protected:
        void DoExecute(TRTContext& context) override {
            NJson::TJsonValue post = NJson::JSON_MAP;
            if (Phone) {
                NJson::InsertField(post, "phone", Phone);
            }
            if (AllowedStatuses) {
                NJson::InsertField(post, "allowed_statuses", AllowedStatuses);
            }
            if (PreferNoDebt) {
                NJson::InsertField(post, " check_no_debt", PreferNoDebt);
            }
            if (PhoneVerified) {
                NJson::InsertField(post, "phone_verified", PhoneVerified);
            }
            if (LastPhoneVerification) {
                NJson::InsertField(post, "check_last_phone_verification", LastPhoneVerification);
            }
            NNeh::THttpRequest request;
            request.SetUri("/api/staff/user/find_by_phone");
            NUtil::THttpReply result = GetSendReply(context, "/api/staff/user/find_by_phone", post);
            NJson::TJsonValue resultReport = NJson::JSON_MAP;
            UNIT_ASSERT_C((result.Code() == 200) == GetExpectOK(), "Code: " + ToString(result.Code()));
            UNIT_ASSERT_C(NJson::ReadJsonFastTree(result.Content(), &resultReport), result.Content());
            if (Checker) {
                Checker.GetRef()(resultReport, context);
            }
        }

    public:
        using TBase::TBase;

        TGetUserByPhone(const TInstant startInstant)
            : TBase(startInstant) {
        }

        TString GetProcessorConfiguration() const override {
            TStringStream ss;
            ss << "<api/staff/user/find_by_phone>" << Endl;
            ss << "    AuthModuleName: fake" << Endl;
            ss << "    ProcessorType: find_user_by_phone" << Endl;
            ss << "</api/staff/user/find_by_phone>" << Endl;
            return ss.Str();
        }
    };

    TAPIAction::TFactory::TRegistrator<TGetUserByPhone> TGetUserByPhone::Registrator("TGetUserByPhone");

    Y_UNIT_TEST_SUITE(UserData) {

        void CheckEventTag(TRTContext& context, const ui32 eventCount, const TInstant start) {
            TVector<TDBTag> dbTags;
            auto session = context.GetDriveAPI().BuildTx<NSQL::ReadOnly>();
            UNIT_ASSERT_C(context.GetDriveAPI().GetTagsManager().GetUserTags().RestoreTags({ context.GetUserId() }, { "find_phone_event" }, dbTags, session), "Could not restore tags");
            UNIT_ASSERT_VALUES_EQUAL(dbTags.size(), 1);
            auto eventTag = dbTags.front().GetTagAs<IEventTag>();
            UNIT_ASSERT(eventTag);
            UNIT_ASSERT_VALUES_EQUAL(eventTag->GetEventCount({"find_user_by_phone"}, start), eventCount);
        }

        Y_UNIT_TEST(UserByPhoneStatusFilter) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            configGenerator.SetLogLevel(6);
            NDrive::NTest::TScript script(configGenerator);
            TMap<TString, TString> users;

            script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
                TString userId = context.GetEGenerator().CreateUser("active", false, "active", "+79874567890", true);
                users["active"] = userId;
                userId = context.GetEGenerator().CreateUser("active_another_phone", false, "active", "+79870000000", true);
                users["active_another_phone"] = userId;
                userId = context.GetEGenerator().CreateUser("onboarding", false, "onboarding", "+79874567890", true);
                users["onboarding"] = userId;
                userId = context.GetEGenerator().CreateUser("onboarding_another_phone", false, "onboarding", "+79334440000", true);
                users["onboarding_another_phone"] = userId;
                userId = context.GetEGenerator().CreateUser("blocked_another_phone", false, "blocked", "+79334440000", true);
                users["blocked_another_phone"] = userId;
                userId = context.GetEGenerator().CreateUser("fastonboarding", false, "fastonboarding", "+79874567890", true);
                users["fastonboarding"] = userId;
                userId = context.GetEGenerator().CreateUser("fastonboarding_not_verified", false, "fastonboarding", "+79874567890", false);
                users["fastonboarding_not_verified"] = userId;
                userId = context.GetEGenerator().CreateUser("fastregistered_not_verified", false, "fastregistered", "+79874567890", false);
                users["fastregistered_not_verified"] = userId;
            });

            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetAllowedStatuses({"active", "onboarding"})
                .SetChecker([&users](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                    UNIT_ASSERT(response["login"].IsString());
                    UNIT_ASSERT(response["user_id"].IsString());
                    TString responseLogin = response["login"].GetString();
                    TString responseUid = response["user_id"].GetString();
                    if (responseLogin == "a****e") {
                        UNIT_ASSERT_VALUES_EQUAL(responseUid, users["active"]);
                    } else if (responseLogin == "on*******g") {
                        UNIT_ASSERT_VALUES_EQUAL(responseUid, users["onboarding"]);
                    } else {
                        UNIT_ASSERT(false);
                    }
            });

            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetAllowedStatuses({"blocked"}).SetPhoneVerified(false).SetExpectOK(false);

            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetAllowedStatuses({"fastonboarding", "fastregistered"}).SetPhoneVerified(true)
                .SetChecker([&users](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                    UNIT_ASSERT_VALUES_EQUAL(response["user_id"].GetString(), users["fastonboarding"]);
            });

            script.Add<TGetUserByPhone>().SetPhone("79874567890").SetAllowedStatuses({"fastonboarding", "fastregistered"}).SetPhoneVerified(true)
                .SetChecker([&users](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                    UNIT_ASSERT_VALUES_EQUAL(response["user_id"].GetString(), users["fastonboarding"]);
            });

            script.Add<TGetUserByPhone>().SetPhone("89874567890").SetAllowedStatuses({"fastonboarding", "fastregistered"}).SetPhoneVerified(true)
                .SetChecker([&users](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                    UNIT_ASSERT_VALUES_EQUAL(response["user_id"].GetString(), users["fastonboarding"]);
            });

            script.Execute();
        }

        Y_UNIT_TEST(AccessEventTag) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            configGenerator.SetLogLevel(6);
            NDrive::NTest::TScript script(configGenerator);
            TMap<TString, TString> users;
            TInstant start = Now();

            NJson::TJsonValue descJson;
            descJson["type"] = "user_event_tag";
            descJson["name"] = "find_phone_event";
            descJson["display_name"] = "phone event tag";
            descJson["comment"] = "";
            script.Add<NDrive::NTest::TRegisterTag>(descJson);

            script.Add<TCommonChecker>([&users](NDrive::NTest::TRTContext& context) {
                UNIT_ASSERT(context.GetServer()->GetSettings().SetValue("handlers.api/staff/user/find_by_phone.event_tag_name", "find_phone_event", USER_ROOT_DEFAULT));
                UNIT_ASSERT(context.GetServer()->GetSettings().SetValue("handlers.api/staff/user/find_by_phone.event_name", "find_user_by_phone", USER_ROOT_DEFAULT));
                UNIT_ASSERT(context.GetServer()->GetSettings().SetValue("handlers.api/staff/user/find_by_phone.requests_per_time", "2", USER_ROOT_DEFAULT));
                TString requestUserId = context.GetEGenerator().CreateUser("active", false, "active");
                users["request_user"] = requestUserId;
                TString userId = context.GetEGenerator().CreateUser("active", false, "active", "+79874567890", true);
                users["active"] = userId;
                context.SetUserId(requestUserId);
                SendGlobalMessage<NDrive::TCacheRefreshMessage>();
            });

            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetAllowedStatuses({"active"})
                .SetChecker([](const NJson::TJsonValue& response, TRTContext& /*context*/) {
                    UNIT_ASSERT(!response["user_id"].GetString().empty());
            });

            script.Add<TCommonChecker>([&start](NDrive::NTest::TRTContext& context) {
                CheckEventTag(context, 1, start);
            });

            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetAllowedStatuses({"blocked"}).SetPhoneVerified(false).SetExpectOK(false);

            script.Add<TCommonChecker>([&start](NDrive::NTest::TRTContext& context) {
                CheckEventTag(context, 2, start);
            });
            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetAllowedStatuses({"active"}).SetPhoneVerified(true).SetExpectOK(false);
            script.Add<TSetSetting>().Set("handlers.api/staff/user/find_by_phone.requests_per_time", "50");
            script.Add<TCommonChecker>([&start](NDrive::NTest::TRTContext& context) {
                CheckEventTag(context, 2, start);
            });

            script.Add<TSetSetting>().Set("handlers.api/staff/user/find_by_phone.count_on_fail", "false");
            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetAllowedStatuses({"blocked"}).SetPhoneVerified(false).SetExpectOK(false);

            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetAllowedStatuses({"active"}).SetPhoneVerified(true).SetExpectOK(true);
            script.Add<TCommonChecker>([&start](NDrive::NTest::TRTContext& context) {
                CheckEventTag(context, 3, start);
            });
            script.Add<TSetSetting>().Set("handlers.api/staff/user/find_by_phone.seconds_between_requests", "86400");
            script.Add<TGetUserByPhone>().SetPhone("+79874567890").SetPhoneVerified(false).SetExpectOK(false);
            script.Add<TCommonChecker>([&start](NDrive::NTest::TRTContext& context) {
                CheckEventTag(context, 3, start);
            });

            script.Execute();
        }

        Y_UNIT_TEST(UserLoginObfuscation) {
            NDrive::TServerConfigGenerator configGenerator;
            TServerConfigConstructorParams params(configGenerator.GetString().data());
            NDrive::TServerConfig config(params);
            NDrive::TServerGuard server(config);
            TEnvironmentGenerator eGenerator(*server.Get());
            eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);
            const TDriveAPI& driveApi = *server->GetDriveAPI();
            {
                auto userId = eGenerator.CreateUser("ab", true, "active");
                auto userData = driveApi.GetUsersData()->FetchInfo(userId).begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(userData.GetObfuscatedLogin(), "ab");
            }
            {
                auto userId = eGenerator.CreateUser("abc", true, "active");
                auto userData = driveApi.GetUsersData()->FetchInfo(userId).begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(userData.GetObfuscatedLogin(), "a*c");
            }
            {
                auto userId = eGenerator.CreateUser("abcd", true, "active");
                auto userData = driveApi.GetUsersData()->FetchInfo(userId).begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(userData.GetObfuscatedLogin(), "a**d");
            }
            {
                auto userId = eGenerator.CreateUser("abcde", true, "active");
                auto userData = driveApi.GetUsersData()->FetchInfo(userId).begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(userData.GetObfuscatedLogin(), "ab**e");
            }
            {
                auto userId = eGenerator.CreateUser("abcdef", true, "active");
                auto userData = driveApi.GetUsersData()->FetchInfo(userId).begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(userData.GetObfuscatedLogin(), "ab***f");
            }
            {
                auto userId = eGenerator.CreateUser("abcdefghijklmnop", true, "active");
                auto userData = driveApi.GetUsersData()->FetchInfo(userId).begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(userData.GetObfuscatedLogin(), "ab*************p");
            }
            {
                auto userId = eGenerator.CreateUser("abc.defg@hij-klmnop", true, "active");
                auto userData = driveApi.GetUsersData()->FetchInfo(userId).begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(userData.GetObfuscatedLogin(), "abc.***g@**j-*****p");
            }
            {
                auto userId = eGenerator.CreateUser(":??;№).<", true, "active");
                auto userData = driveApi.GetUsersData()->FetchInfo(userId).begin()->second;
                UNIT_ASSERT_VALUES_EQUAL(userData.GetObfuscatedLogin(), ":??;№).<");
            }
        }
    }
}
