#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>

#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>
#include <drive/tests/library/database.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/uuid.h>

#include <util/random/fast.h>


#define TEST_USER_UID "4009159420"
#define TEST_USER_ID "fcb4e4d4-24d4-4fd3-b0bb-8463721b832a"
#define NEW_USER_ID "deadbeef-0000-0000-0000-8463721b832a"


Y_UNIT_TEST_SUITE(UsersAdmin) {

    NJson::TJsonValue GetByRevision(const NJson::TJsonValue& json, const TString& revision) {
        for (const auto& revisionMap : json.GetArray()) {
            UNIT_ASSERT(revisionMap.IsMap());
            if (revisionMap["revision"].GetStringRobust() == revision) {
                return revisionMap;
                break;
            }
        }
        return std::move(NJson::JSON_UNDEFINED);
    }

    Y_UNIT_TEST(UserRootView) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        TString currentRevision = "test_main_revision";
        bool hasDocumentType = true;

        // put the required data in the testing "datasync" at first
        {
            auto userInfo = TUserContacts("4009159420");
            TUserPassportData passport;
            passport.Parse(TDocumentsHelper::CreateRandomPassportDataJson(hasDocumentType));
            auto futurePassport = driveApi.GetPrivateDataClient().UpdatePassport(userInfo, currentRevision, passport);

            TUserDrivingLicenseData drivingLicense;
            drivingLicense.Parse(TDocumentsHelper::CreateRandomDrivingLicenseDataJson());
            auto futureDL = driveApi.GetPrivateDataClient().UpdateDrivingLicense(userInfo, currentRevision, drivingLicense);

            auto waiter = NThreading::WaitAll(futurePassport, futureDL);

            waiter.Wait();
            waiter.GetValue();
        }

        // check
        {
            auto response = configGenerator.GetUserAdminInfo(TEST_USER_ID, USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(response != NJson::JSON_NULL, TStringBuilder() << response);
            UNIT_ASSERT_VALUES_EQUAL(response["uid"], TEST_USER_UID);
            UNIT_ASSERT(response["documents"].IsDefined());
            UNIT_ASSERT(response["documents"]["passport"].IsDefined());
            UNIT_ASSERT(response["documents"]["driving_license"].IsDefined());
            const NJson::TJsonValue passport = GetByRevision(response["documents"]["passport"], currentRevision);
            const NJson::TJsonValue divingLicense = GetByRevision(response["documents"]["driving_license"], currentRevision);
            UNIT_ASSERT(divingLicense.IsDefined());
            UNIT_ASSERT(passport.IsDefined());

            INFO_LOG << response << Endl;
            UNIT_ASSERT_VALUES_EQUAL(passport.GetMap().size(), TDocumentsHelper::PassportStringFields.size() + TDocumentsHelper::PassportDateFields.size() + ui16(hasDocumentType) + 1);
            UNIT_ASSERT_VALUES_EQUAL(divingLicense.GetMap().size(), TDocumentsHelper::DrivingLicenseStringFields.size() + TDocumentsHelper::DrivingLicenseDateFields.size() + 1);
        }
    }

    Y_UNIT_TEST(UserBasicInfoEdit) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        {
            for (ui32 iter = 0; iter < 5; ++iter) {
                NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData();
                UNIT_ASSERT(configGenerator.UpdateUserInfo(TEST_USER_ID, updateJson, USER_ROOT_DEFAULT));

                SendGlobalMessage<NDrive::TCacheRefreshMessage>();

                auto response = configGenerator.GetUserAdminInfo(TEST_USER_ID, USER_ROOT_DEFAULT);
                UNIT_ASSERT_C(response != NJson::JSON_NULL, TStringBuilder() << response);
                UNIT_ASSERT_VALUES_EQUAL(response["uid"], TEST_USER_UID);
                for (auto&& field : TDocumentsHelper::UserProfileFields) {
                    if (field == "phone") {
                        UNIT_ASSERT_VALUES_EQUAL(updateJson[field], response["setup"]["phone"]["number"]);
                    } else if (field == "email") {
                        UNIT_ASSERT_VALUES_EQUAL(updateJson[field], response["setup"]["email"]["address"]);
                    } else {
                        UNIT_ASSERT_VALUES_EQUAL(response[field], updateJson[field]);
                    }
                }
            }
        }
    }

    Y_UNIT_TEST(UserDocumentsInfoEdit) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        // With passport, without driving license
        {
            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true);
            UNIT_ASSERT(configGenerator.UpdateUserInfo(TEST_USER_ID, updateJson, USER_ROOT_DEFAULT));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto response = configGenerator.GetUserAdminInfo(TEST_USER_ID, USER_ROOT_DEFAULT);
            UNIT_ASSERT(response["passport_revision"] != NJson::JSON_NULL);
            TString passportRevision = response["passport_revision"].GetString();
            const NJson::TJsonValue responseJson = GetByRevision(response["documents"]["passport"], passportRevision);

            UNIT_ASSERT(responseJson != NJson::JSON_UNDEFINED);
            auto passportJson = updateJson["passport"];

            for(auto&& field : TDocumentsHelper::PassportStringFields) {
                UNIT_ASSERT_VALUES_EQUAL(passportJson[field], responseJson[field]);
            }
            for (auto&& field : TDocumentsHelper::PassportDateFields) {
                UNIT_ASSERT_VALUES_EQUAL(passportJson[field], responseJson[field]);
            }
        }

        // With driving license, without passport
        {
            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(false, true);
            UNIT_ASSERT(configGenerator.UpdateUserInfo(TEST_USER_ID, updateJson, USER_ROOT_DEFAULT));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto response = configGenerator.GetUserAdminInfo(TEST_USER_ID, USER_ROOT_DEFAULT);
            UNIT_ASSERT(response["driving_license_revision"] != NJson::JSON_NULL);
            TString drivingLicenseRevision = response["driving_license_revision"].GetString();
            const NJson::TJsonValue responseJson = GetByRevision(response["documents"]["driving_license"], drivingLicenseRevision);

            UNIT_ASSERT(responseJson != NJson::JSON_UNDEFINED);
            auto drivingLicenseJson = updateJson["driving_license"];

            for(auto&& field : TDocumentsHelper::DrivingLicenseStringFields) {
                UNIT_ASSERT_VALUES_EQUAL(drivingLicenseJson[field], responseJson[field]);
            }
            for (auto&& field : TDocumentsHelper::DrivingLicenseDateFields) {
                UNIT_ASSERT_VALUES_EQUAL(drivingLicenseJson[field], responseJson[field]);
            }
        }

        // Both
        {
            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            UNIT_ASSERT(configGenerator.UpdateUserInfo(TEST_USER_ID, updateJson, USER_ROOT_DEFAULT));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto response = configGenerator.GetUserAdminInfo(TEST_USER_ID, USER_ROOT_DEFAULT);
            UNIT_ASSERT(response["driving_license_revision"] != NJson::JSON_NULL);
            TString drivingLicenseRevision = response["driving_license_revision"].GetString();
            const NJson::TJsonValue responseJson = GetByRevision(response["documents"]["driving_license"], drivingLicenseRevision);

            UNIT_ASSERT(responseJson != NJson::JSON_UNDEFINED);
            auto drivingLicenseJson = updateJson["driving_license"];

            for(auto&& field : TDocumentsHelper::DrivingLicenseStringFields) {
                UNIT_ASSERT_VALUES_EQUAL(drivingLicenseJson[field], responseJson[field]);
            }
            for (auto&& field : TDocumentsHelper::DrivingLicenseDateFields) {
                UNIT_ASSERT_VALUES_EQUAL(drivingLicenseJson[field], responseJson[field]);
            }

            UNIT_ASSERT(response["passport_revision"] != NJson::JSON_NULL);
            TString passportRevision = response["passport_revision"].GetString();
            const NJson::TJsonValue responsePassportJson = GetByRevision(response["documents"]["passport"], passportRevision);

            UNIT_ASSERT(responsePassportJson != NJson::JSON_UNDEFINED);
            auto passportJson = updateJson["passport"];

            for(auto&& field : TDocumentsHelper::PassportStringFields) {
                UNIT_ASSERT_VALUES_EQUAL(passportJson[field], responsePassportJson[field]);
            }
            for (auto&& field : TDocumentsHelper::PassportDateFields) {
                UNIT_ASSERT_VALUES_EQUAL(passportJson[field], responsePassportJson[field]);
            }
        }
    }

    Y_UNIT_TEST(UserDocumentsInfoEditHistory) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        auto userId = eGenerator.CreateUser("skulik-was-there");

        TVector<NJson::TJsonValue> modificationsHistory;
        for (ui32 iter = 0; iter < 10; ++iter) {
            NJson::TJsonValue updateJson = TDocumentsHelper::CreateRandomUserAccountData(true, true);
            modificationsHistory.push_back(updateJson);
            UNIT_ASSERT(configGenerator.UpdateUserInfo(userId, updateJson, USER_ROOT_DEFAULT));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto response = configGenerator.GetUserAdminInfo(userId, USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(response != NJson::JSON_NULL, TStringBuilder() << response);
            for (auto&& field : TDocumentsHelper::UserProfileFields) {
                if (field == "phone") {
                    UNIT_ASSERT_VALUES_EQUAL(updateJson[field], response["setup"]["phone"]["number"]);
                } else if (field == "email") {
                    UNIT_ASSERT_VALUES_EQUAL(updateJson[field], response["setup"]["email"]["address"]);
                } else {
                    UNIT_ASSERT_VALUES_EQUAL(response[field], updateJson[field]);
                }
            }
        }

        {
            auto receivedHistory = configGenerator.GetUserEditHistory(userId, USER_ROOT_DEFAULT);
            UNIT_ASSERT(receivedHistory != NJson::JSON_NULL);

            INFO_LOG << receivedHistory << Endl;

            auto history = receivedHistory["history"].GetArray();
            for (size_t i = 0; i < 10; ++i) {
                auto expected = modificationsHistory[i];
                auto received = history[i+2]["object"];
                INFO_LOG << history[i] << Endl;
                for (auto&& field : TDocumentsHelper::UserProfileFields) {
                    if (field == "phone") {
                        UNIT_ASSERT_VALUES_EQUAL(expected[field], received["setup"]["phone"]["number"]);
                    } else if (field == "email") {
                        UNIT_ASSERT_VALUES_EQUAL(expected[field], received["setup"]["email"]["address"]);
                    } else {
                        UNIT_ASSERT_VALUES_EQUAL(expected[field], received[field]);
                    }
                }

                auto passportRevision = received["passport_revision"].GetString();
                const NJson::TJsonValue passportReceived = GetByRevision(receivedHistory["documents"]["passport"], passportRevision);

                UNIT_ASSERT_C(passportReceived != NJson::JSON_UNDEFINED, TStringBuilder() << receivedHistory["documents"]["passport"]);

                auto drivingLicenseRevision = received["driving_license_revision"].GetString();
                const NJson::TJsonValue drivingLicenseReceived = GetByRevision(receivedHistory["documents"]["driving_license"], drivingLicenseRevision);;

                UNIT_ASSERT_C(drivingLicenseReceived != NJson::JSON_UNDEFINED, TStringBuilder() << receivedHistory["documents"]["driving_license"]);

                {
                    auto passportExpected = modificationsHistory[i]["passport"];

                    for(auto&& field : TDocumentsHelper::PassportStringFields) {
                        UNIT_ASSERT_VALUES_EQUAL(passportExpected[field], passportReceived[field]);
                    }
                    for (auto&& field : TDocumentsHelper::PassportDateFields) {
                        UNIT_ASSERT_VALUES_EQUAL(passportExpected[field], passportReceived[field]);
                    }
                }

                {
                    auto drivingLicenseExpected = modificationsHistory[i]["driving_license"];

                    for(auto&& field : TDocumentsHelper::DrivingLicenseStringFields) {
                        UNIT_ASSERT_VALUES_EQUAL(drivingLicenseExpected[field], drivingLicenseReceived[field]);
                    }
                    for (auto&& field : TDocumentsHelper::DrivingLicenseDateFields) {
                        UNIT_ASSERT_VALUES_EQUAL(drivingLicenseExpected[field], drivingLicenseReceived[field]);
                    }
                }

            }
        }
    }

    Y_UNIT_TEST(NewUserCreation) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TTelematicServerBuilder tmBuilder;
        tmBuilder.Run();
        auto emulator = tmBuilder.BuildEmulator(OBJECT_IMEI_DEFAULT);

        {
            NJson::TJsonValue requestBody = NJson::JSON_MAP;
            requestBody["id"] = NEW_USER_ID;
            requestBody["uid"] = "zzz";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(NEW_USER_ID, requestBody, USER_ROOT_DEFAULT));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto gData = driveApi.GetUsersData()->FetchInfo(NEW_USER_ID);
            auto result = gData.GetResult();
            UNIT_ASSERT(!result.empty());
        }

        // Should work
        {
            NJson::TJsonValue requestBody = NJson::JSON_MAP;
            requestBody["id"] = NEW_USER_ID;
            requestBody["uid"] = 482364;
            UNIT_ASSERT(configGenerator.UpdateUserInfo(NEW_USER_ID, requestBody, USER_ROOT_DEFAULT));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto response = configGenerator.GetUserAdminInfo(NEW_USER_ID, USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(response != NJson::JSON_NULL, TStringBuilder() << response);
        }

        // Should also work
        {
            NJson::TJsonValue requestBody = NJson::JSON_MAP;
            TString newId = NUtil::CreateUUID();
            requestBody["id"] = newId;
            requestBody["phone"] = "+375257064080";
            UNIT_ASSERT(configGenerator.UpdateUserInfo(newId, requestBody, USER_ROOT_DEFAULT));

            SendGlobalMessage<NDrive::TCacheRefreshMessage>();

            auto gData = driveApi.GetUsersData()->FetchInfo(newId);
            UNIT_ASSERT_VALUES_EQUAL(gData.MutableResult().begin()->second.GetPhone(), "+375257064080");
        }
    }

    Y_UNIT_TEST(UserDeletion) {
        NDrive::TServerConfigGenerator configGenerator;
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        const TDriveAPI& driveApi = *server->GetDriveAPI();
        TEnvironmentGenerator eGenerator(*server.Get());
        eGenerator.BuildEnvironment(TEnvironmentGenerator::DefaultTraits);

        auto userId = eGenerator.CreateUser("skulik-was-there", true, "onboarding");
        server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);
        configGenerator.CompleteInitialRegistrationChatSteps(server.Get(), userId, eGenerator, true);
        server->GetDriveAPI()->GetUsersData()->FetchInfo(userId);

        auto defaultAdminUser = eGenerator.CreateUser("support-operator", true, "active");
        {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            TUserRole userRole;
            userRole.SetRoleId("support_assignee").SetUserId(defaultAdminUser).SetActive(true);
            UNIT_ASSERT(driveApi.GetUsersData()->GetRoles().Link(userRole, defaultAdminUser, session) && session.Commit());
        }

        {
            auto response = configGenerator.GetUserAdminInfo(userId, defaultAdminUser);
            UNIT_ASSERT_C(response != NJson::JSON_NULL, TStringBuilder() << response);
        }

        UNIT_ASSERT(configGenerator.ForceRemoveUser(userId));

        {
            auto response = configGenerator.GetUserAdminInfo(userId, USER_ROOT_DEFAULT);
            UNIT_ASSERT_C(response != NJson::JSON_NULL, TStringBuilder() << response);
        }

        {
            auto response = configGenerator.GetUserAdminInfo(userId, defaultAdminUser);
            UNIT_ASSERT_VALUES_EQUAL(response, NJson::JSON_NULL);
        }
    }
}
