#include <drive/backend/ut/library/car_driver.h>
#include <drive/backend/ut/library/helper.h>
#include <drive/backend/ut/library/script.h>

#include <drive/backend/background/car_markers/config.h>
#include <drive/backend/base/config.h>
#include <drive/backend/base/server.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/alerts/tags.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/device_snapshot/image.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/standart.h>
#include <drive/backend/processors/service_app/processor.h>
#include <drive/backend/processors/user_app/processor.h>
#include <drive/backend/tags/tag.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/login.h>

#include <drive/telematics/client/library/handlers.h>
#include <drive/telematics/server/library/server.h>
#include <drive/telematics/server/ut/library/helper.h>

#include <kernel/daemon/config/config_constructor.h>
#include <kernel/daemon/config/daemon_config.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <rtline/library/storage/structured.h>

namespace NDrive::NTest {
    class TStartDeviceVerification: public ITestAction {
    private:
        using TBase = ITestAction;
    protected:
        void DoExecute(TRTContext& context) override {
            UNIT_ASSERT(context.HasDeviceId());
            NJson::TJsonValue resultJson;
            NNeh::THttpRequest request;
            request.SetUri("/user_app/verification/new_device/submit");
            request.AddHeader("Authorization", USER_ID_DEFAULT);
            request.AddHeader("DeviceId", context.GetDeviceIdUnsafe());
            NUtil::THttpReply result = context.GetConfigGenerator().GetSendReply(request);
            UNIT_ASSERT((result.Code() == 200) == GetExpectOK());
        }
    public:
        TStartDeviceVerification(const TInstant startInstant)
            : TBase(startInstant) {
        }
    };

    class TFinishDeviceVerification: public ITestAction {
    private:
        using TBase = ITestAction;
    protected:
        void DoExecute(TRTContext& context) override {
            UNIT_ASSERT(context.HasDeviceId());
            NJson::TJsonValue resultJson;
            bool found = false;
            for (auto&& i : TFakeUserDevicesManager::GetCodes()) {
                NNeh::THttpRequest request;
                request.SetUri("/user_app/verification/new_device/commit");
                request.SetCgiData("code=" + i.second);
                request.AddHeader("Authorization", USER_ID_DEFAULT);
                request.AddHeader("DeviceId", context.GetDeviceIdUnsafe());
                NUtil::THttpReply result = context.GetConfigGenerator().GetSendReply(request);
                if (result.Code() == 200) {
                    found = true;
                    break;
                }
            }
            UNIT_ASSERT_VALUES_EQUAL(found, GetExpectOK());
        }
    public:
        TFinishDeviceVerification(const TInstant startInstant)
            : TBase(startInstant) {

        }
    };
}

Y_UNIT_TEST_SUITE(Verification) {
    using namespace NDrive::NTest;
    Y_UNIT_TEST(Correct) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        TGeoCoord from(37.5848674, 55.7352435);
        TGeoCoord to(37.5675511, 55.7323499);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        script.Add<TSetDeviceId>().SetDeviceId(::ToString(RandomNumber<ui64>()));
        script.Add<TCreateCar>().SetPosition(from);
        script.Add<TSetSetting>().Set("handlers.default.new_device.policy", "fail")
            .Set("handlers.api/yandex/sessions/current.new_device.policy", "fail")
            .Set("handlers.user_app/verification/new_device/submit.new_device.policy", "fail")
            .Set("handlers.user_app/verification/new_device/commit.new_device.policy", "fail");

        const auto sessionChecker = [](const NJson::TJsonValue& userSessionReport) {
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["device_status"].GetString(), "Verified");
        };
        script.Add<TCheckCurrentSession>(sessionChecker).SetExpectOK(true);
        script.Add<TSetDeviceId>().SetDeviceId(::ToString(RandomNumber<ui64>()));
        script.Add<TCreateAndBookOffer>().SetUserDestination(to).SetOfferName("fixpoint_offer_constructor").SetUserPosition(from).SetExpectOK(false);
        script.Add<TCheckCurrentSession>().SetExpectOK(false);
        script.Add<TSetSetting>()
            .Set("handlers.api/yandex/sessions/current.new_device.policy", "verification")
            .Set("handlers.user_app/verification/new_device/submit.new_device.policy", "verification")
            .Set("handlers.user_app/verification/new_device/commit.new_device.policy", "verification");
        const auto sessionCheckerNew = [](const NJson::TJsonValue& userSessionReport) {
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["device_status"].GetString(), "New");
        };
        script.Add<TCheckCurrentSession>(sessionCheckerNew);
        script.Add<TStartDeviceVerification>();
        script.Add<TDropCache>();
        script.Add<TStartDeviceVerification>().SetExpectOK(false);
        script.Add<TDropCache>();
        script.Add<TStartDeviceVerification>(TDuration::Minutes(2));
        script.Add<TDropCache>();
        const auto sessionCheckerVerification = [](const NJson::TJsonValue& userSessionReport) {
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["device_status"].GetString(), "Verification");
        };
        script.Add<TCheckCurrentSession>(sessionCheckerVerification);
        script.Add<TFinishDeviceVerification>();
        const auto sessionCheckerVerified = [](const NJson::TJsonValue& userSessionReport) {
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["segment"]["session"]["specials"]["total_price"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["device_status"].GetString(), "Verified");
        };
        script.Add<TDropCache>();
        script.Add<TCheckCurrentSession>(sessionCheckerVerified);
        script.Add<TStartDeviceVerification>().SetExpectOK(false);
        UNIT_ASSERT(script.Execute());
    }

    Y_UNIT_TEST(EmailVerification) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetLogLevel(6);
        NDrive::NTest::TScript script(configGenerator);
        script.Add<TBuildEnv>();
        const auto undefinedChecker = [](const TMaybe<TString>& stringSetting) -> bool {
            return !stringSetting;
        };
        script.Add<TCheckUserSetting>("verified_mail", undefinedChecker);

        script.Add<TSetSetting>().Set("handlers.default.validate_email_by_passport", "true");

        const auto emptyMailsChecker = [](const NJson::TJsonValue& userSessionReport) {
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["user"]["validated_emails"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["user"]["validated_emails"].GetArray().size(), 0);
        };
        script.Add<TCheckCurrentSession>(emptyMailsChecker).SetExpectOK(true);

        const auto unverifiedChecker = [](const TMaybe<TString>& stringSetting) -> bool {
            return stringSetting && IsFalse(stringSetting.GetRef());
        };
        script.Add<TCheckUserSetting>("verified_mail", unverifiedChecker);

        script.Add<TSetSetting>().Set("handlers.default.fake_auth.default_emails", "zxqfd-test3@yandex.ru");
        const auto defaultMailsChecker = [](const NJson::TJsonValue& userSessionReport) {
            INFO_LOG << userSessionReport.GetStringRobust() << Endl;
            INFO_LOG << userSessionReport["user"]["validated_emails"] << Endl;
            UNIT_ASSERT_VALUES_EQUAL(userSessionReport["user"]["validated_emails"].GetArray().size(), 1);
        };
        script.Add<TCheckCurrentSession>(defaultMailsChecker).SetExpectOK(true);

        const auto verifiedChecker = [](const TMaybe<TString>& stringSetting) -> bool {
            return stringSetting && IsTrue(stringSetting.GetRef());
        };
        script.Add<TCheckUserSetting>("verified_mail", verifiedChecker);
        UNIT_ASSERT(script.Execute());
    }
}
