#include <drive/backend/ut/library/script.h>

#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/fix_point.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/offers/actions/ut/library/helper.h>
#include <drive/backend/processors/warning_screen/checker.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/json/builder.h>

namespace {
    const ui32 PACK_OFFER_MILLAGE = 123;
    const ui32 PACK_OFFER_PRICE = 321;
    const TString PACK_OFFER_NAME = "Test prepay check";
    const ui32 DEPOSIT_PRICE = 456;
    const TString STANDART_LANDING_TEXT = "Landing text";

    TString BuildPackOffer(NDrive::NTest::TRTContext& context, bool useDeposit,
                           const TString offerName = PACK_OFFER_NAME,
                           const TDuration extraDuration = TDuration::Zero(),
                           const ui32 extraMilage = 0) {
        THolder<TPackOffer> offer(new TPackOffer());
        offer->SetMileageLimit(PACK_OFFER_MILLAGE)
            .SetPackPrice(PACK_OFFER_PRICE * 100)
            .SetDuration(TDuration::Hours(1))
            .SetExtraDuration(extraDuration)
            .SetExtraMileageLimit(extraMilage)
            .SetName(offerName);
        offer->MutableRiding().SetPrice(700);
        offer->SetUseDeposit(useDeposit);
        offer->SetOverrunKm(521);
        offer->SetChargableAccounts({"card", "bonus"});
        offer->SetObjectId(OBJECT_ID_DEFAULT)
                .SetUserId(USER_ID_DEFAULT)
                .SetDeadline(Now() + TDuration::Minutes(5));
        offer->SetBehaviourConstructorId("pack_offer_constructor");
        offer->SetPriceConstructorId("pack_offer_constructor");
        TVector<IOfferReport::TPtr> offers;
        context.GetServer()->GetOffersStorage()->StoreOffers({new TPackOfferReport(offer.Release(), nullptr)}, &offers);
        UNIT_ASSERT(offers.size() == 1);
        return offers[0]->GetOffer()->GetOfferId();
    }

    TString BuildStandartOffer(NDrive::NTest::TRTContext& context, const TString& userId, bool useDeposit = true) {
        auto offer = BuildOfferPtr(200, 100, 102400);
        if (useDeposit) {
            offer->SetDepositAmount(DEPOSIT_PRICE * 100);
        }
        offer->SetChargableAccounts({"card", "bonus"});
        offer->SetObjectId(context.GetCar().Id)
                .SetUserId(userId)
                .SetDeadline(Now() + TDuration::Minutes(5));
        TVector<IOfferReport::TPtr> offers;
        context.GetServer()->GetOffersStorage()->StoreOffers({new TStandartOfferReport(offer, nullptr)}, &offers);
        UNIT_ASSERT(offers.size() == 1);
        return offers[0]->GetOffer()->GetOfferId();
    }

    TString BuildFixPointOffer(NDrive::NTest::TRTContext& context,
                               const TVector<TGeoCoord>& finishArea) {
        THolder<TFixPointOffer> offer(new TFixPointOffer());
        offer->SetMileageLimit(PACK_OFFER_MILLAGE)
                .SetPackPrice(PACK_OFFER_PRICE * 100)
                .SetDuration(TDuration::Hours(1))
                .SetName("Fix point warning screens");
        offer->SetChargableAccounts({"card", "bonus"});
        offer->SetObjectId(context.GetCar().Id)
                .SetUserId(USER_ID_DEFAULT)
                .SetDeadline(Now() + TDuration::Minutes(5));
        offer->SetFinishArea(finishArea);

        TVector<IOfferReport::TPtr> offers;
        context.GetServer()->GetOffersStorage()->StoreOffers({new TFixPointOfferReport(offer.Release(), nullptr)}, &offers);
        UNIT_ASSERT(offers.size() == 1);
        return offers[0]->GetOffer()->GetOfferId();
    }

    void PreparePrepayWarningTest(NDrive::NTest::TScript& script) {
        using namespace NDrive::NTest;
        script.Add<TBuildEnv>();
        script.Add<TSetSetting>().Set("warning_screens.events.before_booking.prepay_check.enabled",
                                      "true");
        script.Add<TSetSetting>().Set("warning_screens.checkers.prepay_check.landings.pack",
                                      "{\"text\": \"_OfferName_ _OfferPrice_ _OfferDistance_\"}");
        script.Add<TSetSetting>().Set("warning_screens.checkers.prepay_check.landings.standart",
                                      "{\"text\": \"" + STANDART_LANDING_TEXT + " _Deposit_\"}");
        script.Add<TSetScriptUser>(USER_ID_DEFAULT);
        script.Add<TCreateCar>();
    }
}  // namespace

namespace NDrive::NTest {
    Y_UNIT_TEST_SUITE(WarningScreensChecks) {
        Y_UNIT_TEST(PackOffer) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePrepayWarningTest(script);
            TString offerId;
            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildPackOffer(context, true);
            });
            script.Add<TWarningScreenChecker>(
                    "before_booking",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        const TString messageText = jsonMessage->GetString();
                        UNIT_ASSERT_C(messageText.Contains(PACK_OFFER_NAME), message);
                        UNIT_ASSERT_C(messageText.Contains(::ToString(PACK_OFFER_PRICE)), message);
                        UNIT_ASSERT_C(messageText.Contains(::ToString(PACK_OFFER_MILLAGE)),
                                      message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(PackOfferSpecial) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePrepayWarningTest(script);
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.prepay_check.landings.grouping_tag_pack",
                    "{\"text\": \"Special offer : _OfferName_ : _OfferDistance_ : _OfferPrice_ : _RerunPriceKm_ : _RerunPriceMin_ : _PackDurationHours_\" }");
            TString offerId;
            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildPackOffer(context, true);
            });
            script.Add<TWarningScreenChecker>(
                    "before_booking",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        const TString messageText = jsonMessage->GetString();
                        UNIT_ASSERT_C(messageText ==
                                              "Special offer : Test prepay check : 123 км : 321 : 5,21 ₽/км : 7 ₽/мин : 1\u00A0час",
                                      message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(PackOfferSwitch) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePrepayWarningTest(script);
            script.Add<TSetSetting>().Set("warning_screens.events.tariff_changing.prepay_check.enabled",
                                          "true");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.prepay_check.landings.pack_to_pack_switching",
                    "{\"text\": \"Switching : _OfferName_ : _OfferDistanceTotal_ : _OfferPrice_ : _RerunPriceKm_ : _RerunPriceMin_ : _PackDurationHoursTotal_\" }");

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                TString offerId = BuildPackOffer(context, true);
                UNIT_ASSERT(context.GetConfigGenerator().BookOffer(offerId, USER_ID_DEFAULT));
            });
            script.Add<TAccept>();
            script.Add<TRide>();

            TString newOfferId;
            script.Add<TCommonChecker>([&newOfferId](NDrive::NTest::TRTContext& context) {
                newOfferId = BuildPackOffer(context, true, "New pack offer", TDuration::Minutes(70), 100);
            });
            script.Add<TWarningScreenChecker>(
                    "tariff_changing",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        const TString messageText = jsonMessage->GetString();
                        UNIT_ASSERT_C(messageText ==
                                              "Switching : New pack offer : 223 км : 321 : 5,21 ₽/км : 7 ₽/мин : 2\u00A0часа",
                                      message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&newOfferId]() { return TStringBuilder() << "offer_id=" << newOfferId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(OfferWithoutDeposit) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePrepayWarningTest(script);
            TString offerId;
            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildPackOffer(context, false);
            });
            script.Add<TWarningScreenChecker>(
                    "before_booking",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        UNIT_ASSERT_C(response["items"].GetArray().size() == 0, message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(StandartOfferFirstRide) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePrepayWarningTest(script);
            TString newUserId;
            script.Add<TCommonChecker>([&newUserId](NDrive::NTest::TRTContext& context) {
                newUserId = context.GetEGenerator().CreateUser("user_with_first_riding");
                context.SetUserId(newUserId);
            });
            TString offerId;
            script.Add<TCommonChecker>([&offerId, &newUserId](NDrive::NTest::TRTContext& context) {
                offerId = BuildStandartOffer(context, newUserId);
            });
            script.Add<TWarningScreenChecker>(
                    "before_booking",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        const TString messageText = jsonMessage->GetString();
                        UNIT_ASSERT_C(messageText.Contains(STANDART_LANDING_TEXT), message);
                        UNIT_ASSERT_C(messageText.Contains(::ToString(DEPOSIT_PRICE)), message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(StandartOfferNotFirstRide) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePrepayWarningTest(script);
            TString offerId;
            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildStandartOffer(context, USER_ID_DEFAULT);
            });
            script.Add<TWarningScreenChecker>(
                    "before_booking",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        UNIT_ASSERT_C(response["items"].GetArray().size() == 0, message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(UserAccessDefault) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set("warning_screens.events.test_event.user_phone_check.enabled", "true");
            script.Add<TSetSetting>().Set("warning_screens.events.test_event.user_phone_check.enabled", "true");
            script.Add<TSetSetting>().Set("warning_screens.events.test_event.user_debt_check.enabled", "true");
            script.Add<TSetSetting>().Set("warning_screens.checkers.user_status_check.landings.main",
                                          "{\"text\": \"status\"}");
            script.Add<TSetSetting>().Set("warning_screens.checkers.user_phone_check.landings.main",
                                          "{\"text\": \"phone\"}");
            script.Add<TSetSetting>().Set("warning_screens.checkers.user_debt_check.landings.main",
                                          "{\"text\": \"debt\"}");

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);
            script.Add<TWarningScreenChecker>("test_event", [](const NJson::TJsonValue& response) {
                const TString message = response.GetStringRobust();
                UNIT_ASSERT_C(response["items"].GetArray().size() == 0, message);
                UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
            });

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);
        }

        Y_UNIT_TEST(UserPhoneCheck) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set("warning_screens.events.test_event.user_phone_check.enabled", "true");
            script.Add<TSetSetting>().Set("warning_screens.checkers.user_phone_check.landings.main",
                                          "{\"text\": \"phone check\"}");

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto userFetchResult = context.GetDriveAPI().GetUsersData()->FetchInfo(USER_ID_DEFAULT);
                UNIT_ASSERT_EQUAL(userFetchResult.size(), 1);
                auto userEntry = userFetchResult.begin()->second;
                userEntry.SetPhoneVerified(false);
                auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
                UNIT_ASSERT(
                        context.GetDriveAPI().GetUsersData()->UpdateUser(userEntry, USER_ROOT_DEFAULT, session) &&
                        session.Commit());

                userFetchResult = context.GetDriveAPI().GetUsersData()->FetchInfo(USER_ID_DEFAULT);
                userEntry = userFetchResult.begin()->second;
                UNIT_ASSERT_EQUAL(userEntry.IsPhoneVerified(), false);
            });
            script.Add<TWarningScreenChecker>("test_event", [](const NJson::TJsonValue& response) {
                const TString message = response.GetStringRobust();
                UNIT_ASSERT_C(response["items"].GetArray().size() == 1, message);
                const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                UNIT_ASSERT_EQUAL_C(jsonMessage->GetString(), "phone check", message);
                UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
            });

            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(UserStatusCheck) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set("warning_screens.events.test_event.user_status_check.enabled", "true");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.user_status_check.landings.main",
                    "{\"text\": \"status check\"}");

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);
            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto userFetchResult = context.GetDriveAPI().GetUsersData()->FetchInfo(USER_ID_DEFAULT);
                UNIT_ASSERT_EQUAL(userFetchResult.size(), 1);
                auto userEntry = userFetchResult.begin()->second;
                userEntry.SetStatus("blocked");
                auto session = context.GetDriveAPI().BuildTx<NSQL::Writable>();
                UNIT_ASSERT(
                        context.GetDriveAPI().GetUsersData()->UpdateUser(userEntry, USER_ROOT_DEFAULT, session) &&
                        session.Commit());

                userFetchResult = context.GetDriveAPI().GetUsersData()->FetchInfo(USER_ID_DEFAULT);
                userEntry = userFetchResult.begin()->second;
                UNIT_ASSERT_EQUAL(userEntry.GetStatus(), "blocked");
            });
            script.Add<TWarningScreenChecker>("test_event", [](const NJson::TJsonValue& response) {
                const TString message = response.GetStringRobust();

                UNIT_ASSERT_C(response["items"].GetArray().size() == 1, message);
                const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                UNIT_ASSERT_EQUAL_C(jsonMessage->GetString(), "status check", message);
                UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
            });

            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(CarModelCheck) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set(
                    "warning_screens.events.test_event.car_model_check.enabled", "true");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.car_model_check.landings.kia_rio",
                    "{\"text\": \"rio\"}");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.car_model_check.landings.missing",
                    "{\"text\": \"something else\"}");

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);

            TString offerId;
            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildStandartOffer(context, USER_ID_DEFAULT);
            });
            script.Add<TWarningScreenChecker>(
                    "test_event",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        UNIT_ASSERT_C(response["items"].GetArray().size() == 1, message);
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        UNIT_ASSERT_EQUAL_C(jsonMessage->GetString(), "rio", message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(SessionPaymentDeposit) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set(
                    "warning_screens.events.test_event.session_debt_check.enabled", "true");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.session_debt_check.landings.deposit_fails",
                    "{\"text\": \"Deposit _Deposit_\"}");

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);
            script.Add<TCreateCar>();

            TString offerId;
            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildStandartOffer(context, USER_ID_DEFAULT);
            });

            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                UNIT_ASSERT(context.GetConfigGenerator().BookOffer(offerId, USER_ID_DEFAULT));
            });

            script.Add<TCommonChecker>([](NDrive::NTest::TRTContext& context) {
                auto& billingManager = context.GetDriveAPI().GetBillingManager();
                context.GetEGenerator().GetBillingMock().SetReply(
                        "{\"status\": \"success\", \"bound_payment_methods\" : []}");
                billingManager.WaitBillingCycle();
            });

            script.Add<TWarningScreenChecker>(
                    "test_event",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        UNIT_ASSERT_C(response["items"].GetArray().size() == 1, message);
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        UNIT_ASSERT_EQUAL_C(jsonMessage->GetString(),
                                            TStringBuilder() << "Deposit " << DEPOSIT_PRICE,
                                            message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(SessionPaymentDebt) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set(
                    "warning_screens.events.test_event.session_debt_check.enabled", "true");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.session_debt_check.landings.debt",
                    "{\"text\": \"Debt _Debt_\"}");

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);
            script.Add<TCreateCar>();

            TString offerId;
            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildStandartOffer(context, USER_ID_DEFAULT, false);
            });

            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                UNIT_ASSERT(context.GetConfigGenerator().BookOffer(offerId, USER_ID_DEFAULT));
            });

            const ui32 debt = 10000000u;
            script.Add<NDrive::NTest::TCommonChecker>(
                    [&offerId, debt](NDrive::NTest::TRTContext& context) {
                        auto& billingManager = context.GetDriveAPI().GetBillingManager();

                        auto session = billingManager.BuildSession(false);
                        UNIT_ASSERT(billingManager.SetBillingInfo({{offerId, debt}}, session));
                        UNIT_ASSERT(session.Commit());

                        context.GetEGenerator().GetBillingMock().SetReply(
                                "{\"status\": \"success\", \"bound_payment_methods\" : []}");
                        billingManager.WaitBillingCycle();
                    });

            script.Add<TWarningScreenChecker>(
                    "test_event",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        UNIT_ASSERT_C(response["items"].GetArray().size() == 1, message);
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        UNIT_ASSERT_EQUAL_C(jsonMessage->GetString(),
                                            TStringBuilder() << "Debt " << debt / 100, message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(FixPointFinishArea) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set(
                    "warning_screens.events.test_event.fix_point_finish_area_check.enabled",
                    "true");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.fix_point_finish_area_check.landings.main",
                    "{\"text\": \"Warning text\"}");

            script.Add<TSetScriptUser>(USER_ID_DEFAULT);
            script.Add<TCreateCar>().SetPosition(TGeoCoord(37.50, 55.391));

            TString offerId;
            TGeoCoord c1(37.49, 55.39);
            TGeoCoord c2(37.49, 55.41);
            TGeoCoord c3(37.51, 55.39);
            TGeoCoord c4(37.49, 55.39);
            const TVector<TGeoCoord> finishArea = {c1, c2, c3, c4};

            script.Add<TCommonChecker>([finishArea, &offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildFixPointOffer(context, finishArea);
                UNIT_ASSERT(context.GetConfigGenerator().BookOffer(offerId, USER_ID_DEFAULT));
            });
            script.Add<TAccept>();
            script.Add<TWarningScreenChecker>("test_event", [](const NJson::TJsonValue& response) {
                const TString message = response.GetStringRobust();
                UNIT_ASSERT_C(response["items"].GetArray().size() == 0, message);
                UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                return true;
            });

            script.Add<TRide>().SetCarPosition(TGeoCoord(37.52, 55.40));
            script.Add<TWarningScreenChecker>("test_event", [](const NJson::TJsonValue& response) {
                const TString message = response.GetStringRobust();
                UNIT_ASSERT_C(response["items"].GetArray().size() == 1, message);
                const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                UNIT_ASSERT_C(jsonMessage, message);
                UNIT_ASSERT_EQUAL_C(jsonMessage->GetString(), "Warning text", message);
                UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                return true;
            });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(SimpleCheck) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set("warning_screens.events.test_event.simple_check.enabled",
                                          "true");
            script.Add<TSetSetting>().Set("warning_screens.events.test_event.simple_check.args",
                                          "landing_id=test_landing_id");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.simple_check.landings.test_landing_id",
                    "{\"text\": \"test\"}");
            script.Add<TWarningScreenChecker>(
                    "test_event",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        UNIT_ASSERT_C(response["items"].GetArray().size() == 1, message);
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        UNIT_ASSERT_EQUAL_C(jsonMessage->GetString(), "test", message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(AreasCheck) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);

            TScript script(configGenerator);
            script.Add<TBuildEnv>();
            script.Add<TSetSetting>().Set(
                    "warning_screens.events.test_event.allowed_area_check.enabled", "true");
            script.Add<TSetSetting>().Set(
                    "warning_screens.checkers.allowed_area_check.landings.main",
                    "{\"text\": \"test\"}");
            // "Novaya Zemlya" island.
            script.Add<TCreateCar>().SetPosition(TGeoCoord(62.884511, 75.042543));

            TString offerId;
            script.Add<TCommonChecker>([&offerId](NDrive::NTest::TRTContext& context) {
                offerId = BuildStandartOffer(context, USER_ID_DEFAULT);
            });
            script.Add<TWarningScreenChecker>(
                    "test_event",
                    [](const NJson::TJsonValue& response) {
                        const TString message = response.GetStringRobust();
                        UNIT_ASSERT_C(response["items"].GetArray().size() == 1, message);
                        const auto* jsonMessage = response.GetValueByPath("items.[0].text");
                        UNIT_ASSERT_C(jsonMessage, message);
                        UNIT_ASSERT_EQUAL_C(jsonMessage->GetString(), "test", message);
                        UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                        return true;
                    },
                    [&offerId]() { return TStringBuilder() << "offer_id=" << offerId; });
            UNIT_ASSERT(script.Execute());
        }
    }
}  // namespace NDrive::NTest
