#include <drive/backend/ut/library/script.h>

#include <drive/backend/offers/manager.h>
#include <drive/backend/offers/actions/pack.h>
#include <drive/backend/processors/warning_screen/checker.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/json/builder.h>

namespace {
    const TDuration RETURNING_DURATION = TDuration::Minutes(10);
    const TDuration SKIP_MESSAGE_DURATION = TDuration::Minutes(5);
    const TDuration OFFER_DURATION = TDuration::Minutes(60);

    void AddAndBookOffer(NDrive::NTest::TRTContext& context) {
        THolder<TPackOffer> offer(new TPackOffer);
        offer->SetMileageLimit(70)
                .SetPackPrice(45000)
                .SetDuration(OFFER_DURATION)
                .MutableRiding().SetPrice(700);
        offer->SetChargableAccounts({"card", "bonus"});
        offer->SetReturningDuration(RETURNING_DURATION);
        offer->SetObjectId(OBJECT_ID_DEFAULT)
                .SetUserId(USER_ID_DEFAULT)
                .SetDeadline(Now() + TDuration::Minutes(5));
        const TString offerId = offer->GetOfferId();
        UNIT_ASSERT(context.GetServer()->GetOffersStorage()->StoreOffers({new TPackOfferReport(offer.Release(), nullptr)}));
        UNIT_ASSERT(context.GetConfigGenerator().BookOffer(offerId, USER_ID_DEFAULT));
    }

    void PreparePackOfferWarningTest(NDrive::NTest::TScript& script) {
        using namespace NDrive::NTest;
        script.Add<TBuildEnv>();
        script.Add<TSetScriptUser>(USER_ID_DEFAULT);

        script.Add<TSetSetting>().Set(
                "warning_screens.events.finish_rental.pack_offer_usage_check.enabled", "true");
        script.Add<TSetSetting>().Set(
                "warning_screens.checkers.pack_offer_usage_check.min_remaining_time", "20m");
        script.Add<TSetSetting>().Set(
                "warning_screens.checkers.pack_offer_usage_check.landings.high",
                "{\"text\": \"high landing\"}");
        script.Add<TSetSetting>().Set("warning_screens.checkers.pack_offer_usage_check.landings.low",
                                      "{\"text\": \"low landing\"}");

        script.Add<TCreateCar>();
        script.Add<TCommonChecker>(AddAndBookOffer);
        script.Add<TAccept>(TDuration::Zero());
    }
}  // namespace

namespace NDrive::NTest {
    Y_UNIT_TEST_SUITE(WarningScreenPackOffer) {
        Y_UNIT_TEST(LowLanding) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePackOfferWarningTest(script);
            script.Add<TRide>(RETURNING_DURATION/2);
            script.Add<TWarningScreenChecker>("finish_rental", [](const NJson::TJsonValue& response) {
                const TString message = response.GetStringRobust();
                UNIT_ASSERT_C(response["items"][0]["text"] == "low landing", message);
                UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                return true;
            });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(HighLanding) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePackOfferWarningTest(script);
            script.Add<TRide>(RETURNING_DURATION + TDuration::Minutes(1));
            script.Add<TWarningScreenChecker>("finish_rental", [](const NJson::TJsonValue& response) {
                const TString message = response.GetStringRobust();
                UNIT_ASSERT_C(response["items"][0]["text"] == "high landing", message);
                UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                return true;
            });
            UNIT_ASSERT(script.Execute());
        }

        Y_UNIT_TEST(EndOfRoute) {
            NDrive::TServerConfigGenerator configGenerator;
            configGenerator.SetNeedBackground(0);
            TScript script(configGenerator);
            PreparePackOfferWarningTest(script);
            script.Add<TRide>(OFFER_DURATION-SKIP_MESSAGE_DURATION/2);
            script.Add<TWarningScreenChecker>("finish_rental", [](const NJson::TJsonValue& response) {
                const TString message = response.GetStringRobust();
                UNIT_ASSERT_C(response["items"].GetArray().size() == 0, message);
                UNIT_ASSERT_C(response["errors"].GetMap().size() == 0, message);
                return true;
            });
            UNIT_ASSERT(script.Execute());
        }
    }
}  // namespace NDrive::NTest
