#include <drive/backend/ut/library/helper.h>

#include <drive/backend/data/device_tags.h>
#include <drive/backend/tags/tags.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/users/yang.h>
#include <drive/backend/yang/client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <rtline/library/storage/structured.h>

#include <util/system/env.h>

#define TASKS_DEFAULT_PATH "//tmp/yang-test-requests"
#define RESULTS_DEFAULT_PATH "//tmp/yang-test-results"

Y_UNIT_TEST_SUITE(YangClient) {

    Y_UNIT_TEST(CreatePool) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetIsYtEnabled(true);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TYangYTClient<TYangDocumentVerificationAssignment> client({"hahn"}, true);

        NYT::JoblessInitialize();
        auto ytClient = NYT::NTesting::CreateTestClient();
        ytClient->Create(TASKS_DEFAULT_PATH, NYT::NT_MAP);

        {
            TYangPool<TYangDocumentVerificationAssignment> pool;
            TYangDocumentVerificationAssignment assignment("1");
            pool.AddAssignment(assignment);
            client.CreatePool(TASKS_DEFAULT_PATH, pool);

            NYT::TNode::TListType list = ytClient->List(TASKS_DEFAULT_PATH);
            size_t totalChildren = 0;
            for (auto element : list) {
                ++totalChildren;
            }

            UNIT_ASSERT_VALUES_EQUAL(totalChildren, 1);
        }
    }

    Y_UNIT_TEST(FetchResults) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetIsYtEnabled(true);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TYangYTClient<TYangDocumentVerificationAssignment> client({"hahn"}, true);

        NYT::JoblessInitialize();
        auto ytClient = NYT::NTesting::CreateTestClient();
        ytClient->Create(RESULTS_DEFAULT_PATH, NYT::NT_MAP);

        // Create pool result
        {
            auto writer = ytClient->CreateTableWriter<NYT::TNode>(NYT::TRichYPath(TString(RESULTS_DEFAULT_PATH) + TString("/test_pool")));

            NYT::TNode recordNode1;
            recordNode1["assignmentIds"] = NYT::TNode::CreateList();
            recordNode1["assignmentIds"].Add(NYT::TNode("00003f6103--5bd10571143a7000e7702c33"));
            recordNode1["comments"] = NYT::TNode::CreateList();
            recordNode1["comments"].Add(NYT::TNode("not a passport"));
            recordNode1["secretId"] = "497186e5-85c0-43f4-a205-8f195a1a731d";
            recordNode1["result"]["is_fraud"] = "MAYBE_FRAUD";
            recordNode1["result"]["license_back_status"] = "OK";
            recordNode1["result"]["license_front_status"] = "OK";
            recordNode1["result"]["passport_biographical_status"] = "NEED_INFO";
            recordNode1["result"]["passport_registration_status"] = "NEED_INFO";
            recordNode1["result"]["passport_selfie_status"] = "NEED_INFO";
            recordNode1["workersIds"] = NYT::TNode::CreateList();
            recordNode1["workersIds"].Add(NYT::TNode("3fc84d22cff1b28bb4bd7e126f92fbda"));
            recordNode1["history"] = NYT::TNode::CreateList();
            recordNode1["history"].Add(NYT::TNode("abc"));

            NYT::TNode recordNode2;
            recordNode2["assignmentIds"] = NYT::TNode::CreateList();
            recordNode2["assignmentIds"].Add(NYT::TNode("00003f612d--5bd10760469c7a01189a1e61"));
            recordNode2["comments"] = NYT::TNode::CreateEntity();
            recordNode2["result"]["is_fraud"] = "MAYBE_FRAUD";
            recordNode2["result"]["license_back_status"] = "OK";
            recordNode2["result"]["license_front_status"] = "OK";
            recordNode2["result"]["passport_biographical_status"] = "OK";
            recordNode2["result"]["passport_registration_status"] = "OK";
            recordNode2["result"]["passport_selfie_status"] = "OK";
            recordNode2["secretId"] = "163d42cb-f5f8-437f-bb3b-e74397904b24";
            recordNode2["workersIds"] = NYT::TNode::CreateList();
            recordNode2["workersIds"].Add(NYT::TNode("3fc84d22cff1b28bb4bd7e126f92fbda"));

            writer->AddRow(recordNode1);
            writer->AddRow(recordNode2);
            writer->Finish();
        }

        TYangPool<TYangDocumentVerificationAssignment> pool;

        // Fetch results
        {
            auto results = client.FetchVerificationResults(RESULTS_DEFAULT_PATH);
            UNIT_ASSERT_VALUES_EQUAL(results.size(), 1);
            pool = results[0];
            auto assignments = results[0].GetAssignments();
            UNIT_ASSERT_VALUES_EQUAL(assignments.size(), 2);
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetId(), "497186e5-85c0-43f4-a205-8f195a1a731d");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetIsFraud(), TYangDocumentVerificationAssignment::EFraudStatus::MaybeFraud);
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetLicenseBackStatus(), "OK");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetLicenseFrontStatus(), "OK");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetPassportBiographicalStatus(), "NEED_INFO");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetPassportSelfieStatus(), "NEED_INFO");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetPassportRegistrationStatus(), "NEED_INFO");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetWorkers().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetWorkers()[0], "3fc84d22cff1b28bb4bd7e126f92fbda");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetAssignmentIds().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetAssignmentIds()[0], "00003f6103--5bd10571143a7000e7702c33");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetComments().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetComments()[0], "not a passport");
            UNIT_ASSERT_VALUES_EQUAL(assignments[0].GetHistory(), "[\"abc\"]");

            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetId(), "163d42cb-f5f8-437f-bb3b-e74397904b24");
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetIsFraud(), TYangDocumentVerificationAssignment::EFraudStatus::MaybeFraud);
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetLicenseBackStatus(), "OK");
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetLicenseFrontStatus(), "OK");
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetPassportBiographicalStatus(), "OK");
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetPassportSelfieStatus(), "OK");
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetPassportRegistrationStatus(), "OK");
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetWorkers().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetWorkers()[0], "3fc84d22cff1b28bb4bd7e126f92fbda");
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetAssignmentIds().size(), 1);
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetAssignmentIds()[0], "00003f612d--5bd10760469c7a01189a1e61");
            UNIT_ASSERT_VALUES_EQUAL(assignments[1].GetComments().size(), 0);
        }

        // Mark pool processed
        {
            client.MarkPoolProcessed(TString(RESULTS_DEFAULT_PATH) + "/" + pool.GetId());

            NYT::TNode::TListType list = ytClient->List(RESULTS_DEFAULT_PATH);
            size_t totalChildren = 0;
            for (auto element : list) {
                ++totalChildren;
            }
            UNIT_ASSERT_VALUES_EQUAL(totalChildren, 0);
        }
    }

    Y_UNIT_TEST(UnableToDeletePool) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetIsYtEnabled(true);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TYangYTClient<TYangDocumentVerificationAssignment> client({"hahn"}, true);

        NYT::JoblessInitialize();
        auto ytClient = NYT::NTesting::CreateTestClient();
        {
            TYangPool<TYangDocumentVerificationAssignment> pool;
            TYangDocumentVerificationAssignment assignment("1");

            client.MarkPoolProcessed(TString(RESULTS_DEFAULT_PATH) + "/" + pool.GetId());
        }
    }

    Y_UNIT_TEST(UnableToFetchPool) {
        NDrive::TServerConfigGenerator configGenerator;
        configGenerator.SetIsYtEnabled(true);
        TServerConfigConstructorParams params(configGenerator.GetString().data());
        NDrive::TServerConfig config(params);
        NDrive::TServerGuard server(config);
        TEnvironmentGenerator eGenerator(*server);
        eGenerator.BuildEnvironment((ui32)EEnvironmentFeatures::InfoAccess);

        TYangYTClient<TYangDocumentVerificationAssignment> client({"hahn"}, true);

        NYT::JoblessInitialize();
        auto ytClient = NYT::NTesting::CreateTestClient();
        {
            TYangPool<TYangDocumentVerificationAssignment> pool;
            TYangDocumentVerificationAssignment assignment("1");

            auto results = client.FetchVerificationResults(RESULTS_DEFAULT_PATH);
            UNIT_ASSERT_VALUES_EQUAL(results.size(), 0);
        }
    }
}
