#pragma once

#include "task.h"

#include <drive/backend/logging/events.h>

#include <drive/library/cpp/yt/common/init.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/tests/yt_unittest_lib/yt_unittest_lib.h>

#include <rtline/library/json/cast.h>
#include <rtline/library/json/exception.h>
#include <rtline/util/types/accessor.h>

template <class TAssignment>
class TYangYTClient {
private:
    TMap<TString, TIntrusivePtr<NYT::IClient>> YtClients;

public:
    TYangYTClient(const TVector<TString>& ytClusters, bool isTests = false) {
        if (isTests) {
            NYT::JoblessInitializeOnce();
            YtClients.emplace("tests", NYT::NTesting::CreateTestClient());
        } else {
            NYT::JoblessInitializeOnce();
            for (auto&& clusterName : ytClusters) {
                YtClients.emplace(clusterName, NYT::CreateClient(clusterName, NYT::TCreateClientOptions()));
            }
        }
    }

    TString BuildPoolPath(const TString& poolDirectoryPath, const TString& poolId) const {
        TString tablePath = poolDirectoryPath;
        if (!tablePath.EndsWith("/")) {
            tablePath += "/";
        }
        tablePath += poolId;
        return tablePath;
    }

    bool CreatePool(const TString& poolsDirectoryPath, const TYangPool<TAssignment>& pool) const {
        TString tablePath = BuildPoolPath(poolsDirectoryPath, pool.GetId());
        bool isSaved = (YtClients.size() == 0);  // this is the case only for tests
        for (auto&& [clusterName, client] : YtClients) {
            try {
                pool.SaveToYt(client, tablePath);
                NDrive::TEventLog::Log("YangCreatePool", NJson::TMapBuilder
                    ("id", pool.GetId())
                    ("assignments", NJson::ToJson(pool.GetAssignments()))
                    ("cluster", clusterName)
                    ("table", tablePath)
                );
            } catch (...) {
                NDrive::TEventLog::Log("YangCreatePoolError", NJson::TMapBuilder
                    ("id", pool.GetId())
                    ("assignments", NJson::ToJson(pool.GetAssignments()))
                    ("cluster", clusterName)
                    ("table", tablePath)
                    ("exception", CurrentExceptionInfo())
                );
                continue;
            }
            isSaved = true;
        }
        return isSaved;
    }

    TMap<TString, TSet<TString>> FetchAssignmentIds(const TString& rootPath) const {
        TMap<TString, TSet<TString>> idsByPools;
        for (auto&& [clusterName, client] : YtClients) {
            NYT::TNode::TListType list;
            try {
                list = client->List(rootPath);
                for (auto&& node : list) {
                    if (node.HasValue() && node.IsString()) {
                        TString path = BuildPoolPath(rootPath, node.AsString());
                        auto reader = client->CreateTableReader<NYT::TNode>(NYT::TRichYPath(path));
                        for (; reader->IsValid(); reader->Next()) {
                            const NYT::TNode& row = reader->GetRow();
                            if (row.HasKey("secret_id") && row["secret_id"].IsString()) {
                                idsByPools[node.AsString()].insert(row["secret_id"].AsString());
                            }
                        }
                    }
                }
            } catch (const yexception& e) {
                ERROR_LOG << "unable to fetch assignment ids" << Endl;
                continue;
            }
        }
        return idsByPools;
    }

    TVector<TYangPool<TAssignment>> FetchVerificationResults(const TString& rootPath) const {
        TSet<TYangPool<TAssignment>> pools;  // use set ordered by pool name in order to deduplicate results from different clusters
        for (auto&& [clusterName, client] : YtClients) {
            NYT::TNode::TListType list;
            try {
                list = client->List(rootPath);
            } catch (const yexception& e) {
                ERROR_LOG << "unable to list new pools skipping current proxy for now" << Endl;
                continue;
            }
            for (auto&& i : list) {
                if (!i.IsMap()) {
                    TString path = rootPath + "/" + i.AsString();

                    TYangPool<TAssignment> pool;
                    try {
                        pool.ReadFromYt(client, path);
                        pools.insert(std::move(pool));
                    } catch (const yexception& e) {
                        ERROR_LOG << "unable to catch pool, skipping for now: " << path << Endl;
                    }
                }
            }
        }
        return TVector<TYangPool<TAssignment>>(pools.begin(), pools.end());
    }

    void MarkPoolProcessed(const TString& poolPath) const {
        NYT::TRemoveOptions options;
        for (auto&& [clusterName, client] : YtClients) {
            try {
                client->Remove(NYT::TYPath(poolPath), options);
                NDrive::TEventLog::Log("YangMarkPoolProcessed", NJson::TMapBuilder
                    ("cluster", clusterName)
                    ("table", poolPath)
                );
            } catch (...) {
                NDrive::TEventLog::Log("YangMarkPoolProcessedError", NJson::TMapBuilder
                    ("cluster", clusterName)
                    ("table", poolPath)
                    ("exception", CurrentExceptionInfo())
                );
            }
        }
    }

    bool MovePools(const TSet<TString> poolIds, const TString& sourceDir, const TString destinationDir) const {
        NYT::TMoveOptions options;
        for (auto&& [clusterName, client] : YtClients) {
            try {
                for (auto&& poolId : poolIds) {
                    TString sourceTable = BuildPoolPath(sourceDir, poolId);
                    TString destinationTable = BuildPoolPath(destinationDir, poolId);
                    client->Move(NYT::TYPath(sourceTable), NYT::TYPath(destinationTable), options);
                }
            } catch (...) {
                NDrive::TEventLog::Log("YangMovePoolError", NJson::TMapBuilder
                    ("cluster", clusterName)
                    ("source", sourceDir)
                    ("destination", destinationDir)
                    ("exception", CurrentExceptionInfo())
                );
                return false;
            }
        }
        return true;
    }
};
