#pragma once

#include <rtline/library/storage/structured.h>

#include <rtline/util/types/accessor.h>

#include <mapreduce/yt/interface/client.h>

#include <util/string/vector.h>

class IYangAssignment {
public:
    virtual void SerializeToYtNode(NYT::TNode& row) const = 0;
    virtual bool DeserializeFromYtNode(const NYT::TNode& row) = 0;
    virtual ~IYangAssignment() {
    }
};

template<class TAssignment>
class TYangPool {
    R_READONLY(TString, Id, TInstant::Now().ToString());
    R_READONLY(bool, IsBroken, false);

private:
    TVector<TAssignment> Assignments;

public:

    bool operator < (const TYangPool& other) const {
        return Id < other.Id;
    }

    void SaveToYt(TIntrusivePtr<NYT::IClient> client, const TString& path) const {
        auto tx = client->StartTransaction();
        auto writer = tx->CreateTableWriter<NYT::TNode>(NYT::TRichYPath(path));
        for (auto&& assignment : Assignments) {
            NYT::TNode recordNode;
            assignment.SerializeToYtNode(recordNode);
            writer->AddRow(recordNode);
        }
        writer->Finish();
        tx->Commit();
    }

    void ReadFromYt(TIntrusivePtr<NYT::IClient> client, const TString& path) {
        {
            auto pathElements = SplitString(path, "/");
            Id = pathElements.back();
        }

        Assignments.clear();
        auto reader = client->CreateTableReader<NYT::TNode>(NYT::TRichYPath(path));
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& row = reader->GetRow();
            TAssignment assignment;
            if (assignment.DeserializeFromYtNode(row)) {
                Assignments.push_back(assignment);
            } else {
                IsBroken = true;
            }
        }
    }

    void AddAssignment(const TAssignment& assignment) {
        Assignments.push_back(assignment);
    }

    TVector<TAssignment> GetAssignments () const {
        return Assignments;
    }

    size_t Size () const {
        return Assignments.size();
    }
};
