#pragma once

#include <stdint.h>
#include <drive/contrib/cpp/glonassd/de.h>
#include <drive/contrib/cpp/glonassd/egts.h>

namespace telemetry {

class EGTSSubrecord {
 public:
  EGTSSubrecord(const EGTS_SUBRECORD_HEADER* header, const uint8_t* data)
      : header_(header), data_(data) {}

  const EGTS_SUBRECORD_HEADER* GetHeader() const { return header_; }

  const uint8_t* GetData() const { return data_; }

 private:
  const EGTS_SUBRECORD_HEADER* header_;
  const uint8_t* data_;
};

class EGTSSubrecordIter {
 public:
  EGTSSubrecordIter(const uint8_t* subrecords, const uint8_t* subrecords_end)
      : subrecords_(subrecords), subrecords_end_(subrecords_end) {}

  bool HasNext() const { return subrecords_ < subrecords_end_; }

  EGTSSubrecord Next() {
    const EGTS_SUBRECORD_HEADER* hdr =
        reinterpret_cast<const EGTS_SUBRECORD_HEADER*>(subrecords_);

    EGTSSubrecord r(hdr, subrecords_ + sizeof(EGTS_SUBRECORD_HEADER));
    subrecords_ += sizeof(EGTS_SUBRECORD_HEADER) + hdr->SRL;
    return r;
  }

 private:
  const uint8_t* subrecords_;
  const uint8_t* subrecords_end_;
};

class EGTSRecord {
 public:
  EGTSRecord(const EGTS_RECORD_HEADER* header, const uint8_t* subrecords)
      : header_(header), subrecords_(subrecords) {}

  EGTSSubrecordIter SubrecordsIter() const {
    return EGTSSubrecordIter(subrecords_, subrecords_ + header_->RL);
  }

  const EGTS_RECORD_HEADER* GetHeader() const { return header_; }

 private:
  const EGTS_RECORD_HEADER* header_;
  const uint8_t* subrecords_;
};

class EGTSRecordIter {
 public:
  EGTSRecordIter(const uint8_t* records, const uint8_t* records_end)
      : records_(records), records_end_(records_end) {}

  bool HasNext() const { return records_ < records_end_; }

  EGTSRecord Next() {
    const EGTS_RECORD_HEADER* hdr =
        reinterpret_cast<const EGTS_RECORD_HEADER*>(records_);
    size_t hdr_size = kRecordHeaderSize;

    hdr_size += (hdr->RFL & B0) ? sizeof(uint32_t) : 0;
    hdr_size += (hdr->RFL & B1) ? sizeof(uint32_t) : 0;
    hdr_size += (hdr->RFL & B2) ? sizeof(uint32_t) : 0;

    EGTSRecord record(hdr, records_ + hdr_size);
    records_ += hdr_size + hdr->RL;
    return record;
  }

 private:
  static const size_t kRecordHeaderSize = 7;

 private:
  const uint8_t* records_;
  const uint8_t* records_end_;
};

class EGTSResponse {
 public:
  EGTSResponse(const uint8_t* response) : packet_(response) {
    packet_header_ = reinterpret_cast<const EGTS_PACKET_HEADER*>(packet_);

    records_ = response + sizeof(EGTS_PACKET_HEADER);
    records_size_ = packet_header_->FDL;

    if (packet_header_->PT == 0) {
      pt_response_header_ =
          reinterpret_cast<const EGTS_PT_RESPONSE_HEADER*>(records_);
      records_ += sizeof(EGTS_PT_RESPONSE_HEADER);
      records_size_ -= sizeof(EGTS_PT_RESPONSE_HEADER);
    } else {
      pt_response_header_ = nullptr;
    }
  }

  const EGTS_PT_RESPONSE_HEADER* GetPtResponseHeader() const {
    return pt_response_header_;
  }

  const EGTS_PACKET_HEADER* GetPacketHeader() const { return packet_header_; }

  EGTSRecordIter GetRecordsIter() const {
    return EGTSRecordIter(records_, records_ + records_size_);
  }

 private:
  const uint8_t* packet_;
  const EGTS_PACKET_HEADER* packet_header_;
  const EGTS_PT_RESPONSE_HEADER* pt_response_header_;
  const uint8_t* records_;
  size_t records_size_;
};

}  // namespace telemetry
