#pragma once

#include <stdexcept>
#include <string>
#include <drive/contrib/cpp/glonassd/de.h>
#include <drive/contrib/cpp/glonassd/egts.h>

namespace telemetry {

class Subrecord {
 public:
  virtual int Serialize(char* packet, int position,
                        EGTS_RECORD_HEADER* record_header) const = 0;
  virtual size_t GetSize() const = 0;
  virtual ~Subrecord() {}

 protected:
  static const size_t kSubrecordHeaderSize = sizeof(EGTS_SUBRECORD_HEADER);
};

class Record {
 public:
  Record(uint8_t sender_service_type, uint8_t receiver_service_type)
      : has_oid_(true),
        oid_(0),
        sender_service_type_(sender_service_type),
        receiver_service_type_(receiver_service_type) {}

  Record(uint32_t oid, uint8_t sender_service_type,
         uint8_t receiver_service_type)
      : has_oid_(true),
        oid_(oid),
        sender_service_type_(sender_service_type),
        receiver_service_type_(receiver_service_type) {}

  void AddSubrecord(const std::shared_ptr<Subrecord>& subrecord) {
    subrecords_.push_back(subrecord);
  }

  size_t GetSize() const {
    size_t subrecords_size = 0;

    for (const auto& subrecord : subrecords_) {
      subrecords_size += subrecord->GetSize();
    }

    subrecords_size += has_oid_ ? kOidSize : 0;
    subrecords_size += kRecordHeaderSize;

    return subrecords_size;
  }

  int Serialize(char* packet, int position) const {
    EGTS_RECORD_HEADER* record_header =
        reinterpret_cast<EGTS_RECORD_HEADER*>(packet + position);

    if (has_oid_) {
      position =
          packet_add_record_header(packet, position, sender_service_type_,
                                   receiver_service_type_);
    } else {
      position = packet_add_record_header(
          packet, position, sender_service_type_, receiver_service_type_);
    }

    for (const auto& subrecord : subrecords_) {
      position = subrecord->Serialize(packet, position, record_header);
    }

    return position;
  }

 private:
  static const size_t kOidSize = 4;
  static const size_t kRecordHeaderSize = 7;

 private:
  bool has_oid_;
  uint32_t oid_;
  uint8_t sender_service_type_;
  uint8_t receiver_service_type_;
  std::vector<std::shared_ptr<Subrecord>> subrecords_;
};

class PacketBuilder {
 public:
  PacketBuilder(uint8_t packet_type, uint8_t priority)
      : packet_type_(packet_type), priority_(priority) {}

  void AddRecord(const Record& record) { records_.push_back(record); }

  std::vector<uint8_t> Build() {
    std::vector<uint8_t> packet(GetPacketSize());

    char* buffer = reinterpret_cast<char*>(packet.data());
    int top = packet_create(buffer, packet_type_);
    EGTS_PACKET_HEADER* packet_header =
        reinterpret_cast<EGTS_PACKET_HEADER*>(buffer);

    packet_header->PRF = priority_;

    for (const auto& record : records_) {
      top = record.Serialize(buffer, top);
    }

    top += packet_finalize(buffer, top);
    assert(packet.size() == top);
    return packet;
  }

  size_t GetPacketSize() const {
    size_t r = kPacketHeaderSize;

    for (const auto& record : records_) {
      r += record.GetSize();
    }

    r += kCheckSumFieldSize;
    return r;
  }

 private:
  static const size_t kPacketHeaderSize = 11;
  static const size_t kCheckSumFieldSize = 2;

 private:
  uint8_t packet_type_;
  uint8_t priority_;
  std::vector<Record> records_;
};

class IdentitySubrecord : public Subrecord {
 public:
  IdentitySubrecord(const std::string& imei) : imei_(imei) {
    if (imei_.size() != EGTS_IMEI_LEN) {
      throw std::runtime_error("incorrect IMEI length");
    }
  }

  virtual int Serialize(char* packet, int position,
                        EGTS_RECORD_HEADER* record_header) const override {
    EGTS_SUBRECORD_HEADER* subrecord_header =
        reinterpret_cast<EGTS_SUBRECORD_HEADER*>(packet + position);
    position = packet_add_subrecord_header(packet, position, record_header,
                                           EGTS_SR_TERM_IDENTITY);
    position = packet_add_subrecord_EGTS_SR_TERM_IDENTITY(
        packet, position, record_header, subrecord_header, const_cast<char*>(imei_.c_str()));
    return position;
  }

  virtual size_t GetSize() const override {
    return kSubrecordHeaderSize + kTermIdentitySubrecordSize;
  }

 private:
  static const size_t kTermIdentitySubrecordSize = sizeof(EGTS_SR_TERM_IDENTITY_RECORD) + EGTS_IMEI_LEN + sizeof(uint16_t);

 private:
  std::string imei_;
};

class PosDataSubrecord : public Subrecord {
 public:
  PosDataSubrecord(const CarState& car_state) {
    car_record_.speed = car_state.speed;
    car_record_.lon = car_state.coords.lon;
    car_record_.lat = car_state.coords.lat;
    car_record_.curs =
        (car_state.direction < 0) ? kAbsentDirection : car_state.direction;
    car_record_.data =
        std::chrono::system_clock::to_time_t(car_state.timestamp);
    car_record_.time = 0;
    car_record_.inputs = car_state.free ? 0 : kCarBusy;
    car_record_.valid = 1;
    car_record_.clat = 'N';
    car_record_.clon = 'E';
    car_record_.probeg = 0;
    car_record_.alarm = 0;
    car_record_.zaj = car_state.moving;
    car_record_.height = 0;
  }

  virtual int Serialize(char* packet, int position,
                        EGTS_RECORD_HEADER* record_header) const override {
    EGTS_SUBRECORD_HEADER* subrecord_header =
        reinterpret_cast<EGTS_SUBRECORD_HEADER*>(packet + position);
    position = packet_add_subrecord_header(packet, position, record_header,
                                           EGTS_SR_POS_DATA);

    return packet_add_subrecord_EGTS_SR_POS_DATA_RECORD(
        packet, position, record_header, subrecord_header, &car_record_);
  }

  virtual size_t GetSize() const override {
    return kSubrecordHeaderSize + kCarStateSubrecordSize;
  }

 private:
  static const uint8_t kCarBusy = 0b10000000;
  static const size_t kCarStateSubrecordSize = sizeof(EGTS_SR_POS_DATA_RECORD) + sizeof(uint8_t) * 3;
  static const unsigned kAbsentDirection = 1;

 private:
  ST_RECORD car_record_;
};

class AbsAnSensDataSubrecord : public Subrecord {
 public:
  AbsAnSensDataSubrecord(const CarState& car_state) {
    Busy = car_state.asnBusy;
    Service = car_state.asnService;
  }

  virtual int Serialize(char* packet, int position,
                        EGTS_RECORD_HEADER* record_header) const override {
    EGTS_SUBRECORD_HEADER* subrecord_header = reinterpret_cast<EGTS_SUBRECORD_HEADER*>(packet + position);
    position = packet_add_subrecord_header(packet, position, record_header, EGTS_SR_ABS_AN_SENS_DATA);
    return SerializeData(packet, position, record_header, subrecord_header);
  }

  virtual size_t GetSize() const override {
    return kSubrecordHeaderSize + kAbsAnSensDataSubrecordSize;
  }

 private:
  static const uint8_t kAsn = 0xC5;
  static const uint8_t kAsnCarBusy    = 0b00000001;
  static const uint8_t kAsnCarService = 0b00000010;
  static const size_t kAbsAnSensDataSubrecordSize = sizeof(uint8_t) * 4;

  bool Busy = false;
  bool Service = false;

  int SerializeData(char *packet, int position, EGTS_RECORD_HEADER *record_header, EGTS_SUBRECORD_HEADER *subrecord_header) const {
    EGTS_PACKET_HEADER* packet_header = reinterpret_cast<EGTS_PACKET_HEADER*>(packet);
    int rec_size = kAbsAnSensDataSubrecordSize;
    int new_size = position + rec_size;

    packet_header->FDL += rec_size;
    record_header->RL += rec_size;
    subrecord_header->SRL = rec_size;

    uint8_t& asn = *reinterpret_cast<uint8_t*>(packet + position);
    uint8_t& asv1 = *reinterpret_cast<uint8_t*>(packet + position + 1);
    uint8_t& asv2 = *reinterpret_cast<uint8_t*>(packet + position + 2);
    uint8_t& asv3 = *reinterpret_cast<uint8_t*>(packet + position + 3);

    asn = kAsn;
    asv1 = Busy ? kAsnCarBusy : 0;
    asv1 |= Service ? kAsnCarService : 0;

    return new_size;
  }
};


}  // namespace telemetry
