#include "egts_service.hpp"
#include "egts/packet_builder.hpp"
#include "logging/logger.hpp"

namespace telemetry {

EGTSService::EGTSService(const std::string& host, unsigned short port,
                         const std::string& imei,
                         std::chrono::milliseconds timeout)
    : host_(host),
      port_(port),
      imei_(imei),
      socket_(),
      timeout_(timeout) {
  response_buf_.reset(new uint8_t[kResponseBufferSize]);
}

void EGTSService::CheckConnection() {
  if (socket_) return;

  LOG_INFO() << "Connect to deptrans... " << imei_ << Endl;
  socket_.reset(new Socket(host_, port_, timeout_));
  LOG_INFO() << "Send to deptrans... " << imei_ << Endl;
  SendAuth(imei_);
  ReceiveAuthAck();
  ReceiveAuthResult();
  LOG_INFO() << "Authorization success " << imei_ << Endl;
}

const ssize_t kMaxCarsInPacket = 1600;

void EGTSService::Send(const std::vector<CarState>& cars) {
  std::unique_lock<std::mutex> lock(mutex_);
  try {
    CheckConnection();

    auto it = cars.cbegin();
    auto end = cars.cend();

    while (it != cars.cend()) {
      auto itend = it;
      if (end - it > kMaxCarsInPacket) {
        itend += kMaxCarsInPacket;
      } else {
        itend = end;
      }

      std::vector<uint8_t> cars_packet = SerializeCars(it, itend);
      socket_->Write(cars_packet.data(), cars_packet.size(), timeout_);

      ssize_t ack_cars = ReceiveTeleDataAck();

      if (ack_cars != (itend - it)) {
        throw std::runtime_error("Teledata acception fail");
      }
      it = itend;

      LOG_INFO() << "Teledata packet send done. Cars send: " << ack_cars << Endl;
    }

    LOG_INFO() << "Teledata send done. Cars send: " << cars.size() << Endl;

  } catch (const std::exception& e) {
    socket_.reset();
    throw;
  }
}

std::vector<uint8_t> EGTSService::SerializeCars(
    std::vector<CarState>::const_iterator carsBegin,
    const std::vector<CarState>::const_iterator carsEnd) const {
  PacketBuilder packetBuilder(EGTS_PT_APPDATA, 0);

  for (; carsBegin != carsEnd; carsBegin++) {
    Record record(EGTS_TELEDATA_SERVICE, EGTS_TELEDATA_SERVICE);
    record.AddSubrecord(std::make_shared<PosDataSubrecord>(*carsBegin));
    if (carsBegin->asn) {
      record.AddSubrecord(std::make_shared<AbsAnSensDataSubrecord>(*carsBegin));
    }
    packetBuilder.AddRecord(record);
  }

  return packetBuilder.Build();
}

void EGTSService::ReceiveAuthAck() {
  EGTSResponse response = ReceiveResponse();

  if (response.GetPtResponseHeader() == nullptr ||
      response.GetPtResponseHeader()->PR != EGTS_PC_OK) {
    throw std::runtime_error("Authorization packet process error");
  }

  auto iter_record = response.GetRecordsIter();

  if (!iter_record.HasNext()) {
    throw std::runtime_error("Authorization response hasn't record");
  }

  auto record = iter_record.Next();
  auto iter_subrecord = record.SubrecordsIter();

  if (!iter_subrecord.HasNext()) {
    throw std::runtime_error("Authorization response record hasn't subrecord");
  }

  auto subrecord = iter_subrecord.Next();

  if (subrecord.GetHeader()->SRT != EGTS_SR_RECORD_RESPONSE) {
    throw std::runtime_error("Authorization response subrecord wrong type");
  }

  const EGTS_SR_RECORD_RESPONSE_RECORD* sr_resp =
      reinterpret_cast<const EGTS_SR_RECORD_RESPONSE_RECORD*>(
          subrecord.GetData());

  if (sr_resp->RST != EGTS_PC_OK) {
    throw std::runtime_error("Authorization record process error");
  }
}

void EGTSService::ReceiveAuthResult() {
  EGTSResponse response = ReceiveResponse();

  auto iter_record = response.GetRecordsIter();

  if (!iter_record.HasNext()) {
    throw std::runtime_error("Authorization result response hasn't record");
  }

  auto record = iter_record.Next();
  auto iter_subrecord = record.SubrecordsIter();

  if (!iter_subrecord.HasNext()) {
    throw std::runtime_error(
        "Authorization result response record hasn't subrecord");
  }

  auto subrecord = iter_subrecord.Next();

  if (subrecord.GetHeader()->SRT != EGTS_SR_RESULT_CODE) {
    throw std::runtime_error(
        "Authorization result response subrecord wrong type");
  }

  const EGTS_SR_RESULT_CODE_RECORD* sr_resp =
      reinterpret_cast<const EGTS_SR_RESULT_CODE_RECORD*>(subrecord.GetData());

  if (sr_resp->RCD != EGTS_PC_OK) {
    throw std::runtime_error("Authorization fail");
  }
}

size_t EGTSService::ReceiveTeleDataAck() {
  EGTSResponse response = ReceiveResponse();

  if (response.GetPtResponseHeader() == nullptr ||
      response.GetPtResponseHeader()->PR != EGTS_PC_OK) {
    throw std::runtime_error("Teledata packet process error");
  }

  auto iter_record = response.GetRecordsIter();

  if (!iter_record.HasNext()) {
    throw std::runtime_error("Teledata response hasn't record");
  }

  auto record = iter_record.Next();
  auto iter_subrecord = record.SubrecordsIter();

  size_t ack_cars = 0;

  while (iter_subrecord.HasNext()) {
    auto subrecord = iter_subrecord.Next();

    if (subrecord.GetHeader()->SRT != EGTS_SR_RECORD_RESPONSE) {
      throw std::runtime_error(
          "Wrong subrecord type in teledata response record");
    }

    const EGTS_SR_RECORD_RESPONSE_RECORD* sr_resp =
        reinterpret_cast<const EGTS_SR_RECORD_RESPONSE_RECORD*>(
            subrecord.GetData());

    if (sr_resp->RST != EGTS_PC_OK) {
      throw std::runtime_error("Car teledata subrecord error status");
    }

    ack_cars++;
  }

  return ack_cars;
}

EGTSResponse EGTSService::ReceiveResponse() {
  socket_->Read(response_buf_.get(), sizeof(EGTS_PACKET_HEADER), timeout_);
  const EGTS_PACKET_HEADER* packet_header =
      reinterpret_cast<const EGTS_PACKET_HEADER*>(response_buf_.get());
  unsigned char packet_header_checksum =
      CRC8EGTS(response_buf_.get(), sizeof(EGTS_PACKET_HEADER) - 1);

  if (packet_header_checksum != packet_header->HCS) {
    throw std::runtime_error("EGTS packet header corrupt");
  }

  uint8_t* records = response_buf_.get() + packet_header->HL;
  socket_->Read(records, packet_header->FDL + kChecksumFieldSize, timeout_);
  unsigned short records_checksum = CRC16EGTS(records, packet_header->FDL);

  unsigned short* data_checksum =
      reinterpret_cast<unsigned short*>(records + packet_header->FDL);

  if (records_checksum != *data_checksum) {
    throw std::runtime_error("EGTS packet body corrupt");
  }

  return EGTSResponse(response_buf_.get());
}

void EGTSService::SendAuth(const std::string& imei) {
  std::vector<uint8_t> buf = SerializeAuth(imei);
  socket_->Write(buf.data(), buf.size(), timeout_);
}

std::vector<uint8_t> EGTSService::SerializeAuth(const std::string& imei) const {
  PacketBuilder builder(EGTS_PT_APPDATA, 0x3);

  Record auth_record(EGTS_AUTH_SERVICE, EGTS_AUTH_SERVICE);
  auth_record.AddSubrecord(
      std::make_shared<IdentitySubrecord>(imei));

  builder.AddRecord(auth_record);

  return builder.Build();
}

}  // namespace telemetry
