#include "socket.hpp"

#include <boost/asio/connect.hpp>
#include <boost/asio/read.hpp>
#include <boost/asio/write.hpp>
#include <boost/lambda/lambda.hpp>

namespace telemetry {

namespace ip = boost::asio::ip;

Socket::Socket(const std::string& host, unsigned short port,
               std::chrono::milliseconds connect_timeout)
    : io_service_(), socket_(io_service_), deadline_(io_service_) {
  deadline_.expires_at(boost::posix_time::pos_infin);
  CheckDeadline();

  ip::tcp::resolver::query query(host, std::to_string(port));
  ip::tcp::resolver::iterator iter =
      ip::tcp::resolver(io_service_).resolve(query);

  deadline_.expires_from_now(
      boost::posix_time::milliseconds(connect_timeout.count()));

  boost::system::error_code ec = boost::asio::error::would_block;
  boost::asio::async_connect(socket_, iter,
                             boost::lambda::var(ec) = boost::lambda::_1);

  do {
    io_service_.run_one();
  } while (ec == boost::asio::error::would_block);

  if (ec != boost::system::errc::success) {
    throw std::runtime_error("Socket connect error: " + ec.message());
  }

  socket_.set_option(ip::tcp::socket::keep_alive(true));
}

void Socket::Write(const uint8_t* buf, size_t size,
                   std::chrono::milliseconds timeout) {
  deadline_.expires_from_now(boost::posix_time::milliseconds(timeout.count()));

  boost::system::error_code ec = boost::asio::error::would_block;
  boost::asio::async_write(socket_, boost::asio::buffer(buf, size),
                           boost::lambda::var(ec) = boost::lambda::_1);

  do {
    io_service_.run_one();
  } while (ec == boost::asio::error::would_block);

  if (ec != boost::system::errc::success) {
    throw std::runtime_error("Socket write error: " + ec.message());
  }
}

void Socket::Read(uint8_t* buf, size_t size,
                  std::chrono::milliseconds timeout) {
  deadline_.expires_from_now(boost::posix_time::milliseconds(timeout.count()));
  socket_.set_option(ip::tcp::socket::keep_alive(true));

  boost::system::error_code ec = boost::asio::error::would_block;
  boost::asio::async_read(socket_, boost::asio::buffer(buf, size),
                          boost::lambda::var(ec) = boost::lambda::_1);

  do {
    io_service_.run_one();
  } while (ec == boost::asio::error::would_block);

  if (ec != boost::system::errc::success) {
    throw std::runtime_error("Socket read error: " + ec.message());
  }
}

void Socket::CheckDeadline() {
  if (deadline_.expires_at() <=
      boost::asio::deadline_timer::traits_type::now()) {
    boost::system::error_code ignored_ec;
    socket_.close(ignored_ec);
    deadline_.expires_at(boost::posix_time::pos_infin);
  }

  deadline_.async_wait(std::bind(&Socket::CheckDeadline, this));
}

}  // namespace telemetry
