# -*- coding: utf-8 -*-

from os import makedirs, remove, environ, execv, chmod, chdir, putenv
from os.path import join, expanduser, exists, dirname
from psutil import Process, pid_exists
from contextlib import suppress
from shutil import copyfile
from subprocess import Popen
from requests import get
from requests.exceptions import ConnectionError
from tqdm import tqdm
from tarfile import open as tar_open
from time import sleep

from drive.devops.cli.secrets.vault import VaultClient


_DEFAULT_DRIVE_DIR = "~/arcadia/drive"
_DEFAULT_STATE_DIR = "~/.drive-backend/{environ}"


class FileRes:
    def __init__(self, url):
        self.url = url

    def download(self, path):
        if exists(path):
            print("File already exists")
            return
        resp = get(self.url, stream=True)
        total_size = int(resp.headers.get("content-length", 0))
        progress_bar = tqdm(total=total_size, unit="iB", unit_scale=True)
        with open(path, "wb") as fd:
            for block in resp.iter_content(4096):
                progress_bar.update(len(block))
                fd.write(block)
        progress_bar.close()


class DirRes(FileRes):
    def download(self, path):
        if exists(path):
            print("File already exists")
            return
        resp = get(self.url + "?stream=tgz", stream=True)
        tar = tar_open(fileobj=resp.raw, mode="r|gz")
        tar.extractall(path)


class BackendEnv:
    config_file = ""

    SECRETS = {}
    SECRET_FILES = {}
    RESOURCES = {}
    VARIABLES = {}
    FLAGS = {}

    def __init__(self, state_dir, drive_dir):
        self.state_dir = state_dir
        self.drive_dir = drive_dir
        self.drive_env = {}
        self._state_pid = None
        self._state_port = None
        self._process = None

    def _restore_state(self):
        state_file = join(self.state_dir, ".state")
        if not exists(state_file):
            return
        with open(state_file) as fd:
            pid, port = map(int, fd.read().split())
            if pid_exists(pid):
                self._state_pid = pid
                self._state_port = port
                self._process = Process(pid)

    def prepare(self, ctype):
        with suppress(FileExistsError):
            makedirs(self.state_dir)
        self.drive_env = self.VARIABLES
        print("Running backend with CType: {}".format(ctype))
        yav = VaultClient()
        for key, value in self.SECRETS.items():
            secret = yav.get_version(value[0])
            self.drive_env[key] = secret["value"][value[1]]
        for key, value in self.SECRET_FILES.items():
            secret = yav.get_version(value[0])
            path = join(self.state_dir, key)
            with suppress(FileExistsError):
                makedirs(dirname(path))
            with open(path, "w") as fd:
                fd.write(secret["value"][value[1]])
            chmod(path, 0o600)
        config_template_file = join(self.drive_dir, "backend/configs/{}".format(self.config_file))
        config_file = join(self.state_dir, "drive.conf")
        print("Write file:", config_file)
        copy_config(config_template_file, config_file, ctype)
        environ_template_file = join(self.drive_dir, "backend/configs/environment.lua")
        environ_file = join(self.state_dir, "drive.env")
        print("Write file:", environ_file)
        copyfile(environ_template_file, environ_file)
        environ_ut_template_file = join(self.drive_dir, "backend/configs/ut.lua")
        environ_ut_file = join(self.state_dir, "drive-ut.env")
        print("Write file:", environ_ut_file)
        copy_environ_ut(environ_ut_template_file, environ_ut_file, self.drive_env)
        for path, resource in self.RESOURCES.items():
            print("Downloading:", path)
            resource.download(join(self.state_dir, path))
        with suppress(FileExistsError):
            makedirs(join(self.state_dir, "logs"))
        with suppress(FileExistsError):
            makedirs(join(self.state_dir, "controls"))

    def set_yt_prof_port(self, port):
        self.drive_env["YTPROF_SERVER_PORT"] = str(port)

    def gdb(self, port):
        self._restore_state()
        for key, value in self.drive_env.items():
            putenv(key, value)
        chdir(self.state_dir)
        execv(
            join(self.drive_dir, "../ya"),
            [
                "ya", "gdb", "--args",
                self.get_binary_path(), "drive.conf",
                "-E", "drive.env",
                "-E", "drive-ut.env",
                "-V", "WorkDir=.",
                "-V", "LOG_DIR=logs/",
                "-V", "LOG_POSTFIX=",
                "-V", "BasePort={}".format(port),
            ]
        )

    def start(self, port):
        self._restore_state()
        if self._state_pid:
            print("Backend already running")
            return
        proc = Popen(
            [
                self.get_binary_path(), "drive.conf",
                "-E", "drive.env",
                "-E", "drive-ut.env",
                "-V", "WorkDir=.",
                "-V", "LOG_DIR=logs/",
                "-V", "LOG_POSTFIX=",
                "-V", "BasePort={}".format(port),
            ],
            cwd=self.state_dir,
            env=self.drive_env,
            start_new_session=True,
        )
        print("Starting backend")
        while True:
            sleep(1)
            with suppress(ConnectionError):
                running = proc.poll() is None
                if not running:
                    print("Backend start failed")
                    return
                resp = get("http://localhost:{}/ping".format(port))
                resp.raise_for_status()
                print("Backend started: {}".format(proc.pid))
                state_file = join(self.state_dir, ".state")
                with open(state_file, "w") as fd:
                    fd.write("{} {}".format(proc.pid, port))
                break

    def stop(self):
        self._restore_state()
        if not self._state_pid:
            print("Backend is not running")
            return
        print("Stopping backend: {}".format(self._state_pid))
        state_file = join(self.state_dir, ".state")
        try:
            resp = get(
                "http://localhost:{}/?command=shutdown&rigid_level=1".format(self._state_port+3),
                timeout=600,
            )
            resp.raise_for_status()
            self._process.wait(10)
            remove(state_file)
            print("Backend stopped")
        except:
            print("Force stopping backend")
            try:
                self._process.kill()
                self._process.wait(30)
                remove(state_file)
                print("Backend stopped")
            except:
                print("Unable to force stopping backend")

    def get_pid(self):
        self._restore_state()
        return self._state_pid

    def get_binary_path(self):
        return join(self.drive_dir, "backend/server/server")


class TestingBackendEnv(BackendEnv):
    config_file = "testing.conf"

    SECRETS = {
        "CLICKHOUSE_PASSWORD": ("sec-01fg3v49egd1z1predtxrn3fjn", "password"),
        "MAIN_DB_PASSWORD": ("sec-01f9eczqzk4ej2gww37787pwhs", "password"),
        "TEST_DRIVESMALL_DB_PASSWORD": ("sec-01f9eczqzk4ej2gww37787pwhs", "small_password"),
        "TVM_2000184_SECRET": ("sec-01fcv11knng3vqdfrsxjthp716", "client_secret"),
        "TVM_2000615_SECRET": ("sec-01f9emzxcv7h46r6fkbxnwx46x", "client_secret"),
        "TVM_2010062_SECRET": ("sec-01f9en1xpeda1y3r93qnze828y", "client_secret"),
        "YDB_TOKEN": ("sec-01ff5fax3wxt1d56x0gshphfy9", "token"),
    }
    RESOURCES = {
        "allCAs.pem": FileRes("https://crls.yandex.net/allCAs.pem"),
        "geobase.bin": FileRes("https://proxy.sandbox.yandex-team.ru/1685328775"),
        "ffmpeg": FileRes("https://proxy.sandbox.yandex-team.ru/1117704486"),
    }
    VARIABLES = {
        "SKIP_BACKGROUND_PROCESSES": "True",
    }


class PrestableBackendEnv(BackendEnv):
    config_file = "prestable.conf"

    SECRETS = {
        "CLICKHOUSE_PASSWORD": ("sec-01fg3v49egd1z1predtxrn3fjn", "password"),
        "MDS_ACCESS_KEY_ID": ("sec-01fxt0rts0eb3xgt9thbrxfa0p", "key_id"),
        "MDS_ACCESS_SECRET_KEY": ("sec-01fxt0gfg9e1mwnqj7rr1hb4cv", "secret_key"),
        "TVM_2000615_SECRET": ("sec-01f9emzxcv7h46r6fkbxnwx46x", "client_secret"),
        "TVM_2010450_SECRET": ("sec-01fm9wt5vnr4av9mw6485zg1k6", "client_secret"),
        "YDB_TOKEN": ("sec-01ff5fax3wxt1d56x0gshphfy9", "token"),
        "PRODUCTION_DB_PRESTABLE_PASSWORD": ("sec-01fasvvtc46cy1emtg1bv0r6v8", "password"),
    }
    SECRET_FILES = {
        "drive-prod_drive_db-telematics-password/PASSWORD": ("sec-01fm9vyd4tg0j4jxpbmt2m0wnp", "telematics_password"),
        "drive-prestable_drive_db-telematics-password/PASSWORD": ("sec-01fm9w329j5yydbnxj1yrf0vdc", "telematics_password"),
        "chats_db_password/chats_db_password": ("sec-01fm9wez9w58txvpxt2qgtgrkv", "password"),
    }
    RESOURCES = {
        "allCAs.pem": FileRes("https://crls.yandex.net/allCAs.pem"),
        "geobase.bin": FileRes("https://proxy.sandbox.yandex-team.ru/1685328775"),
        "ffmpeg": FileRes("https://proxy.sandbox.yandex-team.ru/1117704486"),
    }
    VARIABLES = {
        "SKIP_BACKGROUND_PROCESSES": "True",
    }


class DrivematicsTestingBackendEnv(BackendEnv):
    config_file = "drivematics_testing.conf.xml"

    SECRETS = {
        "CLICKHOUSE_PASSWORD": ("sec-01fg3v49egd1z1predtxrn3fjn", "password"),
        "DM_ANALYTICS_CLICKHOUSE_PASSWORD": ("ver-01g0ej6pdctxv9d6qcgshd51bk", "PASSWORD"),
        "MAIN_DB_PASSWORD": ("sec-01fajd57qzp0saqhsevht70d01", "PASSWORD"),
        "SNS_ACCESS_KEY_ID": ("sec-01g7bwd6bydspgnswgdf0074bb", "ACCESS_KEY_ID"),
        "SNS_ACCESS_SECRET_KEY": ("sec-01g7bwd6bydspgnswgdf0074bb", "ACCESS_SECRET_KEY"),
        "TEST_DRIVESMALL_DB_PASSWORD": ("sec-01f9eczqzk4ej2gww37787pwhs", "small_password"),
        "TVM_2000615_SECRET": ("sec-01f9emzxcv7h46r6fkbxnwx46x", "client_secret"),
        "TVM_2010062_SECRET": ("sec-01f9en1xpeda1y3r93qnze828y", "client_secret"),
        "MDS_ACCESS_KEY_ID": ("sec-01g8xbef5qcbc2ty7ab1m75g46", "key_id"),
        "MDS_ACCESS_SECRET_KEY": ("sec-01g8xbef5qcbc2ty7ab1m75g46", "secret_key"),
    }
    RESOURCES = {
        "allCAs.pem": FileRes("https://crls.yandex.net/allCAs.pem"),
        "ffmpeg": FileRes("https://proxy.sandbox.yandex-team.ru/1117704486"),
    }
    VARIABLES = {
        "REPORT_DEBUG_INFO": "true",
        "SKIP_BACKGROUND_PROCESSES": "True",
    }


class DrivematicsPrestableBackendEnv(BackendEnv):
    config_file = "drivematics_prestable.conf.xml"

    SECRETS = {
        "CLICKHOUSE_PASSWORD": ("sec-01fg3v49egd1z1predtxrn3fjn", "password"),
        "DM_ANALYTICS_CLICKHOUSE_PASSWORD": ("ver-01g0ej6pdctxv9d6qcgshd51bk", "PASSWORD"),
        "DOCUMENT_PHOTOS_CONTENT_ENCRYPTION_KEY": ("sec-01fx4txq5avwb22td0099asyyv", "DOCUMENT_PHOTOS_CONTENT_ENCRYPTION_KEY"),
        "MDS_ACCESS_KEY_ID": ("sec-01fxt0rts0eb3xgt9thbrxfa0p", "key_id"),
        "MDS_ACCESS_SECRET_KEY": ("sec-01fxt0gfg9e1mwnqj7rr1hb4cv", "secret_key"),
        "SNS_ACCESS_KEY_ID": ("sec-01g7bwj2jtwpdcvb0km06tqyyz", "ACCESS_KEY_ID"),
        "SNS_ACCESS_SECRET_KEY": ("sec-01g7bwj2jtwpdcvb0km06tqyyz", "ACCESS_SECRET_KEY"),
        "TVM_2000615_SECRET": ("sec-01f9emzxcv7h46r6fkbxnwx46x", "client_secret"),
    }
    SECRET_FILES = {
        "drive-prod_drive_db-telematics-password/PASSWORD": ("sec-01fm9vyd4tg0j4jxpbmt2m0wnp", "telematics_password"),
        "drive-prestable_drive_db-telematics-password/PASSWORD": ("sec-01fm9w329j5yydbnxj1yrf0vdc", "telematics_password"),
        "leasing_db_password/PRESTABLE_PASSWORD": ("sec-01ewxe4ppazasgckyzagf4fk23", "PRESTABLE_PASSWORD"),
    }
    RESOURCES = {
        "allCAs.pem": FileRes("https://crls.yandex.net/allCAs.pem"),
    }
    VARIABLES = {
        "REPORT_DEBUG_INFO": "true",
        "SKIP_BACKGROUND_PROCESSES": "True",
    }


ENVS = {
    "testing": TestingBackendEnv,
    "prestable": PrestableBackendEnv,
    "drivematics_testing": DrivematicsTestingBackendEnv,
    "drivematics_prestable": DrivematicsPrestableBackendEnv,
}


def _add_common_arguments(cmd):
    cmd.add_argument(
        "--environ",
        choices=ENVS.keys(),
        default="testing",
        help="Specifies backend environment configuration",
    )
    cmd.add_argument(
        "--drive-dir",
        default=_DEFAULT_DRIVE_DIR,
        help="Path to drive directory in Arcadia",
    )
    cmd.add_argument(
        "--state-dir",
        default=_DEFAULT_STATE_DIR,
        help="Path to state directory for running backend",
    )


def register(group):
    start_cmd = group.add_parser("start-backend")
    start_cmd.set_defaults(main=start_main)
    _add_common_arguments(start_cmd)
    start_cmd.add_argument(
        "--dry-run",
        dest="dry_run",
        help="Prepare the environment only",
        action="store_true"
    )
    start_cmd.add_argument(
        "--gdb",
        dest="gdb",
        help="Run gdb",
        action="store_true"
    )
    start_cmd.add_argument(
        "--port",
        type=int,
        default=10000,
        help="Port for starting backend",
    )
    start_cmd.add_argument(
        "--ctype",
        default="",
        help="CType for backend",
    )
    start_cmd.add_argument(
        "--yt-prof",
        action="store_true",
        dest="yt_prof",
        help="Enable YT profiler"
    )
    stop_cmd = group.add_parser("stop-backend")
    stop_cmd.set_defaults(main=stop_main)
    _add_common_arguments(stop_cmd)
    gdb_cmd = group.add_parser("gdb-backend")
    _add_common_arguments(gdb_cmd)
    gdb_cmd.set_defaults(main=gdb_main)


def copy_config(src, dst, ctype):
    with open(src) as sfd:
        with open(dst, "w") as dfd:
            for line in sfd:
                if "UnifiedAgentEventLog" in line:
                    continue
                if "CType" in line:
                    continue
                if "ActiveQueue" in line:
                    dfd.write("            ActiveQueue: tests\n")
                    continue
                if "trust-payments.paysys.yandex.net" in line:
                    dfd.write(line.replace("trust-payments", "trust-payments-test"))
                    continue
                dfd.write(line)
                if line.strip() == "<DaemonConfig>":
                    dfd.write("    CType: {}\n".format(ctype))


def copy_environ_ut(src, dst, env):
    with open(src) as sfd:
        with open(dst, "w") as dfd:
            for line in sfd:
                good = True
                for key in env.keys():
                    if key in line:
                        good = False
                        break
                if good:
                    dfd.write(line)


def start_main(opts):
    state_dir = expanduser(opts.state_dir.format(environ=opts.environ))
    drive_dir = expanduser(opts.drive_dir)
    env = ENVS[opts.environ](state_dir, drive_dir)
    ctype = opts.ctype or environ.get("USER", "nobody") + "_cli"
    env.prepare(ctype)

    port = opts.port
    if opts.yt_prof:
        yt_prof_port = port + 100
        env.set_yt_prof_port(yt_prof_port)
        print("YT profiler port: {}".format(yt_prof_port))

    if opts.dry_run:
        print("Environment prepared: {}".format(env.state_dir))
    if opts.gdb:
        env.gdb(opts.port)
    else:
        env.start(opts.port)


def stop_main(opts):
    state_dir = expanduser(opts.state_dir.format(environ=opts.environ))
    drive_dir = expanduser(opts.drive_dir)
    env = ENVS[opts.environ](state_dir, drive_dir)
    env.stop()


def gdb_main(opts):
    state_dir = expanduser(opts.state_dir.format(environ=opts.environ))
    drive_dir = expanduser(opts.drive_dir)
    env = ENVS[opts.environ](state_dir, drive_dir)
    pid = env.get_pid()
    if not pid:
        print("Backend is not running")
        return
    execv("ya", ["ya", "tool", "gdb", "--pid", str(pid), env.get_binary_path()])
