import json
import logging
import re


PROMOCODE_GENERATOR_TYPES=[
    'promo_generator_billing_tag',
    'promo_generator_referral_code',
    'promo_generator_simple_tag',
    'promo_generator_rides',
]


def fill_parser(parser):
    parser.add_argument('-y', dest='cleanup_apply', help='apply', action='store_true')
    subparsers = parser.add_subparsers(help='entity')

    actions_parser = subparsers.add_parser('actions', help='cleanup actions')
    actions_parser.add_argument('--erase', dest='erase', help='erase actions', action='store_true')
    actions_parser.set_defaults(func=cleanup_actions)

    roles_parser = subparsers.add_parser('roles', help='cleanup roles')
    roles_parser.add_argument('--skip', dest='skip', help='skip roles', action='append')
    roles_parser.set_defaults(func=cleanup_roles)

    settings_parser = subparsers.add_parser('settings', help='cleanup settings')
    settings_parser.add_argument('--regex', dest='settings_regex')
    settings_parser.set_defaults(func=cleanup_settings)


def cleanup_actions(client, args):
    apply = args.cleanup_apply
    erase = args.erase
    alive_actions = set()

    roles = client.list_roles(report='compact')
    for role in roles:
        action_ids = role['action_ids']
        for action_id in action_ids:
            alive_actions.add(action_id)

    tag_descriptions = client.list_tag_descriptions()
    for tag_description in tag_descriptions:
        meta = json.loads(tag_description['meta'])
        action_ids_string = meta.get('action_ids')
        if action_ids_string:
            action_ids = action_ids_string.split(',')
            for action_id in action_ids:
                alive_actions.add(action_id)

    actions = client.list_actions()
    for action in actions:
        parent = action.get('parent')
        if parent:
            alive_actions.add(parent)
        action_id = action['action_id']
        action_type = action['action_type']
        if action_type in PROMOCODE_GENERATOR_TYPES:
            alive_actions.add(action_id)

    dead_actions = set()
    for action in actions:
        action_id = action['action_id']
        if action_id not in alive_actions:
            dead_actions.add(action_id)

    logging.info('{} alive actions'.format(len(alive_actions)))
    logging.info('{} dead actions'.format(len(dead_actions)))

    for action_id in dead_actions:
        if apply:
            print('{}: removing'.format(action_id))
            client.remove_action(action_id, erase=erase)
        else:
            print('dead action {}'.format(action_id))


def cleanup_roles(client, args):
    apply = args.cleanup_apply
    custom_skipped = args.skip or []
    default_skipped = [
        'GR_default_user_base',
        'b2b_flow',
        'default_user',
        'discount_plus',
        'fast_registered_access',
        'hidden_manual_transmission',
        'onboarding_role',
        'role_enable_chat',
        'user_access_base',
        'yandex_user',
    ]

    refs = dict()
    roles = client.list_roles(report='compact')
    for role in roles:
        role_id = role['role_id']
        refs[role_id] = 0

    for role_id in custom_skipped:
        refs[role_id] = 1
    for role_id in default_skipped:
        refs[role_id] = 1

    for role in roles:
        slave_role_ids = role.get('slave_role_ids', [])
        for slave_role_id in slave_role_ids:
            refs[slave_role_id] += 1

    tag_descriptions = client.list_tag_descriptions()
    for tag_description in tag_descriptions:
        meta = json.loads(tag_description['meta'])
        tag_roles = meta.get('roles')
        if tag_roles:
            for role_id in tag_roles:
                refs[role_id] += 1

    for role_id, refcount in refs.items():
        if refcount > 0:
            continue
        logging.debug('list_role_users for {}'.format(role_id))
        role_users = client.list_role_users(role_id, limit=1)
        refs[role_id] += len(role_users)

    alive_roles = set()
    dead_roles = set()
    for role_id, refcount in refs.items():
        if refcount > 0:
            alive_roles.add(role_id)
        else:
            dead_roles.add(role_id)
    logging.info('{} alive roles'.format(len(alive_roles)))
    logging.info('{} dead roles'.format(len(dead_roles)))

    for role in roles:
        role_id = role['role_id']
        if role_id not in dead_roles:
            continue
        action_ids = role.get('action_ids', [])
        slave_role_ids = role.get('slave_role_ids', [])
        if apply:
            print('{}: unlinking actions [{}] and roles [{}]'.format(
                role_id,
                ','.join(action_ids),
                ','.join(slave_role_ids)
            ))
            client.unlink_from_role(role_id, action_ids, slave_role_ids)
            print('{}: removing'.format(role_id))
            client.remove_role(role_id)
        else:
            print('dead role {}'.format(role_id))


def cleanup_settings(client, args):
    apply = args.cleanup_apply
    pattern = re.compile(args.settings_regex)
    settings = client.list_settings()
    for setting in settings:
        key = setting['setting_key']
        if pattern.match(key):
            if apply:
                logging.info('removing setting {}'.format(key))
                client.remove_setting(key)
            else:
                logging.info('found setting {}'.format(key))
