import functools
import json
import logging
import multiprocessing
import os
import sys
import texttable
from itertools import combinations

import drive.devops.cli.client.cleanup as cleanup
import drive.devops.cli.client.compare as compare
import drive.devops.cli.client.firmware as firmware
import drive.devops.cli.client.model as model
import drive.devops.cli.client.rtbg as rtbg
import drive.devops.cli.client.scenarios as scenarios
import drive.devops.cli.client.taxicorp as taxicorp
import drive.devops.cli.client.transfer as transfer
import drive.devops.cli.client.user as user
import drive.devops.cli.client.world as world

from drive.backend.api import client as api
from drive.backend.api import objects as apiobjects


def _execute_command(client, cmd, obj):
    assert cmd
    assert client
    try:
        result = client.command(obj, cmd)
        if result.status == "SUCCESS":
            print("{}: {}".format(obj.id, result.value if result.value is not None else result.status))
        else:
            print("{}: {} {}".format(obj.id, result.status, result.message))
    except Exception as e:
        logging.error("{}: an exception occurred {}".format(obj.id, e))


def _execute_scenario(client, scenario, argument, obj):
    assert client
    assert scenario
    assert obj
    try:
        scenario(client, obj, argument)
    except Exception as e:
        logging.error("{}: an exception occurred {}".format(obj.id, e))


def _add_filters(parser):
    parser.add_argument("--fast-object-file", dest="fast_object_file", action="store_true", help="do not invoke car/list")
    parser.add_argument("--fraction", dest="fraction", metavar="0 – 1", type=float)
    parser.add_argument("--object-file", dest="object_file", metavar="FILE", type=str)
    parser.add_argument("--object-id", dest="object_ids", metavar="ID", type=str, action="append")
    parser.add_argument("--imei", dest="imeis", metavar="IMEI", type=str, action="append")
    parser.add_argument("--model", dest="models", metavar="MODEL", type=str, action="append")
    parser.add_argument("--status", dest="statuses", metavar="STATUS", type=str, action="append")
    parser.add_argument("--location-tag", dest="location_tags", metavar="TAG", type=str, action="append")
    parser.add_argument("--tag", dest="tags", metavar="TAG", type=str, action="append")
    parser.add_argument("--vin", dest="vins", metavar="VIN", type=str, action="append")
    return parser


def _apply_filters(objects, args):
    fast_object_file = vars(args).get("fast_object_file")
    fraction = vars(args).get("fraction")
    object_file = vars(args).get("object_file")
    object_ids = vars(args).get("object_ids")
    imeis = vars(args).get("imeis")
    location_tags = vars(args).get("location_tags")
    models = vars(args).get("models")
    statuses = vars(args).get("statuses")
    vins = vars(args).get("vins")

    if object_file and not fast_object_file:
        oids = []
        lines = open(object_file).readlines()
        for line in lines:
            oid = line.strip()
            oids.append(oid)

        copy = []
        for i in objects:
            if i.id in oids:
                logging.debug("matched {}".format(i.id))
                copy.append(i)
            else:
                logging.debug("removing {} due to object_id".format(i.id))
        objects = copy

    if object_ids:
        logging.info("matching object_ids: {}".format('\t'.join(object_ids)))
        copy = []
        for i in objects:
            if i.id in object_ids:
                logging.debug("matched {}".format(i.id))
                copy.append(i)
            else:
                logging.debug("removing {} due to object_id".format(i.id))
        objects = copy

    if imeis:
        logging.info("matching imeis: {}".format('\t'.join(imeis)))
        copy = []
        for i in objects:
            if i.imei in imeis:
                logging.debug("matched {}".format(i.id))
                copy.append(i)
            else:
                logging.debug("removing {} due to imei".format(i.id))
        objects = copy

    if models:
        logging.info("matching models: {}".format('\t'.join(models)))
        copy = []
        for i in objects:
            if i.model in models:
                logging.debug("matched {}".format(i.id))
                copy.append(i)
            else:
                logging.debug("removing {} due to model".format(i.id))
        objects = copy

    if statuses:
        logging.info("matching statuses: {}".format('\t'.join(statuses)))
        copy = []
        for i in objects:
            if i.status in statuses:
                logging.debug("matched {}".format(i.id))
                copy.append(i)
            else:
                logging.debug("removing {} due to status".format(i.id))
        objects = copy

    if location_tags:
        logging.info("matching location tags: {}".format('\t'.join(location_tags)))
        copy = []
        for i in objects:
            matched = False
            if i.location and i.location.tags:
                for tag in location_tags:
                    if tag in i.location.tags:
                        matched = True
                        break
            if matched:
                logging.debug("matched {}".format(i.id))
                copy.append(i)
            else:
                logging.debug("removing {} due to location tags".format(i.id))
        objects = copy

    if vins:
        logging.info("matching vins: {}".format('\t'.join(vins)))
        copy = []
        for i in objects:
            if i.vin in vins:
                logging.debug("matched {}".format(i.id))
                copy.append(i)
            else:
                logging.debug("removing {} due to status".format(i.id))
        objects = copy

    if fraction:
        logging.info("matching fraction: {}".format(fraction))
        copy = []
        for i in objects:
            hash_value = hash(i.id)
            hash_abs = abs(hash_value)
            if hash_abs % 10000 < fraction * 10000:
                logging.debug("matched {}".format(i.id))
                copy.append(i)
            else:
                logging.debug("removing {} due to hash abs {}".format(i.id, hash_abs))
        objects = copy

    return objects


def _ask_confirmation(prompt, args=None):
    if args and args.force:
        return True

    while True:
        sys.stderr.write("{}. Proceed? [y/N]: ".format(prompt))
        answer = input()
        if not answer:
            return False
        if answer not in ['y', 'Y', 'n', 'N']:
            print("Please enter 'y' or 'n'")
            continue
        if answer == 'y' or answer == 'Y':
            return True
        if answer == 'n' or answer == 'N':
            return False


def _list_objects_filtered(client, args):
    tags = vars(args).get("tags")
    if tags:
        logging.info("matching tags: {}".format('\t'.join(tags)))
    objects = []
    if args.object_file and args.fast_object_file:
        lines = open(args.object_file).readlines()
        for line in lines:
            object_id = line.strip()
            objects.append(apiobjects.Object(id=object_id))
    else:
        objects = client.list_cars(tags_filter=tags)
    objects = _apply_filters(objects, args)
    logging.info("matched {} objects".format(len(objects)))
    return objects


def main_analyzer(client, session_id, ride_id):
    tracks_service = "drive_navigator"
    segments = client.analyzer(session_id, ride_id, tracks_service)
    for segment in segments:
        print(json.dumps(segment))


def main_firmware(client, args):
    action = args.firmware_action
    file = args.file
    model = vars(args).get("model")

    if action == "add":
        data = open(file, 'rb').read()
        name = os.path.splitext(os.path.basename(file))[0]
        models = []
        if model:
            models.append(model)
        if len(models) == 0:
            models = firmware.deduce_models(name)
        if not models:
            raise Exception("cannot determine models for {}".format(name))
        for model in models:
            alias = firmware.create_alias(name, model)
            logging.info("adding firmware {} for {} with alias {}".format(name, model, alias))
            client.add_firmware(name, data, alias=alias, model=model)
    elif action == "list":
        values = client.list_firmware()
        print(values)
        for value in values:
            print(value)
    else:
        raise Exception("unknown firmware action: {}".format(action))


def main_clone_action(client, args):
    typ = args.action_type
    suffix = args.suffix
    patch = json.loads(args.patch)

    location_tags = []
    location_tags_string = vars(args).get("location_tags")
    if location_tags_string:
        location_tags = [s for s in location_tags_string.split(',')]
    logging.info("location tags: {}".format(' '.join(location_tags)))

    tags = []
    tags_string = vars(args).get("tags")
    if tags_string:
        tags = [s for s in tags_string.split(',')]
    logging.info("tags: {}".format(' '.join(tags)))

    filtered_actions = []
    actions = client.list_actions()
    for action in actions:
        if action["action_type"] != typ:
            continue
        action_meta = action["action_meta"]
        location_tags_matched = len(location_tags) == 0
        for tag in action_meta.get("tags_in_point", []):
            if tag in location_tags:
                location_tags_matched = True
                break
        if not location_tags_matched:
            continue

        tags_filter = [s for s in action_meta.get("tags_filter", "").split(',')]
        tags_filter_matched = len(tags) == 0
        for tag in tags_filter:
            if tag in tags:
                tags_filter_matched = True
                break
        if not tags_filter_matched:
            continue

        filtered_actions.append(action)

    logging.info("filtered actions:")
    for action in filtered_actions:
        action_id = action["action_id"]
        logging.info(action_id)

    if not args.apply:
        return
    for action in filtered_actions:
        original_action_id = action["action_id"]
        action_id = original_action_id + suffix
        action["action_id"] = action_id
        action["action_meta"].update(patch)
        action["parent"] = original_action_id
        logging.info("adding {}".format(action_id))
        client.add_action(action)


def main_command(client, args):
    command = json.loads(args.command)
    cmd = apiobjects.TelematicsCommand.from_json(command)
    cmd.id = vars(args).get("sensor_id")
    cmd.subid = vars(args).get("sensor_subid")

    objects = _list_objects_filtered(client, args)
    objects_count = len(objects)
    if objects_count > 0:
        if not _ask_confirmation("{} objects".format(objects_count), args):
            return

    proc = functools.partial(_execute_command, client, cmd)
    pool = multiprocessing.Pool(args.threads)
    pool.map(proc, objects)


def main_scenario(client, args):
    name = args.scenario
    argument = args.argument
    scenario = scenarios.get_scenario(name)
    assert scenario

    objects = _list_objects_filtered(client, args)
    objects_count = len(objects)
    if objects_count > 0:
        if not _ask_confirmation("{} objects".format(objects_count), args):
            return

    proc = functools.partial(_execute_scenario, client, scenario, argument)
    pool = multiprocessing.Pool(args.threads)
    pool.map(proc, objects)


def main_search(client, query, fields):
    objects = client.search(query)
    for obj in objects:
        ff = [getattr(obj, field, "null") for field in fields]
        print('\t'.join(ff))


def main_experiment_add(client, args):
    name = args.experiment_name
    if args.is_reverse and not name.endswith("reverse"):
        name += "-reverse"
    description = args.description

    actions = client.list_actions()
    action_ids = []
    for action in actions:
        action_ids.append(action["action_id"])

    disable_correctors_string = args.disable_correctors
    disable_correctors = None
    if disable_correctors_string:
        disable_correctors = [s for s in disable_correctors_string.split(',')]
        for corrector in disable_correctors:
            assert corrector in action_ids, "corrector {} is not registered in system".format(corrector)

    discount = args.discount
    if discount:
        assert discount <= 100, discount
        assert discount >= -100, discount

    price_model = args.price_model
    parking_price_model = args.parking_price_model
    if price_model:
        ranking_models = client.list_ranking_models()
        assert price_model in ranking_models, price_model
    if parking_price_model:
        ranking_models = client.list_ranking_models()
        assert parking_price_model in ranking_models, parking_price_model

    assert disable_correctors is not None or discount is not None or price_model is not None or parking_price_model is not None, \
        "one of disable_correctors, discount or price_model/parking_price_model must be specified"
    has_disable_correctors = 1 if disable_correctors else 0
    has_discount = 1 if discount else 0
    has_price_models = 1 if (price_model or parking_price_model) else 0
    assert has_disable_correctors + has_discount + has_price_models == 1, \
        "only on of disable_correctors, discount and price_model/parking_price_model should be specified"
    priority = vars(args).get("priority") or 900

    all_location_tags = []
    areas = client.list_areas()
    for area in areas:
        for tag in area["area_tags"].split(", "):
            all_location_tags.append(tag)

    location_tags = []
    location_tags_string = vars(args).get("location_tags")
    if location_tags_string:
        location_tags = [s for s in location_tags_string.split(',')]
    logging.info("location tags: {}".format(' '.join(location_tags)))
    for tag in location_tags:
        stripped_tag = tag.strip('!')
        assert stripped_tag in all_location_tags, tag

    all_offers = action_ids
    offers = []
    offers_string = vars(args).get("offers")
    if offers_string:
        offers = [s for s in offers_string.split(',')]
    logging.info("offers: {}".format(' '.join(offers)))
    for offer in offers:
        assert offer in all_offers, offer

    control_buckets = [] if args.control is None else [int(s) for s in args.control.split(',')]
    logging.info("control: {}".format(' '.join(
        [str(i) for i in sorted(control_buckets)]
    )))
    experiment_buckets = [] if args.experiment is None else [int(s) for s in args.experiment.split(',')]
    logging.info("experiment: {}".format(' '.join(
        [str(i) for i in sorted(experiment_buckets)]
    )))

    assert len(control_buckets) == len(experiment_buckets)
    for bucket in sorted(control_buckets):
        assert bucket not in experiment_buckets

    actions = client.list_actions()
    action_revisions = {}
    for action in actions:
        action_id = action["action_id"]
        action_revision = action.get("action_revision")
        if action_revision:
            action_revisions[action_id] = action_revision

    action_type = None
    if disable_correctors:
        action_type = "disable_corrector"
    elif price_model or parking_price_model:
        action_type = "offer_corrector_price_model"
    else:
        action_type = "offer_corrector_discounts"
    assert action_type

    control_name = "{}-control".format(name)
    control = {
        "action_id": control_name,
        "action_description": "Experiment {} – control".format(name),
        "action_type": action_type,
        "action_meta": {
            "bucket_salt": name,
            "buckets": control_buckets,
            "complementary_buckets": experiment_buckets,
            "offers": offers,
            "priority": priority,
            "tags_in_point": location_tags,
        },

        "enabled": True,
    }
    if control_name in action_revisions:
        control["action_revision"] = action_revisions[control_name]

    global_control_name = "global-{}-control".format(name)
    global_control = {
        "action_id": global_control_name,
        "action_description": "Experiment {} – global control".format(name),
        "action_type": action_type,
        "action_meta": {
            "bucket_salt": name,
            "buckets": control_buckets,
            "complementary_buckets": experiment_buckets,
            "priority": priority,
            "tags_in_point": location_tags,
        },
        "enabled": True,
    }
    if global_control_name in action_revisions:
        global_control["action_revision"] = action_revisions[global_control_name]

    experiment_name = "{}-experiment".format(name)
    experiment = {
        "action_id": experiment_name,
        "action_description": "Experiment {} – experiment".format(name),
        "action_type": action_type,
        "action_meta": {
            "bucket_salt": name,
            "buckets": experiment_buckets,
            "complementary_buckets": control_buckets,
            "offers": offers,
            "priority": priority,
            "tags_in_point": location_tags,
        },
        "enabled": True,
    }
    if experiment_name in action_revisions:
        experiment["action_revision"] = action_revisions[experiment_name]

    global_experiment_name = "global-{}-experiment".format(name)
    global_experiment = {
        "action_id": global_experiment_name,
        "action_description": "Experiment {} – global experiment".format(name),
        "action_type": action_type,
        "action_meta": {
            "bucket_salt": name,
            "buckets": experiment_buckets,
            "complementary_buckets": control_buckets,
            "priority": priority,
            "tags_in_point": location_tags,
        },
        "enabled": True,
    }
    if global_experiment_name in action_revisions:
        global_experiment["action_revision"] = action_revisions[global_experiment_name]

    test_name = "{}-test".format(name)
    test = {
        "action_id": test_name,
        "action_description": "Experiment {} – test".format(name),
        "action_type": action_type,
        "action_meta": {
            "offers": offers,
            "priority": priority,
            "tags_in_point": location_tags,
        },
        "enabled": True,
    }
    if test_name in action_revisions:
        test["action_revision"] = action_revisions[test_name]

    if disable_correctors:
        experiment["action_meta"]["correctors"] = disable_correctors
        test["action_meta"]["correctors"] = disable_correctors
    if discount:
        discount_description = {
            "time_shift": 10800,
            "segments": [
                {
                    "start": 0,
                    "value": discount
                }
            ]
        }
        experiment["action_meta"]["discount_timetable"] = discount_description
        test["action_meta"]["discount_timetable"] = discount_description
    if price_model:
        experiment["action_meta"]["model"] = price_model
        test["action_meta"]["model"] = price_model
    if parking_price_model:
        experiment["action_meta"]["parking_price_model"] = parking_price_model
        test["action_meta"]["parking_price_model"] = parking_price_model

    role_name = name
    role_description = "Experiment {}".format(name)
    if description:
        role_description += ": "
        role_description += description

    if args.is_reverse:
        for col in ["action_id", "action_description"]:
            experiment[col], control[col] = control[col], experiment[col]
            global_experiment[col], global_control[col] = global_control[col], global_experiment[col]

    role_snapshot = {
        "role_id": role_name,
        "role_description": role_description,
        "actions": [
            {
                "action_id": control_name,
                "role_id": role_name
            },
            {
                "action_id": experiment_name,
                "role_id": role_name
            },
            {
                "action_id": global_control_name,
                "role_id": role_name
            },
            {
                "action_id": global_experiment_name,
                "role_id": role_name
            },
        ]
    }

    client.add_action(control)
    client.add_action(experiment)
    client.add_action(global_control)
    client.add_action(global_experiment)
    client.propose_role(role_snapshot, comment="setting up experiment {}".format(name))

    if args.test:
        test_role_name = test_name
        test_role_snapshot = {
            "role_id": test_role_name,
            "role_description": "Experiment {} test role".format(name),
            "actions": [
                {
                    "action_id": test_name,
                    "role_id": test_role_name
                },
            ]
        }
        client.add_action(test)
        client.propose_role(test_role_snapshot, comment="setting up test role for experiment {}".format(name))


def find_conflicts(exp_text_table):
    columns = exp_text_table[0]

    exps = []
    for exp_text in exp_text_table[1:]:
        exp = dict(zip(columns, exp_text))
        if exp["offers"] != "*":
            exp["offers"] = set(exp["offers"].split('\n'))
        if exp["zones"] != "*":
            exp["zones"] = set(exp["zones"].split('\n'))

        if exp["buckets"] != "*":
            exp["all_buckets"] = set(map(int, exp["buckets"].split(',')))

        if exp["all_buckets"] != "*" and exp["comp_buckets"] != "*":
            exp["all_buckets"] |= set(map(int, exp["comp_buckets"].split(',')))

        if exp["days"] != "*":
            exp["days"] = set(map(lambda y: y[0], filter(lambda x: x[1] == "1", enumerate(exp["days"].strip("[]")))))
        if exp["from"] == "*" and exp["to"] == "*":
            exp["hours"] = "*"
        else:
            assert exp["from"] != "*" and exp["to"] != "*", "wrong time constraints"
            assert 0 <= int(exp["from"]) < int(exp["to"]) <= 2400, "wrong time constraints"
            exp["hours"] = set(range(int(exp["from"]), int(exp["to"]), 100))

        exps.append(exp)

    def get_intersection(lhs, rhs):
        intersection = {}

        for inter_col in ["offers", "zones", "all_buckets", "days", "hours"]:
            if lhs[inter_col] == "*" and rhs[inter_col] == "*":
                intersection[inter_col] = "*"
            elif lhs[inter_col] == "*" and rhs[inter_col] != "*":
                intersection[inter_col] = rhs[inter_col]
            elif lhs[inter_col] != "*" and rhs[inter_col] == "*":
                intersection[inter_col] = lhs[inter_col]
            else:
                if any(str(c).startswith("!") for c in lhs[inter_col]) or \
                        any(str(c).startswith("!") for c in rhs[inter_col]):
                    print("WARNING: {}-{} and {}-{} have negation sign in {}. "
                          "Please, check them manually".format(lhs["name"], lhs["group"],
                                                               rhs["name"], rhs["group"],
                                                               inter_col))

                intersection[inter_col] = lhs[inter_col] & rhs[inter_col]
        return intersection

    columns = ["all_buckets", "days", "hours", "offers", "zones"]

    groups_conflicts = []
    experiments_conflicts = []

    for lhs_exp, rhs_exp in combinations(exps, 2):
        intersection = get_intersection(lhs_exp, rhs_exp)

        if lhs_exp["name"] == rhs_exp["name"]:
            # find experiment/control conflicts
            conflict = False

            row = [lhs_exp["name"]]

            for col in columns:
                if lhs_exp[col] == rhs_exp[col] == intersection[col]:
                    row.append("OK")
                else:
                    lhs_val = "\n".join(map(str, sorted(list(lhs_exp[col]))))
                    rhs_val = "\n".join(map(str, sorted(list(rhs_exp[col]))))

                    if lhs_exp["group"] == "experiment":
                        lhs_val, rhs_val = rhs_val, lhs_val

                    row.append("Cont:\n{}\n\nExp:\n{}".format(lhs_val, rhs_val))
                    conflict = True

            if conflict:
                groups_conflicts.append(row)
        else:
            # find different experiments conflicts
            conflict = True

            row = [
                "{}-{}".format(lhs_exp["name"], lhs_exp["group"]),
                "{}-{}".format(rhs_exp["name"], rhs_exp["group"])
            ]

            for col in columns:
                if not intersection[col]:
                    conflict = False

            if conflict:
                for col in columns:
                    value = "\n".join(map(str, intersection[col]))
                    row.append(value)
                experiments_conflicts.append(row)

    if len(groups_conflicts):
        print("Group conflicts:")

        table = texttable.Texttable()
        table.set_cols_width([18, 15, 7, 10, 50, 42])
        table.add_rows([["name"] + columns] + groups_conflicts)

        print(table.draw())
    else:
        print("Group conflicts weren't found")

    if len(experiments_conflicts):
        print("Experiment conflicts:")

        table = texttable.Texttable()
        table.set_cols_width([22, 22, 15, 7, 10, 50, 42])
        table.add_rows([["exp 1", "exp 2"] + columns] + experiments_conflicts)

        print(table.draw())
    else:
        print("Experiment conflicts weren't found")


def main_experiment_info(client, args):
    required_location_tags = []
    required_location_tags_string = vars(args).get("location_tags")
    if required_location_tags_string:
        required_location_tags = [s for s in required_location_tags_string.split(',')]

    root = args.experiment_root_role
    roles_list = client.list_roles()
    roles = {}
    for role in roles_list:
        id = role["role_id"]
        roles[id] = role

    experiments_info_table = [["name", "group", "buckets", "comp_buckets", "days", "from", "to", "offers", "zones"]]

    occupied = []
    q = []
    q.append((roles[root], None))

    while True:
        if len(q) == 0:
            break
        role, link_meta = q.pop()

        assert role

        for subrole in role["slave_roles"]:
            q.append((roles[subrole["slave_role_id"]], subrole.get("link_meta")))

        for i in role["actions"]:
            if i["action_id"].startswith("global"):
                continue

            action = i["action"]
            meta = action["action_meta"]
            location_tags = meta.get("tags_in_point", [])
            if len(required_location_tags) > 0 and len(location_tags) > 0:
                matched = False
                for tag in required_location_tags:
                    if tag in location_tags:
                        matched = True
                        break
                if not matched:
                    continue

            exp_info = [role["role_id"], action["action_id"].split('-')[-1]]
            main_buckets = meta.get("buckets")
            complementary_buckets = meta.get("complementary_buckets")
            for buckets in [main_buckets, complementary_buckets]:
                if buckets:
                    exp_info.append(",".join(map(str, buckets)))
                    for bucket in buckets:
                        occupied.append(bucket)
                else:
                    exp_info.append("*")

            if link_meta and "time_restriction" in link_meta:
                tr = link_meta["time_restriction"]
                day_mask = "[" + ''.join(reversed('{0:07b}'.format(tr["day"]))) + "]"
                exp_info.extend([day_mask, tr["time_from"], tr["time_to"]])
            else:
                exp_info.extend(["*"] * 3)

            exp_info.append('\n'.join(meta.get("offers", "*")))
            exp_info.append('\n'.join(meta.get("tags_in_point", "*")))
            experiments_info_table.append(exp_info)

    table = texttable.Texttable()
    table.set_cols_width([18, 10, 15, 15, 9, 4, 4, 50, 42])
    table.add_rows(experiments_info_table)
    print(table.draw())

    find_conflicts(experiments_info_table)

    free = []
    for i in range(60):
        if i not in occupied:
            free.append(str(i))
    print("Free minutes: {}".format(','.join(free)))


def main_DRIVEBACK_742(client, args):
    action_type = args.action_type
    action_name = vars(args).get("action_name")
    short_name = vars(args).get("short_name")
    subname = vars(args).get("subname")
    apply = args.apply

    actions = client.list_actions()
    patched = []
    for action in actions:
        if action.get("action_type") == action_type:
            id = action["action_id"]
            meta = action["action_meta"]
            name = action["action_description"]
            changed = False
            if action_name:
                if action_name not in name:
                    print("skip {} due to name".format(id))
                    continue
            if short_name:
                action_short_name = meta.get("short_name")
                if action_short_name and len(action_short_name) > 0:
                    print("skip {} due to short_name".format(id))
                else:
                    meta["short_name"] = short_name
                    changed = True
            if subname:
                action_subname = meta.get("subname")
                if action_subname and len(action_subname) > 0:
                    print("skip {} due to subname".format(id))
                else:
                    meta["subname"] = subname
                    changed = True
            if changed:
                patched.append(action)

    print("affected actions:")
    for action in patched:
        id = action["action_id"]
        if apply:
            if _ask_confirmation("apply to {}".format(id)):
                client.add_action(action)
        else:
            print(id)


def main_DRIVEBACK_780(client, args):
    client2 = api.BackendClient(endpoint=args.endpoint2, public_token=args.public_token, private_token=args.private_token)
    objects = client.list_cars()
    for obj in objects:
        if obj.status == "service":
            logging.info("skipping {} in service".format(obj.id))
            continue

        logging.info("analyzing {}".format(obj.id))
        try:
            tracks = client.analyzer(object_id=obj.id)
            tracks2 = client2.analyzer(object_id=obj.id)
        except Exception as e:
            logging.error("an exception occurred for {}: {}".format(obj.id, e))
            continue

        rides = [i["ride"] for i in tracks]
        rides2 = [i["ride"] for i in tracks2]
        for ride in rides:
            if ride not in rides2:
                print("segment {} is not in rides2".format(ride))
        for ride in rides2:
            if ride not in rides:
                print("segment {} is not in rides".format(ride))


def main_DRIVEBACK_801(client, args):
    client2 = api.BackendClient(endpoint=args.destination_endpoint, public_token=args.public_token, private_token=args.private_token)
    descriptions = client.list_tag_descriptions()
    for description in descriptions:
        name = description["name"]
        if not name.startswith("ugc_") and not name.startswith("feedback_"):
            continue
        logging.info("transferring {}".format(name))
        client2.add_tag_description(description)


def main_DRIVEBACK_4095(client, args):
    apply = args.apply
    for line in sys.stdin:
        description = json.loads(line)
        object_id = description["user_id"]
        rank = description["rank"]
        score = description["score"]
        timestamp = description["timestamp"]

        object = apiobjects.Object(id=object_id)
        tag = apiobjects.Tag(name="scoring_car_tag", object_id=object_id, data={
            "rank": rank,
            "score": score,
            "timestamp": timestamp
        })

        skip = False
        tags = client.list_tags(object)
        for existing in tags:
            if existing.name != tag.name:
                continue
            existing_timestamp = existing.data["timestamp"]
            if existing_timestamp == timestamp:
                logging.info("skipping {} due to existing tag {}".format(object_id, existing.tag_id))
                skip = True
                break

        if skip:
            continue

        if apply:
            client.add_tag(tag)
        else:
            print(json.dumps(tag.serialize()))


def main_DRIVEMATICSDEV_752(client, args):
    apply = args.apply
    tag_descriptions = client.list_tag_descriptions()
    selected_tag_names = [
        "car_1h_dont_move",
        "car_in_deny_zone",
        "leasing_car_is_blocked",
        "overmileage",
        "telematics_tag_alerting",
    ]
    for tag_description in tag_descriptions:
        name = tag_description["name"]
        type = tag_description["type"]
        if not name.startswith("signal_") and name not in selected_tag_names:
            continue

        if type == "scoring_trace_tag":
            signal_enabled = tag_description["meta"]["signal_enabled"]
            if not signal_enabled:
                if apply:
                    tag_description["meta"]["signal_enabled"] = True
                    print("updating scoring_trace_tag tag {}".format(name))
                    client.add_tag_description(tag_description)
                    print("updated scoring_trace_tag tag {}".format(name))
                else:
                    print("updatable scoring_trace_tag tag {}".format(name))
        # if type == "generic_signal_car_tag":
        if type == "simple_car_tag":
            if apply:
                # tag_description["type"] = "simple_car_tag"
                tag_description["type"] = "generic_signal_car_tag"
                print("updating simple_car_tag tag {}".format(name))
                client.add_tag_description(tag_description, force=True)
                print("updated simple_car_tag tag {}".format(name))
            else:
                print("updatable simple_car_tag tag {}".format(name))


def register(group):
    parser = group.add_parser('client', description="Drive Client")
    parser.add_argument('-y', "--public-token", dest="public_token", metavar="OAUTH", type=str, default=None, help="Yandex.Passport OAuth token")
    parser.add_argument('-s', "--private-token", dest="private_token", metavar="OATH", type=str, default=None, help="Internal Passport OAuth token")
    parser.add_argument('-v', "--verbose", dest="verbose", action="store_true", help="enable debug level logging")
    parser.add_argument("--device-id", dest="device_id", metavar="DID", type=str)
    parser.add_argument("--endpoint", dest="endpoint", metavar="URL", type=str, default="http://prestable.carsharing.yandex.net", help="Drive.Server host")
    parser.add_argument("--force", dest="force", action="store_true", help="do not ask for confirmations")
    parser.add_argument("--authorization-header", dest="authorization_header", metavar="header", type=str, default=None, help="authorization host")

    subparsers = parser.add_subparsers(help="modes")
    analyzer_parser = subparsers.add_parser("analyzer", help="analyzer user tracks")
    analyzer_parser.add_argument("--session", dest="analyzer_session_id", metavar="ID", type=str)
    analyzer_parser.add_argument("--ride", dest="analyzer_ride_id", metavar="ID", type=str)

    firmware_parser = subparsers.add_parser("firmware", help="manage telematics firmware")
    firmware_parser.add_argument("--action", dest="firmware_action", metavar="ACTION", type=str, default="add")
    firmware_parser.add_argument("--file", dest="file", metavar="PATH", type=str)
    firmware_parser.add_argument("--model", dest="model", metavar="CODE", type=str)

    clone_action_parser = subparsers.add_parser("clone_action", help="clone and patch action(s)")
    clone_action_parser.add_argument("--suffix", dest="suffix", metavar="SUFFIX", type=str)
    clone_action_parser.add_argument("--type", dest="action_type", metavar="TYPE", type=str)
    clone_action_parser.add_argument("--location-tags", dest="location_tags", metavar="TAG,TAG,..", type=str)
    clone_action_parser.add_argument("--tags", dest="tags", metavar="TAG,TAG,..", type=str)
    clone_action_parser.add_argument("--patch", dest="patch", metavar="JSON", type=str)
    clone_action_parser.add_argument('-y', dest="apply", action="store_true", help="apply changes")

    command_parser = subparsers.add_parser("command", help="run telematics command")
    command_parser = _add_filters(command_parser)
    command_parser.add_argument("--threads", dest="threads", metavar="NUM", type=int, default=4)
    command_parser.add_argument("--command", dest="command", metavar="NAME", type=str)
    command_parser.add_argument("--sensor-id", dest="sensor_id", metavar="NUM", type=int)
    command_parser.add_argument("--sensor-subid", dest="sensor_subid", metavar="NUM", type=int)

    rtbg_parser = subparsers.add_parser("rtbg", help="control rt_background processes")
    rtbg_parser.set_defaults(func=rtbg.execute)
    rtbg.fill_parser(rtbg_parser)

    scenario_parser = subparsers.add_parser("scenario", help="run scenario")
    scenario_parser = _add_filters(scenario_parser)
    scenario_parser.add_argument("--name", dest="scenario", metavar="NAME", type=str)
    scenario_parser.add_argument("--argument", dest="argument", metavar="STR", type=str)
    scenario_parser.add_argument("--threads", dest="threads", metavar="NUM", type=int, default=4)

    search_parser = subparsers.add_parser("search", help="search cars by query")
    search_parser.add_argument('-q', "--query", dest="search_query", metavar="TERM", type=str, help="search query")
    search_parser.add_argument('-f', "--fields", dest="search_fields", metavar="FIELD", type=str, help="object fields to output", default="id")

    experiment_add_parser = subparsers.add_parser("experiment_add", help="add an experiment")
    experiment_add_parser.add_argument('-n', "--name", dest="experiment_name", metavar="DRIVEANALYTICS-XXX", type=str)
    experiment_add_parser.add_argument("--description", dest="description", metavar="TEXT", type=str)
    experiment_add_parser.add_argument("--disable-correctors", dest="disable_correctors", metavar="ACTION,ACTION,...,ACTION", type=str)
    experiment_add_parser.add_argument("--discount", dest="discount", metavar="INT[-100; 100]", type=int)
    experiment_add_parser.add_argument("--price-model", dest="price_model", metavar="NAME", type=str)
    experiment_add_parser.add_argument("--parking-price-model", dest="parking_price_model", metavar="NAME", type=str)
    experiment_add_parser.add_argument("--control", dest="control", metavar="BUCKET,BUCKET,...,BUCKET", type=str)
    experiment_add_parser.add_argument("--experiment", dest="experiment", metavar="BUCKET,BUCKET,...,BUCKET", type=str)
    experiment_add_parser.add_argument("--location-tags", dest="location_tags", metavar="TAG,TAG,...,TAG", type=str)
    experiment_add_parser.add_argument("--offers", dest="offers", metavar="OFFER,OFFER,...,OFFER", type=str)
    experiment_add_parser.add_argument("--priority", dest="priority", metavar="INT[0; 1000]", type=int)
    experiment_add_parser.add_argument("--reverse", dest="is_reverse", action="store_true",
                                       help="switch experiment and control groups for reverse experiment")
    experiment_add_parser.add_argument("--test", dest="test", action="store_true", help="create test action")

    experiment_info_parser = subparsers.add_parser("experiment_info", help="get info about experiments")
    experiment_info_parser.add_argument('-r', "--experiment-root-role", dest="experiment_root_role", metavar="ROLE", type=str, default="experiments_production")
    experiment_info_parser.add_argument("--location-tags", dest="location_tags", metavar="TAG,TAG,...,TAG", type=str)

    world_parser = subparsers.add_parser("world", help="create new worlds")
    world_parser.set_defaults(func=world.execute)
    world.fill_parser(world_parser)

    cleanup_parser = subparsers.add_parser("cleanup", help="cleanup metadata")
    cleanup.fill_parser(cleanup_parser)

    compare_parser = subparsers.add_parser("compare", help="compare entities")
    compare.fill_parser(compare_parser)

    model_parser = subparsers.add_parser("model", help="process ranking models")
    model_parser.set_defaults(func=model.execute)
    model.fill_parser(model_parser)

    taxicorp_parser = subparsers.add_parser("taxicorp", help="taxicopr maintainance utilities")
    taxicorp_parser.set_defaults(func=taxicorp.execute)
    taxicorp.fill_parser(taxicorp_parser)

    transfer_parser = subparsers.add_parser("transfer", help="transfer metadata between endpoints")
    transfer_parser.set_defaults(func=transfer.execute)
    transfer.fill_parser(transfer_parser)

    user_parser = subparsers.add_parser("user", help="process users")
    user.fill_parser(user_parser)

    DRIVEBACK_742_parser = subparsers.add_parser("DRIVEBACK-742", help="helper for DRIVEBACK-742")
    DRIVEBACK_742_parser.add_argument("--action-type", dest="action_type", metavar="TYPE", type=str)
    DRIVEBACK_742_parser.add_argument("--action-name", dest="action_name", metavar="TYPE", type=str)
    DRIVEBACK_742_parser.add_argument("--short-name", dest="short_name", metavar="TYPE", type=str)
    DRIVEBACK_742_parser.add_argument("--subname", dest="subname", metavar="TYPE", type=str)
    DRIVEBACK_742_parser.add_argument("--apply", dest="apply", action="store_true")

    DRIVEBACK_780_parser = subparsers.add_parser("DRIVEBACK-780", help="helper for DRIVEBACK-780")
    DRIVEBACK_780_parser.add_argument("--endpoint2", dest="endpoint2", metavar="URL", default="http://admin.carsharing.yandex.net", help="Drive.Server host")

    DRIVEBACK_780_parser = subparsers.add_parser("DRIVEBACK-801", help="helper for DRIVEBACK-801")
    DRIVEBACK_780_parser.add_argument("--destination-endpoint", dest="destination_endpoint", metavar="URL", default="http://prestable.carsharing.yandex.net", help="Drive.Server host")

    DRIVEBACK_4095_parser = subparsers.add_parser("DRIVEBACK-4095", help="loader for DRIVEBACK-4095")
    DRIVEBACK_4095_parser.add_argument("--apply", dest="apply", action="store_true")
    DRIVEBACK_4095_parser.set_defaults(func=main_DRIVEBACK_4095)

    DRIVEMATICSDEV_752_parser = subparsers.add_parser("DRIVEMATICSDEV-752", help="script for DRIVEMATICSDEV-752")
    DRIVEMATICSDEV_752_parser.add_argument("--apply", dest="apply", action="store_true")
    DRIVEMATICSDEV_752_parser.set_defaults(func=main_DRIVEMATICSDEV_752)

    parser.set_defaults(main=main_client)


def main_client(args):
    loggingLevel = logging.DEBUG if args.verbose else logging.INFO
    logging.basicConfig(
        format="%(asctime)s [%(levelname)s] %(module)s: %(message)s",
        level=loggingLevel
    )

    try:
        client = api.BackendClient(endpoint=args.endpoint, public_token=args.public_token, private_token=args.private_token, authorization_header=args.authorization_header)
        if "device_id" in args:
            client.set_device_id(args.device_id)
        if "analyzer_session_id" in args or "analyzer_ride_id" in args:
            session_id = vars(args).get("analyzer_session_id")
            ride_id = vars(args).get("analyzer_ride_id")
            main_analyzer(client, session_id, ride_id)
            return
        if "firmware_action" in args:
            main_firmware(client, args)
            return
        if "action_type" in args:
            main_clone_action(client, args)
            return
        if "command" in args:
            main_command(client, args)
            return
        if "scenario" in args:
            main_scenario(client, args)
            return
        if "search_query" in args:
            main_search(client, args.search_query, args.search_fields.split(','))
            return
        if "action_type" in args:
            main_DRIVEBACK_742(client, args)
            return
        if "endpoint2" in args:
            main_DRIVEBACK_780(client, args)
            return
        if "destination_endpoint" in args:
            main_DRIVEBACK_801(client, args)
            return
        if "experiment_name" in args:
            main_experiment_add(client, args)
            return
        if "experiment_root_role" in args:
            main_experiment_info(client, args)
            return
        if hasattr(args, 'func'):
            args.func(client, args)
        else:
            logging.error("cannot determine client mode")
    except Exception as e:
        logging.info("current arguments: {}".format(args))
        logging.exception("an exception has occurred:\n{}".format(e))
