import logging
import json


MODEL_FIELDS = [
    "acceptance_model",
    "duration_model",
    "mileage_limit_model",
    "model",
    "overrun_price_model",
    "pack_price_model",
    "parking_price_model",
    "route_duration_model",
]


def add(client, args):
    meta = json.reads(args.model_add_meta)
    data = open(args.model_add_data).read()
    force = args.model_add_force

    name = meta["name"]
    existing = client.list_ranking_models()
    if name in existing and not force:
        logging.error("model {} already exists".format(name))
        return
    client.add_ranking_model(meta=meta, bin=data)


def cleanup(client, args):
    apply = args.model_cleanup_apply
    exceptions = args.model_cleanup_exceptions
    used_models = set()
    actions = client.list_actions()
    for action in actions:
        meta = action["action_meta"]
        for field in MODEL_FIELDS:
            model = meta.get(field)
            if model:
                used_models.add(model)

    all_models = client.list_ranking_models()
    unused_models = []
    for model in all_models:
        if model not in used_models:
            unused_models.append(model)

    if apply:
        for model in unused_models:
            if exceptions and model in exceptions:
                print("skip model: {}".format(model))
                continue
            print("removing model: {}".format(model))
            client.remove_ranking_model(model)
    else:
        print("unused models:")
        for model in unused_models:
            print(model)


def fill_parser(parser):
    subparsers = parser.add_subparsers(help="actions")
    add_parser = subparsers.add_parser("add", help="add a model")
    add_parser.add_argument('-m', "--meta", dest="model_add_meta", help="model metadata")
    add_parser.add_argument('-d', "--data", dest="model_add_data", help="model datafile")
    add_parser.add_argument('-f', "--force", dest="model_add_force", help="force addition")

    cleanup_parser = subparsers.add_parser("cleanup", help="remove unused models")
    cleanup_parser.add_argument('-e', dest="model_cleanup_exceptions", action="append", help="do not touch these models")
    cleanup_parser.add_argument('-y', dest="model_cleanup_apply", action="store_true", help="do remove models")


def execute(client, args):
    if "model_add_meta" in args:
        add(client, args)
        return
    if "model_cleanup_apply" in args:
        cleanup(client, args)
        return
    logging.error('cannot determine action')
