import json
import logging
import os
import random

from .firmware import create_alias, deduce_device, deduce_models

from drive.backend.api import objects as apiobjects


PARKING_PAYMENT_TAG = "parking_payment"


def scenario_add_tag(client, obj, argument):
    assert argument
    data = None
    if os.path.exists(argument):
        file = open(argument)
        data = file.read()
    else:
        data = argument
    j = json.loads(data)
    tag = apiobjects.Tag.from_json(j)
    tag.object_id = obj.id
    client.add_tag(tag)


def scenario_remove_tag(client, obj, argument):
    assert argument
    tag_name = argument
    tags = client.list_tags(obj)
    for tag in tags:
        if tag and tag.name == tag_name:
            client.remove_tag(tag.tag_id)
            print("removed {} {} from {}".format(tag.name, tag.tag_id, obj.id))


def scenario_detach_vega(client, obj, argument):
    attachments = client.list_car_attachments(obj.id)
    for attachment in attachments:
        id = attachment["id"]
        typ = attachment["cpp_type"]
        if typ == "car_hardware_vega":
            assert not obj.status.startswith("old_state_")
            assert not obj.status.startswith("service")
            assert obj.status not in ("available", "cleaning", "fueling" "new", "transformation")
            print("removing attachment {} from {}: {}".format(id, obj.id, obj.imei))
            client.remove_car_attachment(id)


def scenario_charge_battery(client, obj, argument):
    critical_voltage_threshold = 11.9

    voltage = client.get_parameter(obj, 1252)
    if voltage > critical_voltage_threshold:
        return

    print("{}: voltage {} below threshold".format(obj.id, voltage))
    warming_command = apiobjects.TelematicsCommand("YADRIVE_WARMING")
    warming = client.command(obj, warming_command)
    if warming.succeeded():
        print("{}: successfully started warming".format(obj.id))
    else:
        print("{}: cannot start warming – {}".format(obj.id, warming.get_error()))


def scenario_register_parking_payment(client, obj, argument):
    groups = {}
    tokens = {}
    tokens_filepath = argument or "./parking_tokens.txt"
    logging.debug("looking for tokens file {}".format(tokens_filepath))
    if os.path.exists(tokens_filepath):
        with open(tokens_filepath) as tokens_file:
            lines = tokens_file.readlines()
            logging.info("read {} tokens from tokens file".format(len(lines)))
            for line in lines:
                splitted = line.split('\t')
                token = splitted[0].strip()
                group = splitted[1].strip()
                if not token:
                    continue
                tokens[token] = 0
                groups[token] = group

    parked_objects = client.list_cars(tags_filter=[PARKING_PAYMENT_TAG])
    processed = 0
    for parked_object in parked_objects:
        if parked_object.id == obj.id:
            print("{} is already registered".format(obj.id))
            return
        if processed % 50 == 0:
            print("processed {} parked objects".format(processed))

        tags = client.list_tags(parked_object)
        for tag in tags:
            if tag.name == PARKING_PAYMENT_TAG:
                for slot in tag.data["slots"]:
                    token = slot["token"]
                    tokens[token] = (tokens.get(token) or 0) + 1
        processed += 1
    print("processed {} parked objects".format(processed))

    ordered_tokens = sorted(tokens, key=tokens.get)
    selected_tokens = []
    selected_groups = []
    for token in ordered_tokens:
        group = groups.get(token)
        if group and group in selected_groups:
            continue
        print("selected token {} {}".format(token, tokens.get(token)))
        selected_tokens.append(token)
        if group:
            selected_groups.append(group)
        if len(selected_tokens) > 3:
            break

    tag_data = {
        "slots": []
    }
    for token in selected_tokens:
        slot = {
            "token": token
        }
        tag_data["slots"].append(slot)
    tag = apiobjects.Tag(PARKING_PAYMENT_TAG, obj.id, data=tag_data)

    client.add_tag(tag)
    print("successfully registered {} using {}".format(obj.id, ','.join(selected_tokens)))


def scenario_deregister_parking_payment(client, obj, argument):
    scenario_remove_tag(client, obj, PARKING_PAYMENT_TAG)
    print("successfully deregistered {}".format(obj.id))


def scenario_randomize_location(client, obj, argument):
    assert argument
    arg = json.loads(argument)
    random.seed(obj.id)
    lat = random.uniform(arg[0][1], arg[1][1])
    lon = random.uniform(arg[0][0], arg[1][0])
    set_lat = apiobjects.TelematicsCommand("SET_PARAM", 101, 0, lat)
    set_lon = apiobjects.TelematicsCommand("SET_PARAM", 102, 0, lon)
    client.command(obj, set_lat)
    client.command(obj, set_lon)


def scenario_show_objects(client, obj, argument):
    print(json.dumps(obj.serialize()))


def scenario_show_firmware(client, obj, argument):
    state = client.get_telematics_state(obj)
    firmware = state.get("mcu_firmware") or "unknown"
    print("{}\t{}".format(obj.imei, firmware))


def scenario_show_mnc(client, obj, argument):
    state = client.get_telematics_state(obj)
    mnc = state.get("mnc") or "unknown"
    print("{}\t{}".format(obj.imei, mnc))


def scenario_show_password(client, obj, argument):
    password = client.get_telematics_password(obj)
    print("{}\t{}\t{}".format(obj.id, obj.imei, password))


def scenario_update_firmware(client, obj, argument):
    target = argument
    target_device = deduce_device(target)
    target_models = deduce_models(target)
    if obj.model not in target_models:
        raise Exception("mismatched models: {} is not in {}".format(obj.model, target_models))

    state = client.get_telematics_state(obj)
    current = state.get("mcu_firmware") or "unknown"
    current_device = deduce_device(current)
    if current_device != target_device:
        logging.info("skipping {} due to device mismatch: {} != {}".format(obj.id, current, target))
        return

    alias = create_alias(target, obj.model)
    registered_firmware = client.list_firmware()
    registered_aliases = [x["alias"] for x in registered_firmware]
    if alias not in registered_aliases:
        raise Exception("firmware alias {} is unknown".format(alias))

    tag = apiobjects.Tag(name="telematics_firmware", object_id=obj.id, data={
        "name": alias
    })
    client.add_tag(tag)
    print(json.dumps(tag.serialize()))


def scenario_DRIVESUP_1852(client, obj, argument):
    state = client.get_telematics_state(obj)
    mileage = state.get("mileage")
    firmware = state.get("mcu_firmware") or "unknown"
    if not mileage:
        return
    if (mileage / 256).is_integer():
        print("{}\t{}\t{}\t{}".format(obj.id, obj.number, obj.imei, firmware))


def scenario_DRIVEBACK_1634(client, obj, argument):
    obj.osago_mds_key = "OSAGO/XW8ZZZ61ZJG063131/majorAPI/1563557390.pdf"
    obj.registration_mds_key = "STS/XW8ZZZ61ZJG063131/1550050343.pdf"
    client.edit_car(obj)


def scenario_DRIVESUP_reboot_sim_1(client, obj, argument):
    if not obj.imei:
        return
    state = client.get_telematics_state(obj)
    mnc = state.get("mnc")
    if mnc != 99:
        return
    cmd = apiobjects.TelematicsCommand("SCENARIO_POLITE_RESTART")
    client.command(obj, cmd)
    print("{} success".format(obj.id))


def scenario_emergency_lights(client, obj, argument):
    if not obj.imei:
        return

    state = client.get_parameter(obj, 1176)
    print("{}: emergency lights state {}".format(obj.id, state))


def scenario_DRIVE_tag_free_parking(client, obj, argument):
    location = obj.location
    assert location
    parkings = client.list_parkings(location.lat, location.lon)
    assert parkings is not None
    report = {
        "object_id": obj.id,
        "parkings": parkings,
    }
    print(json.dumps(report))
    if len(parkings) == 0 and argument:
        scenario_add_tag(client, obj, argument)


def scenario_DRIVEBACK_1607(client, obj, argument):
    fuel_level = obj.fuel_level or 0
    if fuel_level < 1:
        print("warming up {}".format(obj.id))
        warming_command = apiobjects.TelematicsCommand("YADRIVE_WARMING")
        warming = client.command(obj, warming_command)
        if warming.succeeded():
            print("{}: successfully started warming".format(obj.id))
        else:
            print("{}: cannot start warming – {}".format(obj.id, warming.get_error()))


def scenario_DRIVEBACK_1738(client, obj, argument):
    state = client.get_telematics_state(obj)
    session_key = state.get("ble_session_key")
    if session_key:
        session_key_data = bytes.fromhex(session_key)
        for b in session_key_data:
            if int(b) > 0:
                return
    print("{} has session_key {}".format(obj.id, session_key))
    ble_reset_command = apiobjects.TelematicsCommand("SCENARIO_BLE_RESET")
    ble_reset = client.command(obj, ble_reset_command)
    if not ble_reset.succeeded():
        print("{}: cannot reset ble – {}".format(obj.id, ble_reset.get_error()))


def scenario_DRIVEBACK_2494(client, obj, argument):
    state = client.get_telematics_state(obj)
    engine_on = state.get("engine_on")
    if engine_on != 1:
        logging.debug("engine is not on for {}".format(obj))
        if obj.status != "available":
            logging.error("{} is busy: {}".format(obj.id, obj.status))
            return

        warming_command = apiobjects.TelematicsCommand("YADRIVE_WARMING")
        warming = client.command(obj, warming_command)
        if not warming.succeeded():
            logging.error("{} warming failed: {}".format(obj.id, warming.get_error()))
            return

    obd_forward_config_command = apiobjects.TelematicsCommand("OBD_FORWARD_CONFIG", raw={
        "duration": "15s",
        "value": 0x77F,
        "mask": 0x7FF,
    })
    obd_forward_config = client.command(obj, obd_forward_config_command)
    if not obd_forward_config.succeeded():
        logging.error("{} OBD forward config failed: {}".format(obj.id, obd_forward_config.get_error()))
        return

    obd_request_command = apiobjects.TelematicsCommand("SCENARIO_OBD_REQUEST", raw={
        "id": 0x715,
        "index": 1,
        "data": [0x03, 0x22, 0x23, 0x64, 0x00, 0x00, 0x00, 0x00]
    })
    obd_request = client.command(obj, obd_request_command)
    if not obd_request.succeeded():
        logging.error("{} OBD request failed: {}".format(obj.id, obd_request.get_error()))
        return

    obd_request2_command = apiobjects.TelematicsCommand("SCENARIO_OBD_REQUEST", raw={
        "id": 0x715,
        "index": 1,
        "data": [0x03, 0x22, 0x23, 0x66, 0x00, 0x00, 0x00, 0x00]
    })
    obd_request = client.command(obj, obd_request2_command)
    if not obd_request.succeeded():
        logging.error("{} OBD request 2 failed: {}".format(obj.id, obd_request.get_error()))
        return

    print("{}: success".format(obj.id))


_SCENARIOS = {
    "add_tag": scenario_add_tag,
    "remove_tag": scenario_remove_tag,
    "detach_vega": scenario_detach_vega,
    "charge_battery": scenario_charge_battery,
    "register_parking_payment": scenario_register_parking_payment,
    "deregister_parking_payment": scenario_deregister_parking_payment,
    "randomize_location": scenario_randomize_location,
    "show_objects": scenario_show_objects,
    "show_firmware": scenario_show_firmware,
    "show_mnc": scenario_show_mnc,
    "show_password": scenario_show_password,
    "update_firmware": scenario_update_firmware,
    "DRIVESUP-1852": scenario_DRIVESUP_1852,
    "DRIVESUP_reboot_sim_1": scenario_DRIVESUP_reboot_sim_1,
    "DRIVE-tag_free_parking": scenario_DRIVE_tag_free_parking,
    "DRIVEBACK-1607": scenario_DRIVEBACK_1607,
    "DRIVEBACK-1634": scenario_DRIVEBACK_1634,
    "DRIVEBACK-1738": scenario_DRIVEBACK_1738,
    "DRIVEBACK-2494": scenario_DRIVEBACK_2494,
    "emergency_lights": scenario_emergency_lights,
}


def get_scenario(name):
    result = _SCENARIOS.get(name)
    if result:
        return result
    else:
        raise NotImplementedError("unknown scenario {}".format(name))
