import logging

from drive.backend.api import client as api


class ActionAccessor:
    description = "action"
    id_key = "action_id"
    type_key = "action_type"

    @staticmethod
    def add(client, entity):
        return client.add_action(entity)

    @staticmethod
    def get_all(client):
        return client.list_actions()


class AreaAccessor:
    description = "area"
    id_key = "area_id"
    type_key = "area_type"

    @staticmethod
    def add(client, entity):
        return client.add_area(entity)

    @staticmethod
    def get_all(client):
        return client.list_areas()


class RoleAccessor:
    description = "role"
    id_key = "role_id"
    type_key = "role_type"

    @staticmethod
    def add(client, entity):
        return client.add_role(entity)

    @staticmethod
    def get_all(client):
        return client.list_roles()


class SettingAccessor:
    description = "setting"
    id_key = "setting_key"
    type_key = "setting_type"

    @staticmethod
    def add(client, entity):
        return client.add_setting(entity)

    @staticmethod
    def get_all(client):
        return client.list_settings()


def transfer(client, source, accessor, name, rename=None, force=False):
    rename = rename or name
    logging.info("transferring {} to {}".format(name, rename))

    source_entities = dict()
    for entity in accessor.get_all(source):
        entity_id = entity[accessor.id_key]
        source_entities[entity_id] = entity

    destination_entities = dict()
    for entity in accessor.get_all(client):
        entity_id = entity[accessor.id_key]
        destination_entities[entity_id] = entity

    assert name in source_entities, "{} is not found in source endpoint".format(name)
    assert force or rename not in destination_entities, "{} is found in destination endpoint and force is not specified".format(rename)

    entity = source_entities.get(name)
    assert entity
    logging.info("entity {} has type {}".format(name, entity.get(accessor.type_key, None)))

    entity[accessor.id_key] = rename
    existing = destination_entities.get(rename)
    if existing:
        entity["action_revision"] = existing["action_revision"]
    accessor.add(client, entity)


def fill_parser(parser):
    parser.add_argument("--action", dest="action", metavar="NAME", help="action name")
    parser.add_argument("--area", dest="area", metavar="NAME", help="area id")
    parser.add_argument("--role", dest="role", metavar="NAME", help="role name")
    parser.add_argument("--setting", dest="setting", metavar="NAME", help="setting key")

    parser.add_argument("--force", dest="force", action="store_true", help="overwrite")
    parser.add_argument("--rename", dest="rename", metavar="NAME", help="entity new name")
    parser.add_argument("--source", dest="source_endpoint", metavar="URL", default="http://testing.carsharing.yandex.net", help="source Drive.Server host")


def execute(client, args):
    force = args.force
    rename = vars(args).get("rename", None)
    source = api.BackendClient(endpoint=args.source_endpoint, public_token=args.public_token, private_token=args.private_token)
    if args.action:
        name = args.action
        transfer(client, source, ActionAccessor, name, rename, force)
        return
    if args.area:
        name = args.area
        transfer(client, source, AreaAccessor, name, rename, force)
        return
    if args.role:
        name = args.role
        transfer(client, source, RoleAccessor, name, rename, force)
        return
    if args.setting:
        name = args.setting
        transfer(client, source, SettingAccessor, name, rename, force)
        return
