import json
import logging


def process_users(func):
    def impl(client, args):
        users = client.search_users_by_tags(
            all_of=args.all_of,
            any_of=args.any_of,
            none_of=args.none_of,
            limit=args.limit
        )
        for user in users:
            func(client, args, user)
    return impl


@process_users
def list_users(client, args, user):
    user_id = user['id']
    print(user_id)


@process_users
def get_tags(client, args, user):
    user_id = user['id']
    try:
        user_tags = client.get_user_tags(user_id)
        for user_tag in user_tags:
            assert user_tag.object_id == user_id
            if user_tag.name not in args.tags:
                continue
            if user_tag.name == 'taxi_fleet_auth_info':
                apikey = None
                client_id = None
                park_id = None
                profile_id = None
                for field in user_tag.data['fields']:
                    key = field['key']
                    value = field['value']
                    if key == 'apikey':
                        apikey = value
                    if key == 'client_id':
                        client_id = value
                    if key == 'park_id':
                        park_id = value
                    if key == 'profile_id':
                        profile_id = value
                serialized = {
                    'user_id': user_id,
                    'tag_id': user_tag.tag_id,
                    'apikey': apikey,
                    'client_id': client_id,
                    'park_id': park_id,
                    'profile_id': profile_id
                }
            else:
                serialized = user_tag.serialize()
            print(json.dumps(serialized))
    except Exception as ex:
        print('cannot process {}: {}'.format(user_id, ex))


@process_users
def remove_tags(client, args, user):
    user_id = user['id']
    try:
        user_tags = client.get_user_tags(user_id)
        for user_tag in user_tags:
            assert user_tag.object_id == user_id
            if user_tag.name not in args.tags:
                continue
            if args.apply:
                logging.info('removing {} {} from {}'.format(user_tag.name, user_tag.tag_id, user_id))
                client.remove_user_tag(user_tag.tag_id)
            else:
                print('{} {} at {}'.format(user_tag.name, user_tag.tag_id, user_id))
    except Exception as ex:
        print('cannot process {}: {}'.format(user_id, ex))


def fill_parser(parser):
    parser.add_argument('--all-of', dest='all_of', help='all of tags', action='append')
    parser.add_argument('--any-of', dest='any_of', help='any of tags', action='append')
    parser.add_argument('--none-of', dest='none_of', help='none of tags', action='append')
    parser.add_argument('--limit', dest='limit', help='user count limit', type=int)
    parser.add_argument('-y', dest='apply', help='apply', action='store_true')
    subparsers = parser.add_subparsers(help='entity')

    list_parser = subparsers.add_parser('list', help='list users')
    list_parser.set_defaults(func=list_users)

    get_tag = subparsers.add_parser('get-tag', help='get user tag')
    get_tag.add_argument('--tag', dest='tags', help='tag name', action='append')
    get_tag.set_defaults(func=get_tags)

    remove_tag = subparsers.add_parser('remove-tag', help='remove user tag')
    remove_tag.add_argument('--tag', dest='tags', help='tag name', action='append')
    remove_tag.set_defaults(func=remove_tags)
