import json
import logging
import psycopg2
import time
import traceback

import library.python.resource as resource

from drive.backend.api import client as api


def acquire_root(client):
    logging.info('acquiring root permissions')
    client.get_root()
    permissions = None
    for _ in range(10):
        permissions = client.get_permissions()
        actions = permissions['actions']
        if '__root_administrative' in actions and '__root_tag_action' in actions:
            break
        else:
            time.sleep(10)
    logging.debug('acquired permissions: {}'.format(json.dumps(permissions)))


def clone_cars(client, source, args):
    skipped_tags = [
        'incorrect_moving',
        'incorrect_v',
        'telematics_firmware',
        'user_fueling_tag',
    ]
    clone_percent = args.cars_percentage
    endpoint_suffix = str(hash(args.cars_random_seed) % 100)

    existing_cars = client.list_cars() or []
    existing_numbers = []
    for car in existing_cars:
        existing_numbers.append(car.number)

    cars = source.list_cars()
    for car in cars:
        if car.number in existing_numbers:
            logging.debug('skipping existing {}'.format(car.number))
            continue
        if hash(car.id + args.cars_random_seed) % 100 > clone_percent:
            logging.debug('skipping {} by clone percent'.format(car.id))
            continue
        if len(car.vin) > 6:
            car.vin = 'FAKE' + endpoint_suffix + car.vin[6:]
        else:
            logging.warning('skipping {} by short vin {}'.format(car.id, car.vin))
            continue

        if len(car.imei) > 6:
            car.imei = '7777' + endpoint_suffix + car.imei[6:]
        else:
            logging.warning('skipping {} by short IMEI {}'.format(car.id, car.imei))
            continue

        if len(car.number) > 1:
            car.number = 'f' + car.number[1:]
        else:
            logging.warning('skipping {} by short number {}'.format(car.id, car.number))
            continue

        attachments = source.list_car_attachments(car.id)
        for attachment in attachments:
            if attachment.get('cpp_type') == 'car_registry_document':
                car.osago_mds_key = attachment.get('data', {}).get('osago_mds_key')
                car.registration_mds_key = attachment.get('data', {}).get('registration_mds_key')

        logging.info('transferring car {}'.format(car.number))
        client.edit_car(car, force=True)
        tags = source.list_tags(car)
        for tag in tags:
            if ' ' in tag.name:
                logging.warning('skipping tag {} from {}'.format(tag.name, car.id))
                tags.remove(tag)
            if tag.name in skipped_tags:
                tags.remove(tag)
            if tag.name.startswith('old_state_') or tag.name == 'transformation':
                tag.name = 'old_state_reservation'
        client.add_tags(car, tags)


def base_checking_tag_description(tag_description, tag_types):
    name = tag_description['name']
    comment = tag_description['comment']
    type_ = tag_description['type']
    if ' ' in name:
        logging.warning('skipping tag description {} due to space in the name'.format(name))
        return False
    if 'base object' == comment:
        logging.info('skipping base object {}'.format(name))
        return False
    if type_ not in tag_types:
        logging.error('skipping tag description {} due to unknown type {}'.format(name, type_))
        return False
    return True


def clone_metadata(client, source):
    def transfer(lister, adder, name_field, comment, force=False, fltr=None, modifier=None):
        logging.info('started transferring of {}'.format(comment))
        existing_objects = lister(client) or []
        source_objects = lister(source) or []

        existing_names = set()
        for obj in existing_objects:
            name = obj[name_field]
            existing_names.add(name)

        for obj in source_objects:
            name = obj[name_field]
            if name in existing_names and not force:
                logging.info('skipping existing {} {}'.format(comment, name))
                continue
            if fltr and not fltr:
                logging.info('skipping filtered out {} {}'.format(comment, name))
                continue
            if modifier:
                modifier(obj)
            logging.info('cloning {} {}'.format(comment, name))
            logging.debug(json.dumps(obj))
            adder(client, obj)

        logging.info('finished transferring of {}'.format(comment))

    def _clear_default_tags(obj):
        obj['default_tags'] = []

    transfer(
        lambda x    : x.list_notifiers(),
        lambda x, y : x.add_notifier(y),
        'name',
        'notifier'
    )

    transfer(
        lambda x    : x.list_state_filters(),
        lambda x, y : x.add_state_filter(y),
        'state_id',
        'state filter'
    )

    transfer(
        lambda x    : x.list_localizations(),
        lambda x, y : x.add_localization(y),
        'resource_id',
        'localization'
    )

    transfer(
        lambda x    : x.list_car_models(),
        lambda x, y : x.add_car_model(y),
        'code',
        'car model',
        modifier=_clear_default_tags
    )

    logging.info('started transferring of ranking models')
    existing_ranking_models = client.list_ranking_models()
    source_ranking_models = source.list_ranking_models()
    for model_name in source_ranking_models:
        if model_name in existing_ranking_models:
            logging.info('skipping existing ranking model {}'.format(model_name))
            continue
        logging.info('cloning ranking model {}'.format(model_name))
        model = source.get_ranking_model(model_name)
        client.add_ranking_model(proto=model, name=model_name)
    logging.info('finished transferring of ranking models')

    logging.info('started transferring of tag descriptions')
    tag_types = client.list_tag_types()
    logging.info('known tag types: {}'.format(','.join(sorted(tag_types))))

    existing_tag_descriptions = client.list_tag_descriptions() or []
    existing_tag_names = set()
    for tag_description in existing_tag_descriptions:
        name = tag_description['name']
        existing_tag_names.add(name)
    logging.info('existing tag names: {}'.format(','.join(existing_tag_names)))

    tag_descriptions = source.list_tag_descriptions()
    for tag_description in tag_descriptions:
        name = tag_description['name']
        if name in existing_tag_names:
            logging.info('skipping existing tag description {}'.format(name))
            continue
        if not base_checking_tag_description(tag_description, tag_types):
            continue

        logging.info('cloning tag description {}'.format(name))
        tag_description.pop('description_index', None)
        client.add_tag_description(tag_description)
    logging.info('finished transferring of tag descriptions')

    transfer(
        lambda x    : x.list_areas(),
        lambda x, y : x.add_area(y),
        'area_id',
        'area'
    )

    logging.info('started transferring of area tags')
    existing_areas = client.list_areas()
    existing_area_tags = []
    for area in existing_areas:
        tags = area['hard_tags']
        for tag in tags:
            object_id = tag['object_id']
            tag_name = tag['tag']
            area_tag = (object_id, tag_name)
            existing_area_tags.append(area_tag)

    areas = source.list_areas()
    for area in areas:
        tags = area['hard_tags']
        for tag in tags:
            object_id = tag['object_id']
            tag_name = tag['tag']
            area_tag = (object_id, tag_name)
            if area_tag in existing_area_tags:
                logging.info('skipping existing {} {}'.format(object_id, tag_name))
                continue
            tag.pop('tag_id', None)
            logging.info('cloning tag {} {}'.format(object_id, tag_name))
            client.add_area_tag(tag)
    logging.info('finished transferring of area tags')

    transfer(
        lambda x    : x.list_landings(),
        lambda x, y : x.add_landing(y),
        'landing_id',
        'landing'
    )

    transfer(
        lambda x    : x.list_actions(),
        lambda x, y : x.add_action(y),
        'action_id',
        'action'
    )

    transfer(
        lambda x    : x.list_roles(),
        lambda x, y : x.add_role(y),
        'role_id',
        'role'
    )

    logging.info('started transferring of action-role links')
    existing_roles = client.list_roles() or []
    filled_roles = []
    for role in existing_roles:
        name = role['role_id']
        actions = role.get('actions', [])
        roles = role.get('slave_roles', [])
        score = len(actions) + len(roles)
        if score > 0:
            filled_roles.append(name)

    roles = source.list_roles() or []
    for role in roles:
        name = role['role_id']
        if name in filled_roles:
            logging.info('skip filled role {}'.format(name))
            continue
        actions = role.get('actions', [])
        roles = role.get('slave_roles', [])
        logging.info('cloning links for {}'.format(name))
        client.link_to_role(actions, roles)
    logging.info('finished transferring of action-role links')


def clone_rtbg(client, source):
    names = [
        'billing_tags_watcher',
        'chat_unread_pushes',
        'fines_charge_state_handler',
        'fines_charger',
        'futures_book',
        'rt_car_factors',
        'rt_car_scanner',
        'user_push_sender',
    ]
    logging.info('started transferring')
    objects = source.list_rtbg(names)
    for obj in objects:
        name = obj['bp_name']
        logging.info('tranferring {}'.format(name))
        obj.pop('bp_revision', None)
        obj['bp_settings'].pop('host_filter', None)
        client.add_rtbg(obj)
        logging.info('tranferred {}'.format(name))


def clone_settings(client, source):
    logging.info('started transferring of settings')
    settings = source.list_settings()
    client.add_settings(settings)
    logging.info('finished transferring of settings')
    client.add_setting('idm.enforce', False)
    client.add_setting('self_requester.extra_cgi', client.get_extra_cgi() or '')


def clone_users(client, source):
    def finalize_user(user, user_id=None):
        if user_id:
            user['id'] = user_id
        user['status'] = 'staff'
        upserted = client.upsert_user(user)
        if user_id:
            upserted_id = upserted['id']
            assert user_id == upserted_id, 'user_id mismatch after upsert: {} {}'.format(user_id, upserted_id)

    self_user_id = source.get_permissions()['user_id']

    existing_users = client.list_users(type_='staff')
    existing_uids = {}
    finalized_uids = []
    for user in existing_users:
        status = user['status']
        user_id = user['id']
        uid = user['uid']
        existing_uids[uid] = user_id
        if status == 'staff':
            finalized_uids.append(uid)

    source_users = source.list_users(type_='staff')
    for source_user in source_users:
        source_user_id = source_user['id']
        if source_user_id == self_user_id:
            logging.info('skipping self')
            continue

        user_login = source_user['username']
        user_uid = source_user['uid']
        if user_uid in finalized_uids:
            logging.info('skipping finalized user {} {} {}'.format(source_user_id, user_login, user_uid))
            continue

        user_id = existing_uids.get(user_uid, None)
        roles = source.get_user_roles(source_user_id)
        if len(roles) == 0:
            logging.info('cloning finalized user {} {} {}'.format(source_user_id, user_login, user_uid))
            finalize_user(source_user, user_id)
            continue

        logging.info('cloning user {} {} {}'.format(source_user_id, user_login, user_uid))
        user = client.upsert_user(source_user)
        user_id = user['id']

        for role in roles:
            role_id = role.role_id
            logging.info('cloning role {} for user {}'.format(role_id, user_id))
            client.add_user_role(role_id, user_id)

        finalize_user(source_user, user_id)


def clone_wallets(client, source):
    logging.info('started transferring of wallets')
    wallets = source.list_wallets()
    for wallet in wallets:
        name = wallet['name']
        if 'corp_yataxi' in name:
            logging.info('skipping {}'.format(name))
            continue
        logging.info('adding wallet {}'.format(name))
        client.add_wallet(wallet)
    logging.info('finished transferring of wallets')


def clone_self(client, source):
    source_permissions = source.get_permissions()
    source_roles = source_permissions['roles']
    permissions = client.get_permissions()
    roles = permissions['roles']
    user_id = permissions['user_id']
    transferred = True
    for role in source_roles:
        if role not in roles:
            transferred = False
            break
    if not transferred:
        for role in source_roles:
            if role in roles:
                continue
            logging.info('cloning role {} for self {}'.format(role, user_id))
            client.add_user_role(role, user_id)


def clone(client, args):
    source_endpoint = args.clone_source_endpoint
    source = api.BackendClient(endpoint=source_endpoint, public_token=args.public_token, private_token=args.private_token)
    for i in range(max(1, args.retries)):
        try:
            if i > 0:
                logging.warning('attempt {}'.format(i + 1))
            acquire_root(client)
            if not args.clone_skip_metadata:
                clone_metadata(client, source)
            if not args.clone_skip_settings:
                clone_settings(client, source)
            if not args.clone_skip_self:
                clone_self(client, source)
            if not args.clone_skip_users:
                clone_users(client, source)
            if not args.clone_skip_cars:
                clone_cars(client, source, args)
            if not args.clone_skip_rtbg:
                clone_rtbg(client, source)
            if not args.clone_skip_wallets:
                clone_wallets(client, source)
            break
        except Exception as e:
            logging.error('an exception has occurred: {}'.format(e))
            traceback.print_tb(e.__traceback__)


def dump(client, args):
    data = {}
    dump_metadata(client, data)
    dump_settings(client, data)
    dumpfile = open(args.dump_filename, "w", encoding='utf-8')
    dumpfile.write(json.dumps(data, ensure_ascii=False))
    dumpfile.close()


def dump_metadata(client, data):
    def get_objects(lister, comment):
        logging.info('started dumping of {}'.format(comment))
        return lister(client) or []

    state_filters = get_objects(
        lambda x    : x.list_state_filters(),
        'state filter'
    )
    data["state_filters"] = state_filters

    localizations = get_objects(
        lambda x    : x.list_localizations(),
        'localization'
    )
    data["localizations"] = localizations

    car_models = get_objects(
        lambda x    : x.list_car_models(),
        'car model'
    )
    data["car_models"] = car_models

    tag_types = get_objects(
        lambda x    : x.list_tag_types(),
        'tag type'
    )

    tag_descriptions = get_objects(
        lambda x    : x.list_tag_descriptions(),
        'tag description'
    )

    valid_tag_descriptions = []
    for tag_description in tag_descriptions:
        if not base_checking_tag_description(tag_description, tag_types):
            continue

        logging.info('dumping tag description {}'.format(tag_description['name']))
        tag_description.pop('description_index', None)
        valid_tag_descriptions.append(tag_description)

    data["tag_descriptions"] = valid_tag_descriptions

    areas = get_objects(
        lambda x    : x.list_areas(),
        'area'
    )
    data["areas"] = areas

    area_tags = []
    for area in areas:
        tags = area['hard_tags']
        for tag in tags:
            object_id = tag['object_id']
            tag_name = tag['tag']
            tag.pop('tag_id', None)
            logging.info('dumping tag {} {}'.format(object_id, tag_name))
            area_tags.append(tag)

    data["area_tags"] = area_tags

    actions = get_objects(
        lambda x    : x.list_actions(),
        'actions'
    )
    data["actions"] = actions

    roles = get_objects(
        lambda x    : x.list_roles(),
        'role'
    )
    data["roles"] = roles


def dump_settings(client, data):
    logging.info('started dumping of settings')
    settings = client.list_settings()
    settings.append({
        "setting_key": 'idm.enforce',
        "setting_value": 'false',
    })
    settings.append({
        "setting_key": 'self_requester.extra_cgi',
        "setting_value": client.get_extra_cgi() or '',
    })
    data["settings"] = settings


def apply_dump(client, args):
    dumpfile = open(args.apply_dump_filename, "r")
    dumpdata = json.loads(dumpfile.read())
    dumpfile.close()

    try:
        acquire_root(client)
        client.set_force_fetch_permissions("true")
        apply_metadata(client, dumpdata)
        apply_settings(client, dumpdata)
    except Exception as e:
        logging.error('an exception has occurred: {}'.format(e))
        traceback.print_tb(e.__traceback__)

    if args.users_filename is None:
        return

    usersfile = open(args.users_filename, "r")
    usersdata = json.loads(usersfile.read())
    usersfile.close()

    client.add_setting("handlers.api/staff/user/edit.force_show_objects_without_tags", "true")
    try:
        apply_users(client, usersdata)
    except Exception as e:
        logging.error('an exception has occurred: {}'.format(e))
        traceback.print_tb(e.__traceback__)
    client.add_setting("user_permissions.ignore_actuality", "false")


def apply_metadata(client, data):
    logging.info('started applying of state_filters')
    state_filters = data["state_filters"]
    for filter in state_filters:
        client.add_state_filter(filter)

    logging.info('started applying of localizations')
    localizations = data["localizations"]
    for localization in localizations:
        client.add_localization(localization)

    logging.info('started applying of car_models')
    car_models = data["car_models"]
    for model in car_models:
        client.add_car_model(model)

    logging.info('started applying of tag_descriptions')
    tag_descriptions = data["tag_descriptions"]
    for description in tag_descriptions:
        client.add_tag_description(description)

    logging.info('started applying of areas')
    areas = data["areas"]
    for area in areas:
        try:
            client.add_area(area)
        except Exception as e:
            logging.error("{} exception: {}".format(area, e))

    logging.info('started applying of area_tags')
    area_tags = data["area_tags"]
    for tag in area_tags:
        client.add_area_tag(tag)

    logging.info('started applying of actions')
    actions = data["actions"]
    for action in actions:
        if action['action_id'].startswith('__root'):
            continue
        client.add_action(action)

    logging.info('started applying of roles')
    roles = data["roles"]
    for role in roles:
        client.add_role(role)

    logging.info('started applying of links')
    for role in roles:
        name = role['role_id']
        actions = role.get('actions', [])
        roles = role.get('slave_roles', [])
        logging.info('applying links for {}'.format(name))
        client.link_to_role(actions, roles)


def apply_settings(client, data):
    logging.info('started applying of settings')
    client.set_timeout(300)
    client.add_settings(data["settings"])


def apply_users(client, data):
    logging.info('apply users')
    apply_users_group(client, data["admin"])
    apply_users_group(client, data["dev"])


def apply_users_group(client, data):
    users = data["users"]
    roles = data["roles"]

    for user in users:
        user = client.upsert_user(user)
        user_id = user['id']

        for role_id in roles:
            client.add_user_role(role_id, user_id)


def create_extensions(cursor=None):
    for extension, q in resource.iteritems('/drive/extensions/', strip_prefix=True):
        query = q.decode('utf-8')
        logging.info('creating extension {}'.format(extension))
        if cursor:
            cursor.execute(query)


def create_tables(cursor=None):
    for type_, q in resource.iteritems('/drive/type/', strip_prefix=True):
        query = q.decode('utf-8')
        logging.info('creating type {}'.format(type_))
        if cursor:
            cursor.execute(query)
    for table, q in resource.iteritems('/drive/table/', strip_prefix=True):
        query = q.decode('utf-8')
        logging.info('creating table {}'.format(table))
        if cursor:
            cursor.execute(query)


def create_indexes(cursor=None):
    for indexes, q in resource.iteritems('/drive/indexes/', strip_prefix=True):
        query = q.decode('utf-8')
        query_lines = query.splitlines()
        for query_line in query_lines:
            logging.info('creating indexes {}'.format(query_line))
            if cursor:
                cursor.execute(query_line)


def drop_tables(cursor=None):
    tables = []
    for table in resource.iterkeys('/drive/table/', strip_prefix=True):
        tables.append(table)
    tables.reverse()
    for table in tables:
        query = 'DROP TABLE IF EXISTS "{}";'.format(table)
        logging.info('dropping table {} with query: {}'.format(table, query))
        if cursor:
            cursor.execute(query)
    types = []
    for type_ in resource.iterkeys('/drive/type/', strip_prefix=True):
        types.append(type_)
    types.reverse()
    for type_ in types:
        query = 'DROP TYPE IF EXISTS "{}";'.format(type_)
        logging.info('dropping type {} with query: {}'.format(type_, query))
        if cursor:
            cursor.execute(query)


def get_host(client, args):
    print('endpoint:' + client.get_endpoint())
    for _ in range(args.get_host_retries):
        print('host:' + client.get_host())


def prepare_database(connection_string):
    with psycopg2.connect(connection_string) as connection:
        with connection.cursor() as cursor:
            create_extensions(cursor)
            create_tables(cursor)

    with psycopg2.connect(connection_string) as connection:
        connection.set_session(autocommit=True)
        with connection.cursor() as cursor:
            create_indexes(cursor)


def wipe_database(connection_string):
    if 'extmaps-carsharing-production' in connection_string:
        raise NotImplementedError()
    with psycopg2.connect(connection_string) as connection:
        with connection.cursor() as cursor:
            drop_tables(cursor)


def fill_parser(parser):
    subparsers = parser.add_subparsers(help='action')
    clone_parser = subparsers.add_parser('clone', help='clone metadata')
    clone_parser.add_argument('-s', '--source', dest='clone_source_endpoint', help='metadata source')
    clone_parser.add_argument('--skip-cars', dest='clone_skip_cars', help='skip cars transfer', action='store_true')
    clone_parser.add_argument('--skip-metadata', dest='clone_skip_metadata', help='skip metadata transfer', action='store_true')
    clone_parser.add_argument('--skip-rtbg', dest='clone_skip_rtbg', help='skip rtbg transfer', action='store_true')
    clone_parser.add_argument('--skip-settings', dest='clone_skip_settings', help='skip settings transfer', action='store_true')
    clone_parser.add_argument('--skip-self', dest='clone_skip_self', help='skip self transfer', action='store_true')
    clone_parser.add_argument('--skip-users', dest='clone_skip_users', help='skip users transfer', action='store_true')
    clone_parser.add_argument('--skip-wallets', dest='clone_skip_wallets', help='skip wallet descriptions transfer', action='store_true')
    clone_parser.add_argument('-r', '--retries', dest='retries', help='number of retries', type=int, default=9)
    clone_parser.add_argument('--cars-percentange', dest='cars_percentage', help='percentage of cars to transfer', type=int, default=1)
    clone_parser.add_argument('--cars-random-seed', dest='cars_random_seed', help='random seed for cars transfer', type=str, default='42')

    get_host_parser = subparsers.add_parser('get_host', help='get underlying host')
    get_host_parser.add_argument('-r', '--retries', dest='get_host_retries', help='number of retries', type=int, default=9)

    prepare_database_parser = subparsers.add_parser('prepare', help='prepare database')
    prepare_database_parser.add_argument('-c', '--connection-string', dest='prepare_connection_string')

    wipe_database_parser = subparsers.add_parser('wipe', help='wipe database')
    wipe_database_parser.add_argument('-c', '--connection-string', dest='wipe_connection_string')

    dump_parser = subparsers.add_parser('dump', help='dump server data')
    dump_parser.add_argument('-f', '--filename', dest='dump_filename')

    dump_parser = subparsers.add_parser('apply_dump', help='apply dump server data')
    dump_parser.add_argument('-f', '--filename', dest='apply_dump_filename')
    dump_parser.add_argument('-u', '--users_filename', dest='users_filename')


def execute(client, args):
    if 'clone_source_endpoint' in args:
        clone(client, args)
        return
    if 'get_host_retries' in args:
        get_host(client, args)
        return
    if 'prepare_connection_string' in args:
        prepare_database(args.prepare_connection_string)
        return
    if 'wipe_connection_string' in args:
        wipe_database(args.wipe_connection_string)
        return
    if 'dump_filename' in args:
        dump(client, args)
        return
    if 'apply_dump_filename' in args:
        apply_dump(client, args)
        return

    logging.error('cannot determine world action')
