import dataclasses
import logging
import os
import pgmigrate

import library.python.resource

from drive.devops.cli.secrets.vault import VaultClient


@dataclasses.dataclass
class MigrateArgs:
    conn: str
    target: str
    base_dir: str


class DrivematicsStableEnvironment:
    host_variable = 'PRODUCTION_DRIVEMATICS_DB_HOSTS'
    port = 6432
    dbname = 'drive-leasing-db'
    user = 'carsharing'
    password_secret_id = ("sec-01ewxe4ppazasgckyzagf4fk23", "PASSWORD")


class DrivematicsTestingEnvironment:
    host_variable = 'TESTING_DRIVEMATICS_DB_HOSTS'
    port = 6432
    dbname = 'drive-leasing-db-testing'
    user = 'carsharing'
    password_secret_id = ("sec-01fajd57qzp0saqhsevht70d01", "PASSWORD")


class StableEnvironment:
    host_variable = 'PRODUCTION_DB_HOSTS'
    port = 6432
    dbname = 'extmaps-carsharing-production'
    user = 'carsharing'
    password_secret_id = ('sec-01fbgx7ex5c9hqmtb3sfwkdsfw', 'password')


class TestingEnvironment:
    host_variable = 'TESTING_DB_HOSTS'
    port = 6432
    dbname = 'extmaps-carsharing-testing'
    user = 'carsharing'
    password_secret_id = ('sec-01f9eczqzk4ej2gww37787pwhs', 'password')


class QAEnvironment:
    host_variable = 'QA_DB_HOSTS'
    port = 6432
    dbname = 'extmaps-carsharing-testing-qa'
    user = 'carsharing'
    password_secret_id = ('sec-01g7v8rw3mrk142ry9n083bnrt', 'password')


class STEnvironment:
    host_variable = 'ST_DB_HOSTS'
    port = 6432
    dbname = 'extmaps-carsharing-testing-support'
    user = 'carsharing'
    password_secret_id = ('sec-01g7v8vjaskaqawc9rwb1v8v34', 'password')


ENVIRONMENTS = {
    'drivematics_stable': DrivematicsStableEnvironment,
    'drivematics_testing': DrivematicsTestingEnvironment,
    'stable': StableEnvironment,
    'testing': TestingEnvironment,
    'qa': QAEnvironment,
    'st': STEnvironment,
}


def _get_config(connection_string, migration_directory):
    return pgmigrate.get_config(migration_directory, MigrateArgs(
        conn=connection_string,
        target='latest',
        base_dir=migration_directory,
    ))


def _get_variables():
    content = library.python.resource.find('environment.lua').decode('utf-8')
    result = {}
    for line in content.splitlines():
        partitioned = line.partition('=')
        key = partitioned[0]
        value = partitioned[2]
        result[key] = value
    return result


def create_connection_string(env, user=None, password=None):
    environment = ENVIRONMENTS[env]
    variables = _get_variables()
    host = variables[environment.host_variable]
    port = environment.port
    dbname = environment.dbname
    if password:
        password = os.getenv(password, password)
    if not user:
        user = environment.user
    if not password:
        vault = VaultClient()
        secret = vault.get_version(environment.password_secret_id[0])
        password = secret['value'][environment.password_secret_id[1]]
    return 'host={} port={} dbname={} user={} password={} target_session_attrs=read-write'.format(
        host,
        port,
        dbname,
        user,
        password
    )


def create_migration_directory(work_directory):
    target = os.path.join(work_directory, 'mgrtns')
    if not os.path.exists(target):
        os.mkdir(target)
    if not os.path.isdir(target):
        raise NotADirectoryError('root {} is not a directory'.format(target))

    migrations_yml = os.path.join(target, 'migrations.yml')
    migrations_yml_content = library.python.resource.find('migrations.yml')
    with open(migrations_yml, 'wb') as migrations_yml_file:
        migrations_yml_file.write(migrations_yml_content)

    migrations = os.path.join(target, 'migrations')
    if not os.path.exists(migrations):
        os.mkdir(migrations)
    if not os.path.isdir(migrations):
        raise NotADirectoryError('migrations {} is not a directory'.format(migrations))
    for key, migration_content in library.python.resource.iteritems():
        if not key.startswith('V'):
            continue
        migration = os.path.join(migrations, key)
        with open(migration, 'wb') as migration_file:
            migration_file.write(migration_content)
    return target


def create_config(opts):
    work_directory = opts.work_directory or '.'
    migration_directory = create_migration_directory(work_directory)
    connection_string = create_connection_string(opts.env, opts.user, opts.password)
    return _get_config(connection_string, migration_directory)


def main_migation_info(opts):
    logging.basicConfig(level=logging.INFO, format='%(asctime)s %(levelname)s: %(message)s')
    config = create_config(opts)
    pgmigrate.info(config)


def main_migration_migrate(opts):
    logging.basicConfig(level=logging.INFO, format='%(asctime)s %(levelname)s: %(message)s')
    config = create_config(opts)
    pgmigrate.migrate(config)


def register(group):
    migrations = group.add_parser('migration')
    migrations.add_argument('-e', '--env', help='Environment')
    migrations.add_argument('--user', help='Database user')
    migrations.add_argument('--password', help='Database password')
    migrations.add_argument('--work-directory', help='Work directory')

    subparsers = migrations.add_subparsers(help='command')
    info = subparsers.add_parser('info', help='display migration info')
    info.set_defaults(main=main_migation_info)

    migrate = subparsers.add_parser('migrate', help='migrate to the latest')
    migrate.set_defaults(main=main_migration_migrate)
