# -*- coding: utf-8 -*-

from json import load, dump
from base64 import b64encode, b64decode
from yt.wrapper import YtClient
from drive.backend.api.client import BackendClient
from .yt_profile import BACKEND_LOG_STREAM_PATH


BACKEND_PRESTABLE_ENDPOINT = "https://prestable.carsharing.yandex.net"


def register(group):
    find_cmd = group.add_parser("find-models")
    find_cmd.set_defaults(main=find_main)
    find_cmd.add_argument("--proxy", default="hahn")
    find_cmd.add_argument("--days", type=int, default=60)
    find_cmd.add_argument("--log", default=BACKEND_LOG_STREAM_PATH)
    find_cmd.add_argument("--endpoint", default=BACKEND_PRESTABLE_ENDPOINT)
    remove_cmd = group.add_parser("remove-models")
    remove_cmd.set_defaults(main=remove_main)
    remove_cmd.add_argument("-n", "--name", nargs='+', required=True)
    remove_cmd.add_argument("--endpoint", default=BACKEND_PRESTABLE_ENDPOINT)
    remove_cmd.add_argument("--backup", default="models-backup.json")
    remove_cmd.add_argument("--rollback", action="store_true")


def mapper(row):
    if row["event"] == "TModelsStorage::GetOfferModel":
        yield dict(model=row["data"]["name"])


def reducer(key, rows):
    count = 0
    for _ in rows:
        count += 1
    yield dict(model=key["model"], count=count)


def find_main(opts):
    dc = BackendClient(opts.endpoint, require_public_token=False)
    all_models = dc.list_ranking_models()
    used_models = {}
    yc = YtClient(opts.proxy)
    log_1d = opts.log.rstrip('/') + "/1d"
    with yc.TempTable() as temp:
        yc.run_map_reduce(
            mapper, reducer,
            source_table=yc.list(log_1d, sort=True, absolute=True)[-opts.days:],
            destination_table=temp,
            reduce_by=("model",),
            sort_by=("model",),
        )
        for row in yc.read_table(temp):
            used_models[row["model"]] = row["count"]
    to_remove = []
    for model in all_models:
        if model in used_models:
            print("{} [v] {}".format(model, used_models[model]))
        else:
            print("{} [x]".format(model))
            to_remove.append(model)
    print("Suggested models to remove:")
    print(", ".join(to_remove))
    print("cli remove-models -n {}".format(" ".join(map(lambda x: "'{}'".format(x), to_remove))))


def remove_main(opts):
    dc = BackendClient(opts.endpoint, require_public_token=False)
    if opts.rollback:
        print("Rollback previous removed models")
        with open(opts.backup) as fd:
            backup = load(fd)
        for name, b64model in backup:
            print("Restore model: {}".format(name))
            model = b64decode(b64model)
            dc.add_ranking_model(name=name, proto=model)
        return
    backup = []
    print("Removing models")
    try:
        for name in opts.name:
            print("Removing model: {}".format(name))
            model = dc.get_ranking_model(name)
            b64model = b64encode(model).decode()
            backup.append((name, b64model))
            dc.remove_ranking_model(name)
    finally:
        with open(opts.backup, "w") as fd:
            dump(backup, fd)
