#pragma once

#include <csp.h>
#include <os.h>

#include <time/timer.h>

#include <util/singleton.h>

namespace NBoard {
    namespace NSpi {
        using TNumber = NChip::NSpi::ENumber;

        enum class EEvent: uint8_t {
            None = 0,
            TransmitComplete,
            ReceiveComplete,
            Error
        };

        template<class TChipSelectPin>
        class TChipSelect final: public NLibrary::NLock::ILockable {
        public:
            TChipSelect()
                : Mutex()
            {
                TChipSelectPin::Init();
                TChipSelectPin::Off();
            }

            virtual ~TChipSelect() {
                TChipSelectPin::Deinit();
            }

            void Lock() override {
                if (!NChip::InInterrupt()) {
                    Mutex.Lock();
                }
                TChipSelectPin::On();
            }

            void Unlock() override {
                TChipSelectPin::Off();
                if (!NChip::InInterrupt()) {
                    Mutex.Unlock();
                }
            }

        private:
            NOs::TMutex Mutex;
        };

        template<TNumber Number, class TChipSelectPin>
        class TSpi final: public NLibrary::NUtil::TSingleton<TSpi<Number, TChipSelectPin>> {
            friend NLibrary::NUtil::TSingleton<TSpi<Number, TChipSelectPin>>;

        public:
            static constexpr size_t EventQueueSize = 2;
            static constexpr auto DefaultTimeout = NLibrary::NTime::TTime::Second(1);
            using TQueue = NOs::NQueue::TQueue<EEvent, EventQueueSize>;
            using TChipLock = TChipSelect<TChipSelectPin>;

        public:
            bool Init() {
                if (!NChip::NSpi::Init(Number)) {
                    return false;
                }

                return true;
            }

            bool Deinit() {
                NChip::NSpi::Deinit(Number);
                Event.Deinit();
                return true;
            }

            TChipLock& GetLock() {
                return ChipSelect;
            }

            bool Transmit(const uint8_t* data, size_t size) {
                if (!NChip::NSpi::Transmit(Number, data, size)) {
                    return false;
                }

                return Wait(EEvent::TransmitComplete);
            }

            bool Receive(uint8_t* data, size_t size) {
                if (!NChip::NSpi::Receive(Number, data, size)) {
                    return false;
                }

                return Wait(EEvent::ReceiveComplete);
            }

            bool TransmitReceive(const uint8_t* txData, size_t txSize, uint8_t* rxData, size_t rxSize) {
                if (!NChip::NSpi::Transmit(Number, txData, txSize)) {
                    return false;
                }
                if (!Wait(EEvent::TransmitComplete)) {
                    return false;
                }
                if (!NChip::NSpi::Receive(Number, rxData, rxSize)) {
                    return false;
                }

                return Wait(EEvent::ReceiveComplete);
            }

            void Append(EEvent event) {
                Event.Put(event);
            }

        private:
            TChipLock ChipSelect;
            TQueue Event;

        private:
            TSpi()
                : ChipSelect()
                , Event()
            {
            }

            bool Wait(EEvent event) {
                NLibrary::NTime::TTimer timer;
                timer.Start();

                while(!timer.HasExpired(DefaultTimeout)) {
                    while (!Event.IsEmpty()) {
                        auto receiveEvent = Event.Get();

                        if (receiveEvent == event) {
                            return true;
                        }

                        if (receiveEvent == EEvent::Error) {
                            return false;
                        }
                    }
                    NOs::NTask::Delay(10);
                }

                return false;
            }
        };
    }
}
