#pragma once

#include <csp.h>
#include <os.h>

#include <data/vector.h>

#include <time/timer.h>

#include <util/singleton.h>

namespace NBoard {
    namespace NUart {
        using TNumber = NChip::NUart::ENumber;
        using TMode = NChip::NUart::EMode;

        enum class EEvent: uint8_t {
            None = 0,
            TransmitComplete,
            ReceiveComplete
        };

        enum class EInterface: uint8_t {
            Rs232,
            Rs485
        };

        struct TDummySwitcher {
            static void Init() {}
            static void Deinit() {}
            static void Transmit() {}
            static void Receive() {}
        };

        using TBaudRate = NChip::NUart::EBaudRate;
        using TParity = NChip::NUart::EParity;
        using TStopBit = NChip::NUart::EStopBit;
        using TFlowControl = NChip::NUart::EFlowControl;
        using TWordLength = NChip::NUart::EWordLength;
        using TSetting = NChip::NUart::TSetting;

        template<TNumber Number, TMode Mode, size_t PacketCount, size_t PacketSize,
                 EInterface Interface = EInterface::Rs232, typename TSwitcher = TDummySwitcher>
        class TUart final: public NLibrary::NUtil::TSingleton<TUart<Number, Mode, PacketCount, PacketSize, Interface, TSwitcher>> {
            friend NLibrary::NUtil::TSingleton<TUart<Number, Mode, PacketCount, PacketSize, Interface, TSwitcher>>;

        public:
            static constexpr size_t EventSize = 2;

            using TWord = uint8_t;
            using TPacket = NLibrary::NData::TVector<TWord, PacketSize>;
            using TRx = NOs::NQueue::TQueue<TPacket, PacketCount>;
            using TEvent = NOs::NQueue::TQueue<EEvent, EventSize>;

        public:
            bool Init(
                TBaudRate baudRate = TBaudRate::BR_115200,
                TParity parity = TParity::None,
                TStopBit stopBit = TStopBit::S1,
                TFlowControl flowControl = TFlowControl::None,
                TWordLength wordLength = TWordLength::W8
            ) {
                TSetting setting = {
                    .BaudRate = baudRate,
                    .WordLength = wordLength,
                    .StopBit = stopBit,
                    .Parity = parity,
                    .FlowControl = flowControl
                };

                bool result = NChip::NUart::Init(Number, setting);

                if constexpr (PacketCount > 0 && PacketCount > 0) {
                    NChip::NUart::Receive(Number, Mode, GetTemp(), PacketSize);
                }

                if constexpr (Interface == EInterface::Rs485) {
                    TSwitcher::Init();
                    TSwitcher::Receive();
                }

                return result;
            }

            bool Deinit() {
                if constexpr (PacketCount > 0 && PacketCount > 0) {
                    NChip::NUart::AbortReceive(Number, Mode);
                    Rx.Deinit();
                }

                if constexpr (Interface == EInterface::Rs485) {
                    TSwitcher::Deinit();
                }

                Event.Deinit();
                return true;
            }

            bool Transmit(const TWord* data, size_t size) {
                NLibrary::NLock::TLock lock(TxMutex);

                if constexpr (Interface == EInterface::Rs485) {
                    TSwitcher::Transmit();
                }

                NChip::NUart::Transmit(Number, Mode, data, size);
                Wait();

                if constexpr (Interface == EInterface::Rs485) {
                    TSwitcher::Receive();
                }

                return true;
            }

            bool HavePacket() const {
                if constexpr (PacketCount > 0 && PacketCount > 0) {
                    return !Rx.IsEmpty();
                } else {
                    return false;
                }
            }

            TPacket Receive() noexcept {
                if constexpr (PacketCount > 0 && PacketCount > 0) {
                    NLibrary::NLock::TLock lock(RxMutex);
                    return Rx.Get();
                } else {
                    return TPacket();
                }
            }

            void Idle(size_t size) {
                if constexpr (PacketCount > 0 && PacketCount > 0) {
                    Rx.Put(TPacket(GetTemp(), size));

                    NChip::NUart::AbortReceive(Number, Mode);
                    NChip::NUart::Receive(Number, Mode, GetTemp(), PacketSize);
                }
            }

            void Error() {
                if constexpr (PacketCount > 0 && PacketCount > 0) {
                    NChip::NUart::AbortReceive(Number, Mode);
                    NChip::NUart::Receive(Number, Mode, GetTemp(), PacketSize);
                }
            }

            void Append(EEvent event) {
                Event.Put(event);
            }

        private:
            TRx Rx;
            TEvent Event;
            NOs::TMutex RxMutex;
            NOs::TMutex TxMutex;

        private:
            TUart()
                : Rx()
                , Event()
                , RxMutex()
                , TxMutex()
            {
            }

            TWord* GetTemp() {
                if constexpr (PacketCount > 0 && PacketCount > 0) {
                    static TWord instance[PacketSize];
                    return instance;
                } else {
                    return nullptr;
                }
            }

            void Wait() {
                const auto timeout = NLibrary::NTime::TTime::Second(1);
                NLibrary::NTime::TTimer timer;
                timer.Start();

                while(!timer.HasExpired(timeout)) {
                    if (!Event.IsEmpty()) {
                        Event.Clear();
                        return;
                    }
                    NLibrary::NTime::TTime::Delay(NLibrary::NTime::TTime::MilliSecond(1));
                }
            }
        };
    }
}