#pragma once

#include <cstdint>

namespace NChip {
    namespace NGpio {
        enum class EPort: uint8_t {
            None = 0,
            A,
            B,
            C,
            D,
            E,
            F,
            G,
            H,
            I,
            J
        };

        enum class EPin: uint8_t {
            P0 = 0,
            P1,
            P2,
            P3,
            P4,
            P5,
            P6,
            P7,
            P8,
            P9,
            P10,
            P11,
            P12,
            P13,
            P14,
            P15,
            P16,
            P17,
            P18,
            P19,
            P20,
            P21,
            P22,
            P23,
            P24,
            P25,
            P26,
            P27,
            P28,
            P29,
            P30,
            P31,
            None = 255
        };

        enum class EMode: uint8_t {
            PushPull = 0,
            OpenDrain,
            Input,
            Analog,
            Rising,
            Falling,
            RisingFalling
        };

        enum class ESpeed: uint8_t {
            Low = 0,
            Medium,
            High,
            VeryHigh
        };

        enum class EPullup: uint8_t {
            None = 0,
            Up,
            Down
        };

        enum class ELine: int8_t {
            None = -1,
            L0 = 0,
            L1 = 1,
            L2 = 2,
            L3 = 3,
            L4 = 4,
            L5 = 5,
            L6 = 6,
            L7 = 7,
            L8 = 8,
            L9 = 9,
            L10 = 10,
            L11 = 11,
            L12 = 12,
            L13 = 13,
            L14 = 14,
            L15 = 15
        };

        void Init(EPort port, EPin pin, EMode mode, ESpeed speed, EPullup pullup);
        void Deinit(EPort port, EPin pin);
        void On(EPort port, EPin pin);
        void Off(EPort port, EPin pin);
        void Toggle(EPort port, EPin pin);
        bool State(EPort port, EPin pin);
        void EnableInterrupt(EPin pin);
        void SoftwareInterrupt(EPin pin);
        void DisableInterrupt(EPin pin);
        void LineHandler(EPin pin);

        template<EPort port, EPin pin, EMode mode, ESpeed speed = ESpeed::Low, EPullup pullup = EPullup::None>
        struct TGpio {
            inline static void Init() {
                NChip::NGpio::Init(port, pin, mode, speed, pullup);
            }
            inline static void Deinit() {
                NChip::NGpio::Deinit(port, pin);
            }
            inline static void On() {
                static_assert(mode == EMode::PushPull || mode == EMode::OpenDrain);
                NChip::NGpio::On(port, pin);
            }
            inline static void Off() {
                static_assert(mode == EMode::PushPull || mode == EMode::OpenDrain);
                NChip::NGpio::Off(port, pin);
            }
            inline static void Toggle() {
                static_assert(mode == EMode::PushPull || mode == EMode::OpenDrain);
                NChip::NGpio::Toggle(port, pin);
            }
            inline static bool State() {
                return NChip::NGpio::State(port, pin);
            }
            inline static void EnableInterrupt() {
                NChip::NGpio::EnableInterrupt(pin);
            }
            inline static void DisableInterrupt() {
                NChip::NGpio::DisableInterrupt(pin);
            }
            inline static void SoftwareInterrupt() {
                NChip::NGpio::SoftwareInterrupt(pin);
            }
        };
    }
}
