#include "csp.h"
#include "csp_hal.h"

namespace
{
    using TPin = uint16_t;
    using TDef = GPIO_TypeDef;
}

namespace NChip {
    namespace NGpio {
        void InitClock(EPort port);
        void DeinitClock(EPort port);
        TDef* PortToDef(EPort port);
        EPort DefToPort(TDef* def);
        uint32_t SpeedParse(ESpeed speed);
    }
}

namespace {
    NChip::NGpio::EPin DefToPin(TPin pin) {
        for (uint16_t i = 0; i < 16; ++i) {
            if ((pin >> i) & 0x01) {
                return static_cast<NChip::NGpio::EPin>(i);
            }
        }
        return NChip::NGpio::EPin::None;
    }
}

void NChip::NGpio::InitClock(EPort port) {
    switch (port) {
        case EPort::A:
            __HAL_RCC_GPIOA_CLK_ENABLE();
            break;
        case EPort::B:
            __HAL_RCC_GPIOB_CLK_ENABLE();
            break;
        case EPort::C:
            __HAL_RCC_GPIOC_CLK_ENABLE();
            break;
        case EPort::D:
            __HAL_RCC_GPIOD_CLK_ENABLE();
            break;
        case EPort::E:
            __HAL_RCC_GPIOE_CLK_ENABLE();
            break;
        case EPort::F:
            __HAL_RCC_GPIOF_CLK_ENABLE();
            break;
        case EPort::G:
            __HAL_RCC_GPIOG_CLK_ENABLE();
            break;
        default:
            break;
    }
}

void NChip::NGpio::DeinitClock(EPort port) {
    switch (port) {
        case EPort::A:
            __HAL_RCC_GPIOA_CLK_DISABLE();
            break;
        case EPort::B:
            __HAL_RCC_GPIOB_CLK_DISABLE();
            break;
        case EPort::C:
            __HAL_RCC_GPIOC_CLK_DISABLE();
            break;
        case EPort::D:
            __HAL_RCC_GPIOD_CLK_DISABLE();
            break;
        case EPort::E:
            __HAL_RCC_GPIOE_CLK_DISABLE();
            break;
        case EPort::F:
            __HAL_RCC_GPIOF_CLK_DISABLE();
            break;
        case EPort::G:
            __HAL_RCC_GPIOG_CLK_DISABLE();
            break;
        default:
            break;
    }
}

TDef* NChip::NGpio::PortToDef(EPort port) {
    switch (port) {
        case EPort::A:
            return GPIOA;
        case EPort::B:
            return GPIOB;
        case EPort::C:
            return GPIOC;
        case EPort::D:
            return GPIOD;
        case EPort::E:
            return GPIOE;
        case EPort::F:
            return GPIOF;
        case EPort::G:
            return GPIOG;
        case EPort::H:
            return GPIOH;
        case EPort::None:
        default:
            return nullptr;
    }
}

auto NChip::NGpio::DefToPort(TDef* def) -> EPort {
    if (!def) {
        return EPort::None;
    }

    if (def == GPIOA) {
        return EPort::A;
    } else if (def == GPIOB) {
        return EPort::B;
    } else if (def == GPIOC) {
        return EPort::C;
    } else if (def == GPIOD) {
        return EPort::D;
    } else if (def == GPIOE) {
        return EPort::E;
    } else if (def == GPIOF) {
        return EPort::F;
    } else if (def == GPIOG) {
        return EPort::G;
    } else if (def == GPIOH) {
        return EPort::H;
    }

    return EPort::None;
}

uint32_t NChip::NGpio::SpeedParse(ESpeed speed) {
    switch (speed) {
        default:
        case ESpeed::Low:
            return GPIO_SPEED_FREQ_LOW;
        case ESpeed::Medium:
            return GPIO_SPEED_FREQ_MEDIUM;
        case ESpeed::High:
            return GPIO_SPEED_FREQ_HIGH;
        case ESpeed::VeryHigh:
            return GPIO_SPEED_FREQ_VERY_HIGH;
    }
}

void NChip::NGpio::EnableInterrupt(EPin pin) {
    TPin pinDefine = static_cast<TPin>(pin);

    if (pin == EPin::P0) {
        HAL_NVIC_SetPriority(EXTI0_IRQn, 15, 0);
        HAL_NVIC_EnableIRQ(EXTI0_IRQn);
    } else if (pin == EPin::P1) {
        HAL_NVIC_SetPriority(EXTI1_IRQn, 15, 0);
        HAL_NVIC_EnableIRQ(EXTI1_IRQn);
    } else if (pin == EPin::P2) {
        HAL_NVIC_SetPriority(EXTI2_IRQn, 15, 0);
        HAL_NVIC_EnableIRQ(EXTI2_IRQn);
    } else if (pin == EPin::P3) {
        HAL_NVIC_SetPriority(EXTI3_IRQn, 15, 0);
        HAL_NVIC_EnableIRQ(EXTI3_IRQn);
    } else if (pin == EPin::P4) {
        HAL_NVIC_SetPriority(EXTI4_IRQn, 15, 0);
        HAL_NVIC_EnableIRQ(EXTI4_IRQn);
    } else if (pinDefine >= 5 && pinDefine <= 9) {
        HAL_NVIC_SetPriority(EXTI9_5_IRQn, 13, 0);
        HAL_NVIC_EnableIRQ(EXTI9_5_IRQn);
    }
}

void NChip::NGpio::DisableInterrupt(EPin pin) {
    TPin pinDefine = static_cast<TPin>(pin);

    if (pin == EPin::P0) {
        HAL_NVIC_DisableIRQ(EXTI0_IRQn);
    } else if (pin == EPin::P1) {
        HAL_NVIC_DisableIRQ(EXTI1_IRQn);
    } else if (pin == EPin::P2) {
        HAL_NVIC_DisableIRQ(EXTI2_IRQn);
    } else if (pin == EPin::P3) {
        HAL_NVIC_DisableIRQ(EXTI3_IRQn);
    } else if (pin == EPin::P4) {
        HAL_NVIC_DisableIRQ(EXTI4_IRQn);
    } else if (pinDefine >= 5 && pinDefine <= 9) {
        HAL_NVIC_DisableIRQ(EXTI9_5_IRQn);
    }
}

namespace {
    void HalGpioExtiHandler(TPin pin) {
        NChip::NGpio::LineHandler(DefToPin(pin));
    }
}

extern "C" {
    void EXTI0_IRQHandler() {
        HAL_GPIO_EXTI_IRQHandler(GPIO_PIN_0);
    }
    void EXTI1_IRQHandler() {
        HAL_GPIO_EXTI_IRQHandler(GPIO_PIN_1);
    }
    void EXTI2_IRQHandler() {
        HAL_GPIO_EXTI_IRQHandler(GPIO_PIN_2);
    }
    void EXTI3_IRQHandler() {
        HAL_GPIO_EXTI_IRQHandler(GPIO_PIN_3);
    }
    void EXTI4_IRQHandler() {
        HAL_GPIO_EXTI_IRQHandler(GPIO_PIN_4);
    }
    void EXTI9_5_IRQHandler() {
        const size_t pinsCount = 5;
        uint16_t pins[pinsCount] = {
            GPIO_PIN_5,
            GPIO_PIN_6,
            GPIO_PIN_7,
            GPIO_PIN_8,
            GPIO_PIN_9
        };

        for (size_t i = 0; i < pinsCount; ++i) {
            uint16_t pin = pins[i];
            if(__HAL_GPIO_EXTI_GET_IT(pin) != RESET) {
                __HAL_GPIO_EXTI_CLEAR_IT(pin);
                HAL_GPIO_EXTI_Callback(pin);
            }
        }
    }

    void HAL_GPIO_EXTI_Callback(uint16_t pin) {
        HalGpioExtiHandler(pin);
    }
}
