#include <csp.h>
#include <csp_hal.h>

namespace {
    using THandler = SPI_HandleTypeDef;
    using TDmaHandler = DMA_HandleTypeDef;
    using TInstance = SPI_TypeDef;

    THandler hspi1; // 1 -- memory
    THandler hspi2; // 2 -- lis
    THandler hspi5; // 3 -- nrf
    TDmaHandler hdmaSpi1Rx;
    TDmaHandler hdmaSpi1Tx;

    constexpr uint16_t DefaultTimeout = 1000;
}

namespace NChip {
    namespace NSpi {
        THandler* GetHandler(ENumber number) {
            switch (number) {
                case ENumber::N1:
                    return &hspi1;
                case ENumber::N2:
                    return &hspi2;
                case ENumber::N3:
                    return &hspi5;
                default:
                    return nullptr;
            }
        }

        TInstance* GetInstance(ENumber number) {
            switch (number) {
                case ENumber::N1:
                    return SPI1;
                case ENumber::N2:
                    return SPI2;
                case ENumber::N3:
                    return SPI5;
                default:
                    return nullptr;
            }
        }

        ENumber GetNumber(THandler* handler) {
            if (handler->Instance == SPI1) {
                return ENumber::N1;
            } else if (handler->Instance == SPI2) {
                return ENumber::N2;
            } else if (handler->Instance == SPI5) {
                return ENumber::N3;
            }

            return ENumber::None;
        }
    }
}

bool NChip::NSpi::Init(ENumber number) {
    auto handler = GetHandler(number);
    auto instance = GetInstance(number);

    if (!handler || !instance) {
        return false;
    }

    handler->Instance = instance;
    handler->Init.Mode = SPI_MODE_MASTER;
    handler->Init.Direction = SPI_DIRECTION_2LINES;
    handler->Init.DataSize = SPI_DATASIZE_8BIT;
    handler->Init.CLKPolarity = SPI_POLARITY_LOW;
    handler->Init.CLKPhase = SPI_PHASE_1EDGE;
    handler->Init.NSS = SPI_NSS_SOFT;
    handler->Init.FirstBit = SPI_FIRSTBIT_MSB;
    handler->Init.TIMode = SPI_TIMODE_DISABLE;
    handler->Init.CRCCalculation = SPI_CRCCALCULATION_DISABLE;
    handler->Init.CRCPolynomial = 10;

    if (number == ENumber::N1) {
        handler->Init.BaudRatePrescaler = SPI_BAUDRATEPRESCALER_2;
    } else if (number == ENumber::N2) {
        handler->Init.BaudRatePrescaler = SPI_BAUDRATEPRESCALER_4;
    } else if (number == ENumber::N3) {
        handler->Init.BaudRatePrescaler = SPI_BAUDRATEPRESCALER_8;
    }

    if (HAL_SPI_Init(handler) != HAL_OK) {
        NChip::ErrorCallback(__FILE__, __LINE__);
        return false;
    }

    return true;
}

bool NChip::NSpi::Deinit(ENumber number) {
    auto handler = GetHandler(number);

    if (!handler) {
        return false;
    }

    if (HAL_SPI_DeInit(handler) != HAL_OK) {
        NChip::ErrorCallback(__FILE__, __LINE__);
        return false;
    }

    return true;
}

bool NChip::NSpi::Transmit(ENumber number, const uint8_t* data, size_t size) {
    auto handler = GetHandler(number);

    if (!handler) {
        return false;
    }

    HAL_StatusTypeDef result = HAL_ERROR;

    if (number == ENumber::N1) {
        result = HAL_SPI_Transmit_DMA(handler, const_cast<uint8_t*>(data), size);
    } else {
        result = HAL_SPI_Transmit(handler, const_cast<uint8_t*>(data), size, size * DefaultTimeout);
        TransmitComplete(number);
    }

    return result == HAL_OK;
}

bool NChip::NSpi::Receive(ENumber number, uint8_t* data, size_t size) {
    auto handler = GetHandler(number);

    if (!handler) {
        return false;
    }

    HAL_StatusTypeDef result = HAL_ERROR;

    if (number == ENumber::N1) {
        result = HAL_SPI_Receive_DMA(handler, data, size);
    } else {
        result = HAL_SPI_Receive(handler, data, size, size * DefaultTimeout);
        ReceiveComplete(number);
    }

    return result == HAL_OK;
}

__weak void NChip::NSpi::TransmitComplete(ENumber number) {
    UNUSED(number);
}

__weak void NChip::NSpi::ReceiveComplete(ENumber number) {
    UNUSED(number);
}

namespace {
    void HalSpiMspInit(THandler* handler) {
        GPIO_InitTypeDef GPIO_InitStruct;
        if(handler->Instance == SPI1) {
            __HAL_RCC_SPI1_CLK_ENABLE();

            __HAL_RCC_GPIOA_CLK_ENABLE();
            /**SPI1 GPIO Configuration
            PA5     ------> SPI1_SCK
            PA6     ------> SPI1_MISO
            PA7     ------> SPI1_MOSI
            */
            GPIO_InitStruct.Pin = GPIO_PIN_5 | GPIO_PIN_6 | GPIO_PIN_7;
            GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
            GPIO_InitStruct.Pull = GPIO_NOPULL;
            GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_VERY_HIGH;
            GPIO_InitStruct.Alternate = GPIO_AF5_SPI1;
            HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

            /* SPI1 DMA Init */
            /* SPI1_RX Init */
            hdmaSpi1Rx.Instance = DMA2_Stream0;
            hdmaSpi1Rx.Init.Channel = DMA_CHANNEL_3;
            hdmaSpi1Rx.Init.Direction = DMA_PERIPH_TO_MEMORY;
            hdmaSpi1Rx.Init.PeriphInc = DMA_PINC_DISABLE;
            hdmaSpi1Rx.Init.MemInc = DMA_MINC_ENABLE;
            hdmaSpi1Rx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
            hdmaSpi1Rx.Init.MemDataAlignment = DMA_MDATAALIGN_BYTE;
            hdmaSpi1Rx.Init.Mode = DMA_NORMAL;
            hdmaSpi1Rx.Init.Priority = DMA_PRIORITY_LOW;
            hdmaSpi1Rx.Init.FIFOMode = DMA_FIFOMODE_DISABLE;
            if (HAL_DMA_Init(&hdmaSpi1Rx) != HAL_OK) {
                NChip::ErrorCallback(__FILE__, __LINE__);
            }

            __HAL_LINKDMA(handler, hdmarx, hdmaSpi1Rx);

            hdmaSpi1Tx.Instance = DMA2_Stream3;
            hdmaSpi1Tx.Init.Channel = DMA_CHANNEL_3;
            hdmaSpi1Tx.Init.Direction = DMA_MEMORY_TO_PERIPH;
            hdmaSpi1Tx.Init.PeriphInc = DMA_PINC_DISABLE;
            hdmaSpi1Tx.Init.MemInc = DMA_MINC_ENABLE;
            hdmaSpi1Tx.Init.PeriphDataAlignment = DMA_PDATAALIGN_BYTE;
            hdmaSpi1Tx.Init.MemDataAlignment = DMA_MDATAALIGN_BYTE;
            hdmaSpi1Tx.Init.Mode = DMA_NORMAL;
            hdmaSpi1Tx.Init.Priority = DMA_PRIORITY_LOW;
            hdmaSpi1Tx.Init.FIFOMode = DMA_FIFOMODE_DISABLE;
            if (HAL_DMA_Init(&hdmaSpi1Tx) != HAL_OK) {
                NChip::ErrorCallback(__FILE__, __LINE__);
            }

            __HAL_LINKDMA(handler, hdmatx, hdmaSpi1Tx);
        } else if(handler->Instance==SPI2) {
            __HAL_RCC_SPI2_CLK_ENABLE();

            __HAL_RCC_GPIOB_CLK_ENABLE();
            /**SPI2 GPIO Configuration
            PB14     ------> SPI2_MISO
            PB15     ------> SPI2_MOSI
            PB13     ------> SPI2_SCK
            */
            GPIO_InitStruct.Pin = GPIO_PIN_14 | GPIO_PIN_15 | GPIO_PIN_13;
            GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
            GPIO_InitStruct.Pull = GPIO_NOPULL;
            GPIO_InitStruct.Speed = GPIO_SPEED_FAST;
            GPIO_InitStruct.Alternate = GPIO_AF5_SPI2;
            HAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
        } else if(handler->Instance == SPI5) {
            __HAL_RCC_SPI5_CLK_ENABLE();

            __HAL_RCC_GPIOE_CLK_ENABLE();
            /**SPI5 GPIO Configuration
            PE2     ------> SPI5_SCK
            PE5     ------> SPI5_MISO
            PE6     ------> SPI5_MOSI
            */
            GPIO_InitStruct.Pin = GPIO_PIN_2 | GPIO_PIN_5 | GPIO_PIN_6;
            GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
            GPIO_InitStruct.Pull = GPIO_NOPULL;
            GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_VERY_HIGH;
            GPIO_InitStruct.Alternate = GPIO_AF6_SPI5;
            HAL_GPIO_Init(GPIOE, &GPIO_InitStruct);
        }
    }

    void HalSpiMspDeinit(THandler* handler) {
        if(handler->Instance == SPI1) {
            __HAL_RCC_SPI1_CLK_DISABLE();

            /**SPI1 GPIO Configuration
            PA5     ------> SPI1_SCK
            PA6     ------> SPI1_MISO
            PA7     ------> SPI1_MOSI
            */
            HAL_GPIO_DeInit(GPIOA, GPIO_PIN_5 | GPIO_PIN_6 | GPIO_PIN_7);

            HAL_DMA_DeInit(handler->hdmarx);
            HAL_DMA_DeInit(handler->hdmatx);
        } else if(handler->Instance == SPI2) {
            __HAL_RCC_SPI2_CLK_DISABLE();

            /**SPI2 GPIO Configuration
            PB14     ------> SPI2_MISO
            PB15     ------> SPI2_MOSI
            PB13     ------> SPI2_SCK
            */
            HAL_GPIO_DeInit(GPIOB, GPIO_PIN_14 | GPIO_PIN_15 | GPIO_PIN_13);
        } else if(handler->Instance == SPI5) {
            __HAL_RCC_SPI5_CLK_DISABLE();

            /**SPI5 GPIO Configuration
            PE2     ------> SPI5_SCK
            PE5     ------> SPI5_MISO
            PE6     ------> SPI5_MOSI
            */
            HAL_GPIO_DeInit(GPIOE, GPIO_PIN_2 | GPIO_PIN_5 | GPIO_PIN_6);
        }
    }

    void HalSpiTxCompleteCallback(THandler* handler) {
        auto number = NChip::NSpi::GetNumber(handler);

        if (number != NChip::NSpi::ENumber::None) {
            NChip::NSpi::TransmitComplete(number);
        }
    }

    void HalSpiRxCOmpleteCallback(THandler* handler) {
        auto number = NChip::NSpi::GetNumber(handler);

        if (number != NChip::NSpi::ENumber::None) {
            NChip::NSpi::ReceiveComplete(number);
        }
    }

    void HalSpiErrorCallback(THandler* handler) {
        auto number = NChip::NSpi::GetNumber(handler);

        if (number != NChip::NSpi::ENumber::None) {
            NChip::NSpi::ErrorComplete(number);
        }
    }
}

extern "C" {
    void HAL_SPI_TxCpltCallback(SPI_HandleTypeDef *hspi) {
        HalSpiTxCompleteCallback(hspi);
    }

    void HAL_SPI_RxCpltCallback(SPI_HandleTypeDef *hspi) {
        HalSpiRxCOmpleteCallback(hspi);
    }

    void HAL_SPI_ErrorCallback(SPI_HandleTypeDef *hspi) {
        HalSpiErrorCallback(hspi);
    }

    void HAL_SPI_MspInit(SPI_HandleTypeDef* handler) {
        HalSpiMspInit(handler);
    }

    void HAL_SPI_MspDeInit(SPI_HandleTypeDef* handler) {
        HalSpiMspDeinit(handler);
    }

    void DMA2_Stream0_IRQHandler(void) {
        HAL_DMA_IRQHandler(&hdmaSpi1Rx);
    }

    void DMA2_Stream3_IRQHandler(void) {
        HAL_DMA_IRQHandler(&hdmaSpi1Tx);
    }
}
