#include <csp.h>
#include <csp_hal.h>

namespace {
    using THandler = UART_HandleTypeDef;
    using TInstance = USART_TypeDef;

    THandler huart7; // 1 -- debug
    THandler huart8; // 2 -- external
    THandler huart2; // 3 -- gsm
    THandler huart3; // 4 -- gps
    THandler huart6; // 5 -- lin

}

namespace NChip {
    namespace NUart {
        TInstance* GetInstance(ENumber number) {
            switch (number) {
                case ENumber::N1:
                    return UART7;
                case ENumber::N2:
                    return UART8;
                case ENumber::N3:
                    return USART2;
                case ENumber::N4:
                    return USART3;
                case ENumber::N5:
                    return USART6;
                default:
                    return nullptr;
            }
        }

        THandler* GetHandler(ENumber number) {
            switch (number) {
                case ENumber::N1:
                    return &huart7;
                case ENumber::N2:
                    return &huart8;
                case ENumber::N3:
                    return &huart2;
                case ENumber::N4:
                    return &huart3;
                case ENumber::N5:
                    return &huart6;
                default:
                    return nullptr;
            }
        }

        ENumber GetNumber(THandler* handler) {
            if (handler->Instance == UART7) {
                return ENumber::N1;
            } else if (handler->Instance == UART8) {
                return ENumber::N2;
            } else if (handler->Instance == USART2) {
                return ENumber::N3;
            } else if (handler->Instance == USART3) {
                return ENumber::N4;
            } else if (handler->Instance == USART6) {
                return ENumber::N5;
            }

            return ENumber::None;
        }

        void PreTransmit(ENumber number) {
            if (number == ENumber::N5) {
                HAL_LIN_SendBreak(&huart6);
            }
        }

        void PostReceive(ENumber number) {
            UNUSED(number);
        }
    }
}

bool NChip::NUart::Init(ENumber number, TSetting setting) {
    auto* handler = GetHandler(number);
    auto* instance = GetInstance(number);

    if (!handler or !instance) {
        return false;
    }

    handler->Instance = instance;
    handler->Init.BaudRate = static_cast<uint32_t>(setting.BaudRate);

    if (setting.WordLength == EWordLength::W8) {
        handler->Init.WordLength = UART_WORDLENGTH_8B;
    } else if (setting.WordLength == EWordLength::W9) {
        handler->Init.WordLength = UART_WORDLENGTH_9B;
    }

    if (setting.StopBit == EStopBit::S1) {
        handler->Init.StopBits = UART_STOPBITS_1;
    } else if (setting.StopBit == EStopBit::S2) {
        handler->Init.StopBits = UART_STOPBITS_2;
    }

    if (setting.Parity == EParity::None) {
        handler->Init.Parity = UART_PARITY_NONE;
    } else if (setting.Parity == EParity::Even) {
        handler->Init.Parity = UART_PARITY_EVEN;
    } else if (setting.Parity == EParity::Odd) {
        handler->Init.Parity = UART_PARITY_ODD;
    }

    handler->Init.Mode = UART_MODE_TX_RX;

    if (setting.FlowControl == EFlowControl::None) {
        handler->Init.HwFlowCtl = UART_HWCONTROL_NONE;
    } else if (setting.FlowControl == EFlowControl::Rts) {
        handler->Init.HwFlowCtl = UART_HWCONTROL_RTS;
    } else if (setting.FlowControl == EFlowControl::Cts) {
        handler->Init.HwFlowCtl = UART_HWCONTROL_CTS;
    } else if (setting.FlowControl == EFlowControl::RtsAndCts) {
        handler->Init.HwFlowCtl = UART_HWCONTROL_RTS_CTS;
    }

    handler->Init.OverSampling = UART_OVERSAMPLING_16;

    bool isInit = false;

    if (number == ENumber::N5) {
        isInit = (HAL_OK == HAL_LIN_Init(handler, UART_LINBREAKDETECTLENGTH_11B));
    } else {
        isInit = (HAL_OK == HAL_UART_Init(handler));
    }

    if (!isInit) {
        NChip::ErrorCallback(__FILE__, __LINE__);
        return false;
    }

    return true;
}

extern void UartInterruptHandler(THandler* handler);

namespace {
    void HalUartMspInit(THandler* handler) {
        auto number = NChip::NUart::GetNumber(handler);

        if (number == NChip::NUart::ENumber::None) {
            return;
        }

        GPIO_InitTypeDef GPIO_InitStruct;

        if (number == NChip::NUart::ENumber::N1) {
            __HAL_RCC_UART7_CLK_ENABLE();

            __HAL_RCC_GPIOE_CLK_ENABLE();
            /**UART7 GPIO Configuration
            PE8     ------> UART7_TX
            PE7     ------> UART7_RX
            */
            GPIO_InitStruct.Pin = GPIO_PIN_8|GPIO_PIN_7;
            GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
            GPIO_InitStruct.Pull = GPIO_PULLUP;
            GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
            GPIO_InitStruct.Alternate = GPIO_AF8_UART7;
            HAL_GPIO_Init(GPIOE, &GPIO_InitStruct);

            HAL_NVIC_SetPriority(UART7_IRQn, 11, 0);
            HAL_NVIC_EnableIRQ(UART7_IRQn);
        } else if (number == NChip::NUart::ENumber::N2) {
            __HAL_RCC_UART8_CLK_ENABLE();

            __HAL_RCC_GPIOE_CLK_ENABLE();
            /**UART8 GPIO Configuration
            PE1     ------> UART8_TX
            PE0     ------> UART8_RX
            */
            GPIO_InitStruct.Pin = GPIO_PIN_1|GPIO_PIN_0;
            GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
            GPIO_InitStruct.Pull = GPIO_NOPULL;
            GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_MEDIUM;
            GPIO_InitStruct.Alternate = GPIO_AF8_UART8;
            HAL_GPIO_Init(GPIOE, &GPIO_InitStruct);

            HAL_NVIC_SetPriority(UART8_IRQn, 12, 0);
            HAL_NVIC_EnableIRQ(UART8_IRQn);
        } else if (number == NChip::NUart::ENumber::N3) {
            __HAL_RCC_USART2_CLK_ENABLE();

            __HAL_RCC_GPIOD_CLK_ENABLE();
            /**USART2 GPIO Configuration
            PD6     ------> USART2_RX
            PD5     ------> USART2_TX
            PD4     ------> USART2_RTS
            PD3     ------> USART2_CTS
            */
            GPIO_InitStruct.Pin = GPIO_PIN_6|GPIO_PIN_5|GPIO_PIN_4|GPIO_PIN_3;
            GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
            GPIO_InitStruct.Pull = GPIO_NOPULL;
            GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
            GPIO_InitStruct.Alternate = GPIO_AF7_USART2;
            HAL_GPIO_Init(GPIOD, &GPIO_InitStruct);

            HAL_NVIC_SetPriority(USART2_IRQn, 12, 0);
            HAL_NVIC_EnableIRQ(USART2_IRQn);
        } else if (number == NChip::NUart::ENumber::N4) {
            __HAL_RCC_USART3_CLK_ENABLE();

            __HAL_RCC_GPIOC_CLK_ENABLE();
            /**USART3 GPIO Configuration
            PC11     ------> USART3_RX
            PC10     ------> USART3_TX
            */
            GPIO_InitStruct.Pin = GPIO_PIN_11;
            GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
            GPIO_InitStruct.Pull = GPIO_PULLUP;
            GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
            GPIO_InitStruct.Alternate = GPIO_AF7_USART3;
            HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

            GPIO_InitStruct.Pin = GPIO_PIN_10;
            GPIO_InitStruct.Mode = GPIO_MODE_AF_PP;
            GPIO_InitStruct.Pull = GPIO_NOPULL;
            GPIO_InitStruct.Speed = GPIO_SPEED_FREQ_HIGH;
            GPIO_InitStruct.Alternate = GPIO_AF7_USART3;
            HAL_GPIO_Init(GPIOC, &GPIO_InitStruct);

            HAL_NVIC_SetPriority(USART3_IRQn, 13, 0);
            HAL_NVIC_EnableIRQ(USART3_IRQn);
        }

        __HAL_UART_ENABLE_IT(handler, UART_IT_IDLE);
    }

    void HalUartMspDeinit(THandler* handler) {
        auto number = NChip::NUart::GetNumber(handler);

        if (number == NChip::NUart::ENumber::None) {
            return;
        }

        __HAL_UART_DISABLE_IT(handler, UART_IT_IDLE);

        if (number == NChip::NUart::ENumber::N1) {
            __HAL_RCC_UART7_CLK_DISABLE();

            /**UART7 GPIO Configuration
            PE8     ------> UART7_TX
            PE7     ------> UART7_RX
            */
            HAL_GPIO_DeInit(GPIOE, GPIO_PIN_8|GPIO_PIN_7);
            HAL_NVIC_DisableIRQ(UART7_IRQn);
        } else if (number == NChip::NUart::ENumber::N2) {
            __HAL_RCC_UART8_CLK_DISABLE();

            /**UART8 GPIO Configuration
            PE1     ------> UART8_TX
            PE0     ------> UART8_RX
            */
            HAL_GPIO_DeInit(GPIOE, GPIO_PIN_1|GPIO_PIN_0);
            HAL_NVIC_DisableIRQ(UART8_IRQn);
        } else if (number == NChip::NUart::ENumber::N3) {
            __HAL_RCC_USART2_CLK_DISABLE();

            /**USART2 GPIO Configuration
            PD6     ------> USART2_RX
            PD5     ------> USART2_TX
            PD4     ------> USART2_RTS
            PD3     ------> USART2_CTS
            */
            HAL_GPIO_DeInit(GPIOD, GPIO_PIN_6|GPIO_PIN_5|GPIO_PIN_4|GPIO_PIN_3);
            HAL_NVIC_DisableIRQ(USART2_IRQn);
        } else if (number == NChip::NUart::ENumber::N4) {
            __HAL_RCC_USART3_CLK_DISABLE();

            /**USART3 GPIO Configuration
            PC11     ------> USART3_RX
            PC10     ------> USART3_TX
            */
            HAL_GPIO_DeInit(GPIOC, GPIO_PIN_11|GPIO_PIN_10);

            /* USART3 interrupt Deinit */
            HAL_NVIC_DisableIRQ(USART3_IRQn);
        }
    }
}

extern "C" {
    void HAL_UART_MspInit(UART_HandleTypeDef* handler) {
        HalUartMspInit(handler);
    }

    void HAL_UART_MspDeInit(UART_HandleTypeDef* handler) {
        HalUartMspDeinit(handler);
    }

    void UART7_IRQHandler(void) {
        UartInterruptHandler(&huart7);
    }

    void UART8_IRQHandler(void) {
        UartInterruptHandler(&huart8);
    }

    void USART2_IRQHandler(void) {
        UartInterruptHandler(&huart2);
    }

    void USART3_IRQHandler(void) {
        UartInterruptHandler(&huart3);
    }
}
