#include <csp.h>
#include <csp_hal.h>

namespace {
    using TPin = uint16_t;
    using TDef = GPIO_TypeDef;

    TPin PinValue(NChip::NGpio::EPin pin) {
        return static_cast<TPin>(1 << static_cast<TPin>(pin));
    }
}

namespace NChip {
    namespace NGpio {
        void InitClock(EPort port);
        void DeinitClock(EPort port);
        TDef* PortToDef(EPort port);
        EPort DefToPort(TDef* def);
        uint32_t SpeedParse(ESpeed speed);
    }
}

void NChip::NGpio::Init(EPort port, EPin pin, EMode mode, ESpeed speed, EPullup pullup) {
    GPIO_InitTypeDef config;

    switch (mode) {
        case EMode::PushPull:
            config.Mode = GPIO_MODE_OUTPUT_PP;
            break;
        case EMode::OpenDrain:
            config.Mode = GPIO_MODE_OUTPUT_OD;
            break;
        case EMode::Input:
            config.Mode = GPIO_MODE_INPUT;
            break;
        case EMode::Rising:
            config.Mode = GPIO_MODE_IT_RISING;
            break;
        case EMode::Falling:
            config.Mode = GPIO_MODE_IT_FALLING;
            break;
        case EMode::RisingFalling:
            config.Mode = GPIO_MODE_IT_RISING_FALLING;
            break;
        default:
        case EMode::Analog:
            break;
    }

    config.Speed = SpeedParse(speed);

    switch (pullup) {
        default:
        case EPullup::None:
            config.Pull = GPIO_NOPULL;
            break;
        case EPullup::Up:
            config.Pull = GPIO_PULLUP;
            break;
        case EPullup::Down:
            config.Pull = GPIO_PULLDOWN;
            break;
    }

    config.Pin = PinValue(pin);

    InitClock(port);

    auto portDefine = PortToDef(port);

    if (!portDefine) {
        NChip::ErrorCallback(__FILE__, __LINE__);
    }

    HAL_GPIO_Init(portDefine, &config);
}

void NChip::NGpio::Deinit(EPort port, EPin pin) {
    auto pinDefine = PinValue(pin);
    auto portDefine = PortToDef(port);

    if (!portDefine) {
        NChip::ErrorCallback(__FILE__, __LINE__);
    }

    DeinitClock(port);

    HAL_GPIO_DeInit(portDefine, pinDefine);
}

void NChip::NGpio::On(EPort port, EPin pin) {
    auto pinDefine = PinValue(pin);
    auto portDefine = PortToDef(port);

    if (!portDefine) {
        NChip::ErrorCallback(__FILE__, __LINE__);
    }

    HAL_GPIO_WritePin(portDefine, pinDefine, GPIO_PIN_SET);
}

void NChip::NGpio::Off(EPort port, EPin pin) {
    auto pinDefine = PinValue(pin);
    auto portDefine = PortToDef(port);

    if (!portDefine) {
        NChip::ErrorCallback(__FILE__, __LINE__);
    }

    HAL_GPIO_WritePin(portDefine, pinDefine, GPIO_PIN_RESET);
}

void NChip::NGpio::Toggle(EPort port, EPin pin) {
    auto pinDefine = PinValue(pin);
    auto portDefine = PortToDef(port);

    if (!portDefine) {
        NChip::ErrorCallback(__FILE__, __LINE__);
    }

    HAL_GPIO_TogglePin(portDefine, pinDefine);
}

bool NChip::NGpio::State(EPort port, EPin pin) {
    auto pinDefine = PinValue(pin);
    auto portDefine = PortToDef(port);

    if (!portDefine) {
        NChip::ErrorCallback(__FILE__, __LINE__);
    }

    return HAL_GPIO_ReadPin(portDefine, pinDefine) == GPIO_PIN_SET;
}

void NChip::NGpio::SoftwareInterrupt(EPin pin) {
    __HAL_GPIO_EXTI_GENERATE_SWIT(PinValue(pin));
}

__weak void NChip::NGpio::LineHandler(EPin pin) {
    UNUSED(pin);
}
