#include <csp.h>
#include <csp_hal.h>

namespace {
    using THandler = UART_HandleTypeDef;
    constexpr uint16_t DefaultTimeout = 10;
}

namespace NChip {
    namespace NUart {
        THandler* GetHandler(ENumber number);
        ENumber GetNumber(THandler* handler);
        void InitInterrupt(ENumber number);
        void DeinitInterrupt(ENumber number);

        void PreTransmit(ENumber number);
        void PostReceive(ENumber number);
    }
}

bool NChip::NUart::Deinit(ENumber number) {
    auto* handler = NChip::NUart::GetHandler(number);

    if (!handler) {
        NChip::ErrorCallback(__FILE__, __LINE__);
        return false;
    }

    if (HAL_UART_DeInit(handler) != HAL_OK) {
        return false;
    }

    return true;
}

bool NChip::NUart::Transmit(ENumber number, EMode mode, const uint8_t* data, size_t size) {
    auto* handler = GetHandler(number);

    if (!handler) {
        return false;
    }

    HAL_StatusTypeDef result = HAL_ERROR;

    PreTransmit(number);

    if (mode == EMode::Simple) {
        result = HAL_UART_Transmit(handler, const_cast<uint8_t*>(data), size, size * DefaultTimeout);
        NChip::NUart::TransmitCallback(number);
    } else if (mode == EMode::Interrupt) {
        result = HAL_UART_Transmit_IT(handler, const_cast<uint8_t*>(data), size);
    }

    return result == HAL_OK;
}

bool NChip::NUart::Receive(ENumber number, EMode mode, uint8_t* data, size_t size) {
    auto* handler = GetHandler(number);

    if (!handler) {
        return false;
    }

    HAL_StatusTypeDef result = HAL_ERROR;

    if (mode == EMode::Simple) {
        result = HAL_UART_Receive(handler, data, size, size * DefaultTimeout);
        NChip::NUart::ReceiveCallback(number, size);
    } else if (mode == EMode::Interrupt) {
        result = HAL_UART_Receive_IT(handler, data, size);
    }

    return result == HAL_OK;
}

bool NChip::NUart::AbortReceive(ENumber number, EMode mode) {
    auto* handler = GetHandler(number);

    if (!handler) {
        return false;
    }

    HAL_StatusTypeDef result = HAL_ERROR;

    if (mode == EMode::Interrupt) {
        result = HAL_UART_AbortReceive_IT(handler);
    }

    return result == HAL_OK;
}

bool NChip::NUart::AbortTransmit(ENumber number, EMode mode) {
    auto* handler = GetHandler(number);

    if (!handler) {
        return false;
    }

    HAL_StatusTypeDef result = HAL_ERROR;

    if (mode == EMode::Interrupt) {
        result = HAL_UART_AbortTransmit_IT(handler);
    }

    return result == HAL_OK;
}

__weak void NChip::NUart::TransmitCallback(ENumber number) {
    (void)number;
}

__weak void NChip::NUart::ReceiveCallback(ENumber number, size_t size) {
    (void)number;
    (void)size;
}

__weak void NChip::NUart::ReceiveErrorCallback(ENumber number) {
    (void)number;
}

void UartInterruptHandler(THandler* handler) {
    HAL_UART_IRQHandler(handler);
    auto number = NChip::NUart::GetNumber(handler);

    if (number != NChip::NUart::ENumber::None) {
        if (RESET != __HAL_UART_GET_FLAG(handler, UART_FLAG_IDLE)) {
            size_t size = handler->RxXferSize - handler->RxXferCount;
            if (size > 0) {
                NChip::NUart::ReceiveCallback(number, size);
            }

            __HAL_UART_CLEAR_IDLEFLAG(handler);
        }

        NChip::NUart::PostReceive(number);
    }
}

namespace
{
    void HalUartTxCompleteCallback(THandler* handler) {
        auto number = NChip::NUart::GetNumber(handler);

        if (number == NChip::NUart::ENumber::None) {
            NChip::ErrorCallback(__FILE__, __LINE__);
            return;
        }

        NChip::NUart::TransmitCallback(number);
    }

    void HalUsartRxErrorCallback(THandler* handler) {
        auto number = NChip::NUart::GetNumber(handler);

        if (number == NChip::NUart::ENumber::None) {
            NChip::ErrorCallback(__FILE__, __LINE__);
            return;
        }

        NChip::NUart::ReceiveErrorCallback(number);
    }
}

extern "C" {
    void HAL_UART_TxCpltCallback(UART_HandleTypeDef *huart) {
        HalUartTxCompleteCallback(huart);
    }

    void HAL_UART_ErrorCallback(UART_HandleTypeDef *huart) {
        HalUsartRxErrorCallback(huart);
    }
}
