/*
** ============================================================================
** FILE			sf_sense_moving.c
** DESCRIPTION		Contains all subroutine for LIS3DH chip
**
** TERGET		STM32L151
** CREATED		Solovyev A.F. Vega-Absolute Ltd.
** DATA			14.12.2012
** ============================================================================
*/

/* Includes ------------------------------------------------------------------*/
#include <stdbool.h>

#include <accelerator/legacy/config.h>
#include <accelerator/legacy/Accel_move.h>
#include <accelerator/legacy/Accel_filtr.h>

/* Private variables ---------------------------------------------------------*/
//       КОРРЕКТИРОВКА АЛГОРИТМА ПОД LIS331DLH
// Исходные настройки датчика движения под LIS3DH
// MS_MAX_STEP_DLT = 6
// MovingThreshold[1] = 230 // максимальный порог
// MovingThreshold[15] = 80 // минимальный порог
// Под LIS331DLH изменены пороги, так как показалось что датчик движения стал
// менене чувствительным. Размеры буферов не поменялись. Так как пробовал уменьшить
// размер буферов, это привело к увеличению чувствительности к ударам, вч-шумам.

/* Defines -------------------------------------------------------------------*/
#define MS_BUF_SIZE_SMPL                100 //120       // 100  // ~200 ms
#define MS_BUF_SIZE_MA1                  80//100        // 80   // ~200 ms
#define MS_BUF_SIZE_MA2                  80 // 80       //80   // ~200 ms
//#define MS_BUF_SIZE_MA3                80 // 80       // ~200 ms
#define MS_PERIOD_GET_SAMPLES            4        // 10 ms - период выборки семплов для анализа
//#define MS_BUF_DELTA_SIZE                80       // Буфер отфильтрованных от ВЧ семплов с периодом дескретизации 10 ms
#define MS_BUF_SIZE_DLT                  40       // размер буфера дельт

#define MS_DELAY_CHECK_THRESHOLD    	80 // 2080 dots // 5200 ms
#define MS_DELAY_TO_READY_BUF     	10

//ACCEL_MOVE_DELAY_TO_STOP
//#define MS_DELAY_TO_STOP		5*400 // 20*400 = 20s * 400kHz

//#define THRESHOLD_MIN                    72

//- Настраиваемые параметры пороги чувствительноти для LIS331DLH, LIS3DH
// =0 - отключен, [1] - низкая чувствительность - [15] - высокая чувствительность
#ifdef ACCEL_MEMS_TYPE_LIS331DLH  //(ACCEL_LIS_MEMS == LIS331DLH)
 #define MS_MAX_STEP_DLT                  5        // максимальный размер дельты
 //#define THRESHOLD_MAX                    ((MS_BUF_SIZE_DLT * MS_MAX_STEP_DLT)-8) // 40*5=200-8=192
 const uint16_t MovingThreshold[16] = {  0, 190, 175, 165, 157, 149, 141, 133,
   				       125, 117, 109, 101,  94,  87,  81,  75,};
#elif defined(ACCEL_MEMS_TYPE_LIS3DH)  //(ACCEL_LIS_MEMS == LIS3DH)
 #define MS_MAX_STEP_DLT                  6 //3//6        // максимальный размер дельты
 //#define THRESHOLD_MAX                    ((MS_BUF_SIZE_DLT * MS_MAX_STEP_DLT)-8) // 40*6=240-8=132
// const uint16_t MovingThreshold[16] = {  0, 240, 220, 205, 191, 178, 168, 158,
//  				       148, 138, 128, 118, 108,  98,  89,  80,};
 const uint16_t MovingThreshold[16] = {  0, 240, 220, 205, 191, 178, 168, 158,
  				       148, 138, 128, 118, 108,  98,  80,  80,};   // 98, 89, 80 // 20-40-80  idx=14
  	// 20-40 сработка на писк зона на антенне 60 - в конкретном месте антенны, 80 - маленькая зона
#else
 //#define MS_MAX_STEP_DLT                  5        // максимальный размер дельты
 //#define THRESHOLD_MAX                    ((MS_BUF_SIZE_DLT * MS_MAX_STEP_DLT)-8) // 40*5=200-8=192
 //const uint16_t MovingThreshold[16] = {0,0,0,0,  0,0,0,0,  0,0,0,0,  0,0,0,0 };
#endif


/* Global variables ----------------------------------------------------------*/
static uint8_t AccelMoveSenseLevel = 14; 	// уровень чувствительности датчика движения-
static uint8_t AccelMoveState = 0;		// Состояние движения

/* Private variables ---------------------------------------------------------*/
 typedef enum {RESET=0, SET=1} FlagStatus;
 typedef struct {
  FlagStatus climb; 	// текущее направления движения
  FlagStatus ready; 	// готовность к корректировке коэффициента движения путем вычитания старых дельт
  uint8_t count; 	// счетчик семплов, двигающихся в одном направлении
  uint16_t sum; 	// коэффициент движения, сумма дельт двигающихся в одном направлении
  uint16_t previous; 	// предыдущая дельта
  uint16_t delta; 	// буфер для временного хранения дельты, модуля разницы между текущим и предыдущим значением потока данных
} GROWTH_CTRL;
GROWTH_CTRL GrowthCtrlX;
GROWTH_CTRL GrowthCtrlY;
GROWTH_CTRL GrowthCtrlZ;

//- Буфера дельт
uint16_t DELTA_BUF_X[MS_BUF_SIZE_DLT];
uint16_t DELTA_BUF_Y[MS_BUF_SIZE_DLT];
uint16_t DELTA_BUF_Z[MS_BUF_SIZE_DLT];

//- ВХОДНЫЕ СЭМПЛЫ
ACCEL_POINT_16BIT MS_BufSmpl[MS_BUF_SIZE_SMPL]; // буфер входных семплев
ACCEL_POINT_CTRL MS_CtrlSmpl; // управляющая структура входных семплов
//- СКОЛЬЗЯЩАЯ СРЕДНЯЯ 1
ACCEL_POINT_16BIT MS_ItemMa1;
ACCEL_POINT_16BIT MS_BufMa1[MS_BUF_SIZE_MA1];
ACCEL_POINT_CTRL MS_CtrlMa1;
//- СКОЛЬЗЯЩАЯ СРЕДНЯЯ 2
ACCEL_POINT_16BIT MS_ItemMa2;
ACCEL_POINT_16BIT MS_BufMa2[MS_BUF_SIZE_MA2];
ACCEL_POINT_CTRL MS_CtrlMa2;
//- СКОЛЬЗЯЩАЯ СРЕДНЯЯ 3
ACCEL_POINT_16BIT MS_ItemMa3;
//ACCEL_POINT_16BIT MS_BufMa3[MS_BUF_SIZE_MA3];
//ACCEL_POINT_CTRL MS_CtrlMa3;
//- СКОЛЬЗЯЩАЯ СРЕДНЯЯ 4
//ACCEL_POINT_16BIT MS_ItemMa4;

ACCEL_POINT_16BIT MS_Q[2]; // буфер двух соседних точек
//ACCEL_POINT_1BIT S_FLUX_DIRECTION_UP; // состояние - направление движения потока
//ACCEL_POINT_1BIT S_BUF_FULL; // состояние - буфер заполнен
//ACCEL_POINT_16BIT MS_CurrentSum; // текущая сумма на интервале
//ACCEL_POINT_8BIT MS_FluxCount; // счетчик сэмплев двигающихся в одном направлении
//ACCEL_POINT_8BIT MS_BufDelta[MS_BUF_DELTA_SIZE]; // буфер отклонений
//ACCEL_POINT_16BIT MS_IntervalSwing; // отклонение на интервале одного направления

static uint8_t Count_MsSamples = 0x00; // делитель частоты семплов
static uint16_t Count_MsDelayLoadSamples = 0x0000; // Счетчик задержки включения датчика
static uint8_t MS_StartCount = 0x00;
static uint32_t Count_MsDelayCheckStop = 0; // DEF_ACCEL_MOVE_DELAY_TO_STOP;
static uint32_t AccelMoveDelayToStop =  DEF_ACCEL_MOVE_DELAY_TO_STOP;

/* Function prototypes -------------------------------------------------------*/
void MovingSensor_Service(void);
void MovingSensor_Reset(void);
bool MovingSensor_CheckThreshold(void);
//void search_direction(void);
void growth_calc(uint16_t prev, uint16_t cur, GROWTH_CTRL* _growth_ctrl, uint16_t* in_buf, uint8_t in_buf_size, uint8_t in_max_delta_step);
void growth_reset(GROWTH_CTRL* _growth_ctrl);

/* Functions -----------------------------------------------------------------*/
//- Init
void AccelMove_Init(void)
{
  AccelMoveState=0;
  MovingSensor_Reset();
}

void SetAccelMoveDelayToStop(uint16_t seconds)
{
  AccelMoveDelayToStop=seconds*ACCEL_FRAME_RATE_kHZ;
}

uint16_t GetAccelMoveDelayToStop(void)
{
  return AccelMoveDelayToStop/ACCEL_FRAME_RATE_kHZ;
}


//- Обработчик для датчика движения
void AccelMove_Do(ACCEL_POINT_16BIT AcceluSmpl)
{
 // Check enable if necessary here
 // Check to stop
  if(Count_MsDelayCheckStop>0)  {Count_MsDelayCheckStop--;}
  else AccelMoveState=0;

 // Задержка включения датчика
  if(Count_MsDelayLoadSamples>0)    {
      Count_MsDelayLoadSamples--;
  }
  else {
     // ОБРАБОТКА ФИЛЬТРОМ
      if(Accel_math_ma(AcceluSmpl, &MS_ItemMa1, &MS_CtrlSmpl, MS_BufSmpl, MS_BUF_SIZE_SMPL) == true)
      {
        if(Accel_math_ma(MS_ItemMa1, &MS_ItemMa2, &MS_CtrlMa1, MS_BufMa1, MS_BUF_SIZE_MA1) == true)
        {
          if(Accel_math_ma(MS_ItemMa2, &MS_ItemMa3, &MS_CtrlMa2, MS_BufMa2, MS_BUF_SIZE_MA2) == true)
          {
// if(Accel_math_ma(MS_ItemMa3, &MS_ItemMa4, &MS_CtrlMa3, MS_BufMa3, MS_BUF_SIZE_MA3) == true)
//{
           // ВЫБОРКА СЕМПЛОВ КАЖДЫЕ 10ms
            Count_MsSamples++;
            if(Count_MsSamples >= MS_PERIOD_GET_SAMPLES)    // 10ms
            {
              Count_MsSamples = 0x00;
              MS_Q[0] = MS_Q[1];
              MS_Q[1] = MS_ItemMa3; // MS_ItemMa3,4
              if(MS_StartCount == 0x00)
              {
                if(MovingSensor_CheckThreshold() == true)
                {
		  AccelMoveState=1;
		  Count_MsDelayCheckStop = AccelMoveDelayToStop;
                  MovingSensor_Reset();

                }
              }
              else  {  MS_StartCount--; }  // Задержка на старт
            }
          }//math_ma
        }//math_ma
      }//math_ma
//}
  }//else
}


//- Поиск порога срабатывания датчика движения
bool MovingSensor_CheckThreshold(void)
{
  growth_calc(MS_Q[0].X, MS_Q[1].X, &GrowthCtrlX, DELTA_BUF_X, MS_BUF_SIZE_DLT, MS_MAX_STEP_DLT);
  growth_calc(MS_Q[0].Y, MS_Q[1].Y, &GrowthCtrlY, DELTA_BUF_Y, MS_BUF_SIZE_DLT, MS_MAX_STEP_DLT);
  growth_calc(MS_Q[0].Z, MS_Q[1].Z, &GrowthCtrlZ, DELTA_BUF_Z, MS_BUF_SIZE_DLT, MS_MAX_STEP_DLT);

 uint16_t  threshold = MovingThreshold[AccelMoveSenseLevel];
  if(GrowthCtrlX.sum >= threshold) {return(true);}
  if(GrowthCtrlY.sum >= threshold) {return(true);}
  if(GrowthCtrlZ.sum >= threshold) {return(true);}
  return(false);
}


//- Сброс всех фильтров и переменных
void MovingSensor_Reset(void)
{
  Accel_math_reset(&MS_CtrlSmpl);
  Accel_math_reset(&MS_CtrlMa1);
  Accel_math_reset(&MS_CtrlMa2);
  growth_reset(&GrowthCtrlX);
  growth_reset(&GrowthCtrlY);
  growth_reset(&GrowthCtrlZ);
  Count_MsSamples = 0x00;
  Count_MsDelayLoadSamples = MS_DELAY_CHECK_THRESHOLD;
  MS_StartCount = MS_DELAY_TO_READY_BUF; // первая точка с фильтра не обработана
}


//- Изменение чувствительности датчика движения
void MovingSensor_ChangeSenseLevel(uint8_t _level)
{
  if(_level < 16)
  {
    AccelMoveSenseLevel = _level;  // изменение чувствительности датчика движения
    MovingSensor_Reset();  // Перезапуск сервиса датчика движения
  }
}

//- Запрос чувствительности датчика движения
uint8_t MovingSensor_GetSenseLevel(void)
{
  return AccelMoveSenseLevel;
}

//- Запрос состояния датчика движения
uint8_t MovingSensor_GetState(void)
{
  return AccelMoveState;
}

//==============================================================================
// Вычисление коэффициента роста
// Входные данные:      prev - предыдущая точка потока данных
//                      cur - текущая точка потока данных
//                      _growth_ctrl - управляющая структура функции
//                      in_buf[] - буфер дельт, т.е. буфер для хранения модуля разницы между текущей и предыдущей точками потока
//                      in_buf_size - размер буфера дельт
//                      in_max_delta_step - ограничитель размера дельты, максимальное значение которое может иметь дельта
void growth_calc(uint16_t prev, uint16_t cur, GROWTH_CTRL* _growth_ctrl, uint16_t* in_buf, uint8_t in_buf_size, uint8_t in_max_delta_step)
{
  // КОНТРОЛЬ НАПРАВЛЕНИЯ
  if(_growth_ctrl->climb == SET) // рост вверх
  {
    if(prev > cur)
    {
      _growth_ctrl->climb = RESET;
      _growth_ctrl->delta = prev - cur;
      growth_reset(_growth_ctrl);
    }
    else // cur >= prev
    {
      _growth_ctrl->delta = cur - prev;
    }
  }
  else // climp == RESET // рост вниз
  {
    if(prev < cur)
    {
      _growth_ctrl->climb = SET;
      _growth_ctrl->delta = cur - prev;
      growth_reset(_growth_ctrl);
    }
    else // cur >= prev
    {
      _growth_ctrl->delta = prev-cur;
    }
  }

  // КОНТРОЛЬ ВЕЛИЧИНЫ ДЕЛЬТЫ
  if(_growth_ctrl->delta >= in_max_delta_step)
  {
    _growth_ctrl->delta = in_max_delta_step;
  }

  // ВЫЧИСЛЕНИЕ ВЕЛИЧИНЫ РОСТА
  in_buf[_growth_ctrl->count] = _growth_ctrl->delta;      // записать входную дельту в буфер дельт
  _growth_ctrl->sum = _growth_ctrl->sum + _growth_ctrl->delta - _growth_ctrl->previous; // набор суммы дельт
  _growth_ctrl->count++;
  if(_growth_ctrl->count >= in_buf_size)
  {
    _growth_ctrl->count = 0;
    _growth_ctrl->ready = SET;
  }
  if(_growth_ctrl->ready == SET)
  {
    _growth_ctrl->previous = in_buf[_growth_ctrl->count];
  }
}


//- Cброс функции вычисление коэффициента движения
void growth_reset(GROWTH_CTRL* _growth_ctrl)
{
  _growth_ctrl->delta = 0;

  _growth_ctrl->ready = RESET;
  _growth_ctrl->previous = 0;
  _growth_ctrl->count = 0;
  _growth_ctrl->sum = 0;
}

