/*
** ============================================================================
** FILE			Accel_tilt.c
** DESCRIPTION		Contains all subroutine for LIS3DH chip
**
** TERGET		STM32L151
** CREATED		Solovyev A.F. Vega-Absolute Ltd.
** DATA			14.12.2012
** ============================================================================
*/

/* Includes ------------------------------------------------------------------*/
#include <stdbool.h>

#include <accelerator/legacy/Accel_tilt.h>
#include <accelerator/legacy/config.h>

/* Defines -------------------------------------------------------------------*/
//#define TS_REAL_THRESHOLD                     0
//#define TS_CALIBRATION_THRESHOLD              1
#define TS_CUT_HF_THRESHOLD                     36

// Внимание!
// TS_COUNT_OUTSIDE_THRESHOLD > в 2 раза чем TS_BUF_SIZE_SMPL + TS_BUF_SIZE_MA1 + TS_BUF_SIZE_MA2 + TS_BUF_SIZE_MA3

#define TS_BUF_SIZE_SMPL                 50       // 750 ms
#define TS_BUF_SIZE_MA1                  70       // 1040 ms
#define TS_BUF_SIZE_MA2                  90       // 1350 ms
#define TS_BUF_SIZE_MA3                  110      // 1650 ms
#define TS_BUF_SIZE_AVERAGE              50       // 5000 ms - поиск среднего уровня
#define TS_COUNT_OUTSIDE_THRESHOLD       500      // 4 секунды - счетчик нахождения вне порога
#define TS_PERIOD_GET_SAMPLES            6        // 15 ms
#define TS_PERIOD_SEARCH_AVERAGE         7        // 105 ms период взятия точек для вычисления среднего базового уровня

//#define TILT_THR_STEP                  1        // шаг изменения чувствительности
#define TILT_THR_LOW                     5        // высокая чувствительность
#define TS_CALIBRATION_PERIOD            600      // каждые 30 секунд перекалибровка

//ACCEL_TILT_DELAY_TO_STOP
//#define TS_DELAY_TO_STOP		5*400 // 30*400 = 30s * 400kHz

//// Настраиваемые параметры //////
const uint32_t TiltThreshold[8] = {
  0,    // 0              датчик выключен
  18,   // 1
  15,   // 2
  13,   // 3
  11,   // 4
  9,    // 5
  7,    // 6
  TILT_THR_LOW,    // 7               высокая чувствительность
};


/* Global variables ----------------------------------------------------------*/
static uint8_t AccelTiltSenseLevel = 3; 	// уровень чувствительности датчика наклона. 3 - 1градус
static uint8_t AccelTiltState = 0;		// Состояние датчика наклона

extern ACCEL_POINT_16BIT AcceluSmpl; // входной uint16 семпл

/* Private variables ---------------------------------------------------------*/
typedef enum {RESET=0, SET=1} FlagStatus;
typedef enum {
  tss_SetThreshold,
  tss_CheckThreshold,
  tss_CheckImmobility,
  tss_SendAlarm,
} TILT_SENSOR_STATE;
TILT_SENSOR_STATE TiltSensorState = tss_SetThreshold; // состояние автомата алгоритма анализа


//- ВХОДНЫЕ СЭМПЛЫ
ACCEL_POINT_16BIT CmpSmpl; // входной семпл
ACCEL_POINT_16BIT TS_BufSmpl[TS_BUF_SIZE_SMPL]; // буфер входных семплев
ACCEL_POINT_CTRL TS_CtrlSmpl; // управляющая структура входных семплов
//- СКОЛЬЗЯЩАЯ СРЕДНЯЯ 1
ACCEL_POINT_16BIT TS_ItemMa1;
ACCEL_POINT_16BIT TS_BufMa1[TS_BUF_SIZE_MA1];
ACCEL_POINT_CTRL TS_CtrlMa1;
//- СКОЛЬЗЯЩАЯ СРЕДНЯЯ 2
ACCEL_POINT_16BIT TS_ItemMa2;
ACCEL_POINT_16BIT TS_BufMa2[TS_BUF_SIZE_MA2];
ACCEL_POINT_CTRL TS_CtrlMa2;
//- СКОЛЬЗЯЩАЯ СРЕДНЯЯ 3
ACCEL_POINT_16BIT TS_ItemMa3;
ACCEL_POINT_16BIT TS_BufMa3[TS_BUF_SIZE_MA3];
ACCEL_POINT_CTRL TS_CtrlMa3;
//- СКОЛЬЗЯЩАЯ СРЕДНЯЯ 4
ACCEL_POINT_16BIT TS_ItemMa4;
//- ВЫЧИСЛЕНИЕ AVERAGE
ACCEL_POINT_16BIT TS_StartAverage;
ACCEL_POINT_16BIT TS_BufAvrg[TS_BUF_SIZE_AVERAGE];
ACCEL_POINT_CTRL TS_CtrlAvrg;


ACCEL_POINT_16BIT TS_CalibrationThresholdLevelUpper; // верхний порог канала калибровки
ACCEL_POINT_16BIT TS_CalibrationThresholdLevelLower; // нижний порог канала калибровки
ACCEL_POINT_16BIT TS_ThresholdLevelUpper;   		// верхний порог канала движения средней арифметической
ACCEL_POINT_16BIT TS_ThresholdLevelLower;  		// нижний порог канала движения средней арифметической

static uint8_t Count_TsSamples = 0x00;				// период взятия семплов для анализа алгоритмом
static uint8_t Count_TsSearchAverage = 0x00; 			// период взятия семплов для усреднения и получения базового уровня
static FlagStatus F_TS_FAST_NEXT_STATE = RESET; 		// флаг быстрого переключения в автомате анализа точек
static uint16_t Count_OutsideThreshold = 0x0000; 		// счетчик точек вышедших за границу валидного канала

static FlagStatus SA_TS_CALIBRATION_TRESHOLD = RESET;
static FlagStatus SA_TS_CHECK_TRESHOLD = RESET;
static FlagStatus SA_AVERAGE_LEVEL_IS_READY = RESET;
static uint16_t Count_TiltSensorCalibrationPeriod = 0;
static uint16_t Count_TsDelayCheckStop = 0; // ACCEL_TILT_DELAY_TO_STOP;

/* Function prototypes -------------------------------------------------------*/
void TiltSensor_CutHighFrequencySurge(ACCEL_POINT_16BIT* _in_val);
void ts_CutAxis(uint16_t* _in_dot, uint16_t* _in_average);

void TiltSensor_SetTheshold(ACCEL_POINT_16BIT* _in_avr, uint8_t _in_delta);
void TiltSensor_SetCalibrationTheshold(ACCEL_POINT_16BIT* _in_avr, uint8_t _in_delta);
bool TiltSensor_CheckTheshold(ACCEL_POINT_16BIT* _in_val);
bool TiltSensor_CheckCalibrationTheshold(ACCEL_POINT_16BIT* _in_val);
void CalibrationTiltSensor_Count(void);
void TiltSensor_CalibrationLaunch(void);
void TiltSensor_ReLaunch(void);
void TiltSensor_Reset(void);

/* Functions -----------------------------------------------------------------*/
//- Init
void AccelTilt_Init(void)
{
  AccelTiltState=0;
  TiltSensor_Reset();
}



//- Фильтр от высокочастотных пичков, ударов, звуковых хлопков
// Обрезает высокочастотные ипульсы порогами канала образуемого средней линией канала
// Сначала с помощью фильтра НЧ "math_ma" вычисляется средняя линия канала (уставка),
// после того как срений уровень найден вверх и вниз от него выставляются пороги,
// аплитудное превышение которых входными семплами запрещено.
// Если входной семпл выше порога, то значению семпла перед тем как попасть
// в фильтр низкой частоты "math_ma" присваевается значение соответствующего порога.
void TiltSensor_CutHighFrequencySurge(ACCEL_POINT_16BIT* _in_val)
{
  ts_CutAxis(&_in_val->X, &TS_ItemMa4.X);
  ts_CutAxis(&_in_val->Y, &TS_ItemMa4.Y);
  ts_CutAxis(&_in_val->Z, &TS_ItemMa4.Z);
}

//==============================================================================
void ts_CutAxis(uint16_t* _in_dot, uint16_t* _in_average)
{
  uint16_t TS_CHFS_Upper;
  uint16_t TS_CHFS_Lower;

  TS_CHFS_Upper = *_in_average + TS_CUT_HF_THRESHOLD;
  TS_CHFS_Lower = *_in_average - TS_CUT_HF_THRESHOLD;


  if(*_in_dot > TS_CHFS_Upper)
  {
    *_in_dot = TS_CHFS_Upper;
  }
  else if(*_in_dot < TS_CHFS_Lower)
  {
    *_in_dot = TS_CHFS_Lower;
  }
}


//- Обработка датчика наклона
        // Калибровка выполняется автоматически периодически для обновления
        // среднего уровня неподвижного состояния. Средний уровень двигается
        // в зависимости от температуры (температурный дрейф). Средний уровень
        // может двигаться если машина стоит на снегу или льду который тает.
        // Или же спускает колесо. Тем самым датчик не должен срабатывать на
        // медленные изменения наклона. Это достигается путем каллибровки.
        // В то же время датчик не должен реагировать на покачивания автомабиля,
        // и удары (например раскачивание ветром). Для подавления каких-либо сигналов
        // ударов применяется полосовой фильтр, для подавления раскачиваний
        // используется алгоритм допустимого таймаута пребывания вне рабочего
        // канала. Рабочий канал - канал границами которого являются пороги
        // срабатывания датчика наклона
void AccelTilt_Do(ACCEL_POINT_16BIT AcceluSmpl)
{
 // Check to stop
  if(Count_TsDelayCheckStop>0)  {Count_TsDelayCheckStop--;}
  else AccelTiltState=0;

    Count_TsSamples++;
    if(Count_TsSamples>= TS_PERIOD_GET_SAMPLES)    // 15 ms
    {
      Count_TsSamples = 0x00;
      CmpSmpl = AcceluSmpl;
     // ИЗБАВЛЯЕМСЯ ОТ ВЫСОКИХ ЧАСТОТ После того как фильтр выдал средний уровень, включаем дополнительный алгоритм
     // отфильтровывания (удаления высоких частот)
      if(SA_AVERAGE_LEVEL_IS_READY == SET) { TiltSensor_CutHighFrequencySurge(&CmpSmpl); }
     // ОБРАБОТКА ФИЛЬТРОМ
      if(Accel_math_ma(CmpSmpl, &TS_ItemMa1, &TS_CtrlSmpl, TS_BufSmpl, TS_BUF_SIZE_SMPL) == true)
      {
        if(Accel_math_ma(TS_ItemMa1, &TS_ItemMa2, &TS_CtrlMa1, TS_BufMa1, TS_BUF_SIZE_MA1) == true)
        {
          if(Accel_math_ma(TS_ItemMa2, &TS_ItemMa3, &TS_CtrlMa2, TS_BufMa2, TS_BUF_SIZE_MA2) == true)
          {
            if(Accel_math_ma(TS_ItemMa3, &TS_ItemMa4, &TS_CtrlMa3, TS_BufMa3, TS_BUF_SIZE_MA3) == true)
            {
              SA_AVERAGE_LEVEL_IS_READY = SET;  // средний уровень вычислен, применить дополнительный предварительную функцию обрезки пичков каналом
             // ПРОВЕРКА ПОРОГА ДАТЧИКА НАКЛОНА
              if(SA_TS_CHECK_TRESHOLD == SET)
              {
                if(TiltSensor_CheckTheshold(&TS_ItemMa4) == true)
                {
                  Count_OutsideThreshold++;
                  if(Count_OutsideThreshold>TS_COUNT_OUTSIDE_THRESHOLD)
                  {
		    AccelTiltState=1; Count_TsDelayCheckStop = DEF_ACCEL_TILT_DELAY_TO_STOP; // MS_DELAY_TO_STOP;
		    Count_OutsideThreshold = 0x0000;  	// Порог превышен!!!
                    TiltSensor_Reset();
                  }
                }
                else { if(Count_OutsideThreshold != 0)
		          Count_OutsideThreshold = 0x0000; }  // вернулись в пороги
              }

              // КАЛИБРОВКА СРЕДНЕГО УРОВНЯ
              if(SA_TS_CALIBRATION_TRESHOLD == SET)
              {
                do
                { F_TS_FAST_NEXT_STATE = RESET;
                  switch(TiltSensorState)
                  {
                   // (2) ПРОВЕРКА НА НЕПОДВИЖНОСТЬ
                    case tss_CheckImmobility:
                     // AVERAGE В ДОПУСТИМОМ КАНАЛЕ
                      if(TiltSensor_CheckCalibrationTheshold(&TS_ItemMa4) == false)
                      {
                        Count_TsSearchAverage++;
                        if(Count_TsSearchAverage >= TS_PERIOD_SEARCH_AVERAGE)    // 15 ms
                        {
                          Count_TsSearchAverage = 0x00;
                          if(Accel_math_ma(TS_ItemMa4, &TS_StartAverage, &TS_CtrlAvrg, TS_BufAvrg, TS_BUF_SIZE_AVERAGE) == true)
                          {
                            TiltSensor_SetTheshold(&TS_StartAverage, TiltThreshold[AccelTiltSenseLevel]); // установить новые пороги
                            SA_TS_CALIBRATION_TRESHOLD = RESET;    // закончить калибровку
                            SA_TS_CHECK_TRESHOLD = SET;            // начать боевое отслеживание порога
                            TiltSensorState = tss_SetThreshold;    // начать проверку превышения границ канала
                          }
                        }
                      }
                      else // ПРЕВЫШЕН ПОРОГ
                      {
                        F_TS_FAST_NEXT_STATE = SET;
                        TiltSensorState = tss_SetThreshold;    // установить новый канал порогов
                      }
                    break;

                   // (1) УСТАНОВКА ПОРОГОВ КАЛИБРОВОЧНОГО КАНАЛА
                    case tss_SetThreshold:
                      TiltSensor_SetCalibrationTheshold(&TS_ItemMa4, TiltThreshold[7]); // установить канал поиска среднего
                      Accel_math_reset(&TS_CtrlAvrg);  		// сбосить параметры поиска
                      Count_TsSearchAverage = 0x00;
                      TiltSensorState = tss_CheckImmobility;
                      break;

                    default: break;
                  }
                }while(F_TS_FAST_NEXT_STATE == SET);

              }

            }
          }
        }
      }

    }   // Count_Samples
}


//- Отсчет таймаута перекалибровки
void CalibrationTiltSensor_Count(void)
{
  Count_TiltSensorCalibrationPeriod++;
  if(Count_TiltSensorCalibrationPeriod == TS_CALIBRATION_PERIOD)
  {
    Count_TiltSensorCalibrationPeriod = 0;
    TiltSensor_CalibrationLaunch();
  }
}


//- Установка порогов сработки
void TiltSensor_SetTheshold(ACCEL_POINT_16BIT* _in_avr, uint8_t _in_delta)
{
  TS_ThresholdLevelUpper.X = _in_avr->X+_in_delta;
  TS_ThresholdLevelLower.X = _in_avr->X-_in_delta;
  TS_ThresholdLevelUpper.Y = _in_avr->Y+_in_delta;
  TS_ThresholdLevelLower.Y = _in_avr->Y-_in_delta;
  TS_ThresholdLevelUpper.Z = _in_avr->Z+_in_delta;
  TS_ThresholdLevelLower.Z = _in_avr->Z-_in_delta;
}

//- Установка порогов калибровки
void TiltSensor_SetCalibrationTheshold(ACCEL_POINT_16BIT* _in_avr, uint8_t _in_delta)
{
  TS_CalibrationThresholdLevelUpper.X = _in_avr->X+_in_delta;
  TS_CalibrationThresholdLevelLower.X = _in_avr->X-_in_delta;
  TS_CalibrationThresholdLevelUpper.Y = _in_avr->Y+_in_delta;
  TS_CalibrationThresholdLevelLower.Y = _in_avr->Y-_in_delta;
  TS_CalibrationThresholdLevelUpper.Z = _in_avr->Z+_in_delta;
  TS_CalibrationThresholdLevelLower.Z = _in_avr->Z-_in_delta;
}


//- Проверка порогов сработки
bool TiltSensor_CheckTheshold(ACCEL_POINT_16BIT* _in_val)
{
  if((_in_val->X > TS_ThresholdLevelUpper.X) || (_in_val->X < TS_ThresholdLevelLower.X))
  {
    return(true);
  }
  if((_in_val->Y > TS_ThresholdLevelUpper.Y) || (_in_val->Y < TS_ThresholdLevelLower.Y))
  {
    return(true);
  }
  if((_in_val->Z > TS_ThresholdLevelUpper.Z) || (_in_val->Z < TS_ThresholdLevelLower.Z))
  {
    return(true);
  }
  return(false);
}


//- Проверка порогов калибровки
bool TiltSensor_CheckCalibrationTheshold(ACCEL_POINT_16BIT* _in_val)
{
  if((_in_val->X > TS_CalibrationThresholdLevelUpper.X) || (_in_val->X < TS_CalibrationThresholdLevelLower.X))
  {
    return(true);
  }
  if((_in_val->Y > TS_CalibrationThresholdLevelUpper.Y) || (_in_val->Y < TS_CalibrationThresholdLevelLower.Y))
  {
    return(true);
  }
  if((_in_val->Z > TS_CalibrationThresholdLevelUpper.Z) || (_in_val->Z < TS_CalibrationThresholdLevelLower.Z))
  {
    return(true);
  }
  return(false);
}


// Перезапуск датчика наклона, без пересброса фильтров
void TiltSensor_CalibrationLaunch(void)
{
  SA_TS_CALIBRATION_TRESHOLD = SET; 	// включить калибровку, определение новых пороговых уровней
  TiltSensorState = tss_SetThreshold;   // начать с установки калибровочных пороговых уровней
  Accel_math_reset(&TS_CtrlAvrg);
  Count_TsSearchAverage = 0;
}


//- Перезапуск датчика наклона, без пересброса фильтров
void TiltSensor_ReLaunch(void)
{
  SA_TS_CHECK_TRESHOLD = RESET; 	// выключить проверку порога
  SA_TS_CALIBRATION_TRESHOLD = SET; 	// включить калибровку, определение новых пороговых уровней
  TiltSensorState = tss_SetThreshold;   // начать с установки калибровочных пороговых уровней
  Accel_math_reset(&TS_CtrlAvrg);
  Count_OutsideThreshold = 0x0000;
  Count_TsSearchAverage = 0;
  Count_TiltSensorCalibrationPeriod = 0; // сбросить счетчик периода калибровки
}


//- Полный сброс датчика наклона
void TiltSensor_Reset(void)
{
  Accel_math_reset(&TS_CtrlSmpl);
  Accel_math_reset(&TS_CtrlMa1);
  Accel_math_reset(&TS_CtrlMa2);
  Accel_math_reset(&TS_CtrlMa3);
  SA_AVERAGE_LEVEL_IS_READY = RESET;
  TiltSensor_ReLaunch();
}

//- Запрос состояния датчика наклона
uint8_t AccelTilt_GetState(void)
{
  return AccelTiltState;
}
