#pragma once

#include <cstddef>

namespace NLibrary {
    namespace NData {
        template<class TItem, size_t Length>
        class TQueue {
        public:
            TQueue();
            ~TQueue() = default;

            bool IsEmpty() const;
            size_t GetSize() const;
            bool IsFull() const {
                return Full;
            }
            size_t GetCapacity() const {
                return Length;
            }

            void Push(const TItem& item);
            TItem Pop();
            bool Peek(TItem& item) const;
            void Clear();

        private:
            TItem Data[Length];
            size_t Head;
            size_t Tail;
            bool Full;
        };

        template <class TItem, size_t Length>
        TQueue<TItem, Length>::TQueue()
            : Data()
            , Head(0)
            , Tail(0)
            , Full(false)
        {
        }

        template <class TItem, size_t Length>
        bool TQueue<TItem, Length>::IsEmpty() const {
            return (!IsFull() && (Head == Tail));
        }

        template <class TItem, size_t Length>
        size_t TQueue<TItem, Length>::GetSize() const {
            size_t size = GetCapacity();

            if (!IsFull()) {
                if (Head >= Tail) {
                    size = Head - Tail;
                } else {
                    size = GetCapacity() + Head - Tail;
                }
            }

            return size;
        }

        template <class TItem, size_t Length>
        void TQueue<TItem, Length>::Push(const TItem& item) {
            if (IsFull()) {
                return;
            }

            Data[Head] = item;
            Head = (Head + 1) % GetCapacity();
            Full = Head == Tail;
        }

        template <class TItem, size_t Length>
        TItem TQueue<TItem, Length>::Pop() {
            if (IsEmpty()) {
                return TItem();
            }

            auto result = Data[Tail];
            Full = false;
            Tail = (Tail + 1) % GetCapacity();

            return result;
        }

        template <class TItem, size_t Length>
        bool TQueue<TItem, Length>::Peek(TItem& item) const {
            if (IsEmpty()) {
                return false;
            }
            item = Data[Tail];
            return true;
        }

        template <class TItem, size_t Length>
        void TQueue<TItem, Length>::Clear() {
            Head = Tail;
            Full = false;
        }
    }
}
