#pragma once

#include <algorithm>
#include <cstddef>
#include <cstring>

namespace NLibrary {
    namespace NData {

        template<size_t Length>
        class TString {
        public:
            static const char Terminator = '\0';

        public:
            using iterator = char*;
            using const_iterator = const char*;

        public:
            TString();
            explicit TString(const char* data);
            template<size_t OtherLength>
            TString(const TString<OtherLength>& other);

            ~TString() = default;

            size_t GetSize() const {
                return Size;
            }
            size_t GetFree() const {
                return Length - Size;
            }
            bool IsFull() const {
                return Size == Length;
            }
            bool IsEmpty() const {
                return Size == 0;
            }
            void Clear() {
                Size = 0;
                AddTerminator();
            }
            const char* AsString() const {
                return Data;
            }
            const uint8_t* AsBytes() const {
                return (const uint8_t*)(Data);
            }

            iterator begin() {
                return &Data[0];
            }
            const_iterator begin() const {
                return &Data[0];
            }
            iterator end() {
                return &Data[Size];
            }
            const_iterator end() const {
                return &Data[Size];
            }

            void Append(char symbol);
            void Append(const char* data);
            template<size_t OtherLength>
            void Append(const TString<OtherLength>& other);

            bool StartsWith(char symbol) const;
            bool StartsWith(const char* data) const;
            template <size_t OtherLength>
            bool StartsWith(const TString<OtherLength>& other) const;

            bool Equal(char symbol) const;
            bool Equal(const char* data) const;
            template<size_t OtherLength>
            bool Equal(const TString<OtherLength>& other) const;

        private:
            char Data[Length + sizeof(Terminator)];
            size_t Size;

        private:
            void AddTerminator();
            void Copy(const char* data, size_t size);
            bool Contains(const char* data, size_t size);
        };


        template<size_t Length>
        TString<Length>::TString()
            : Data()
            , Size(0)
        {
        }

        template<size_t Length>
        TString<Length>::TString(const char* data)
            : Data()
            , Size(0)
        {
            Copy(data, strlen(data));
        }

        template<size_t Length>
        template<size_t OtherLength>
        TString<Length>::TString(const TString<OtherLength>& other)
            : Data()
            , Size(0)
        {
            Copy(other.AsString(), other.GetSize());
        }

        template<size_t Length>
        void TString<Length>::Append(char symbol) {
            if (IsFull()) {
                return;
            }

            Data[Size++] = symbol;
            AddTerminator();
        }

        template<size_t Length>
        void TString<Length>::Append(const char* data) {
            size_t length = strlen(data);
            size_t copySize = std::min(GetFree(), length);

            memcpy(&Data[Size], data, copySize);
            Size += copySize;

            AddTerminator();
        }

        template<size_t Length>
        template<size_t OtherLength>
        void TString<Length>::Append(const TString<OtherLength>& other) {
            size_t copySize = std::min(GetFree(), other.GetSize());

            memcpy(&Data[Size], other.AsString(), copySize);
            Size += copySize;

            AddTerminator();
        }

        template<size_t Length>
        bool TString<Length>::StartsWith(char symbol) const {
            if (IsEmpty()) {
                return false;
            }

            return Data[0] == symbol;
        }

        template<size_t Length>
        bool TString<Length>::StartsWith(const char* data) const {
            if (!data || IsEmpty()) {
                return false;
            }

            size_t length = strlen(data);
            if (GetSize() < length) {
                return false;
            }

            return strncmp(AsString(), data, length) == 0;
        }

        template<size_t Length>
        template <size_t OtherLength>
        bool TString<Length>::StartsWith(const TString<OtherLength>& other) const {
            if (IsEmpty() || other.IsEmpty()) {
                return false;
            }
            return strncmp(AsString(), other.AsString(), other.GetSize()) == 0;
        }

        template<size_t Length>
        bool TString<Length>::Equal(char symbol) const {
            if (IsEmpty() || GetSize() > 1) {
                return false;
            }
            return Data[0] == symbol;
        }

        template<size_t Length>
        bool TString<Length>::Equal(const char* data) const {
            size_t length = strlen(data);

            if (length == 0 && IsEmpty()) {
                return true;
            }

            return strcmp(AsString(), data) == 0;
        }

        template<size_t Length>
        template<size_t OtherLength>
        bool TString<Length>::Equal(const TString<OtherLength>& other) const {
            if (other.IsEmpty() && IsEmpty()) {
                return true;
            }

            return strcmp(AsString(), other.AsString()) == 0;
        }

        template <size_t Length>
        void TString<Length>::AddTerminator() {
            Data[Size] = Terminator;
        }

        template <size_t Length>
        void TString<Length>::Copy(const char* data, size_t size) {
            size_t copySize = std::min(Length, size);
            memcpy(Data, data, copySize);
            Size = copySize;
            AddTerminator();
        }

        template <size_t Length>
        bool TString<Length>::Contains(const char* data, size_t size) {
            if (!data || size == 0 || IsEmpty()) {
                return false;
            }

            for (size_t i = 0; i < GetSize(); ++i) {
                if (Data[i] == data[i] && GetSize() >= (i + size) && (strncmp(&Data[i], data, size) == 0)) {
                    return true;
                }
            }

            return false;
        }



    }
}

template<size_t Length>
inline bool operator==(NLibrary::NData::TString<Length>& lhs, char rhs) {
    return lhs.Equal(rhs);
}

template <size_t Length>
inline bool operator==(char lhs, const NLibrary::NData::TString<Length>& rhs) {
    return rhs.Equal(lhs);
}

template <size_t Length>
inline bool operator==(const NLibrary::NData::TString<Length>& lhs, const char* rhs) {
    return lhs.Equal(rhs);
}

template <size_t Length>
inline bool operator==(const char* lhs, const NLibrary::NData::TString<Length>& rhs) {
    return rhs.Equal(lhs);
}

template <size_t LeftLength, size_t RightLength>
inline bool operator==(const NLibrary::NData::TString<LeftLength>& lhs, const NLibrary::NData::TString<RightLength>& rhs) {
    return lhs.Equal(rhs);
}
