#pragma once

#include <algorithm>
#include <cstddef>
#include <span>

namespace NLibrary {
    namespace NData {
        template<typename TItem, size_t Length>
        class TVector {
        public:
            using iterator = TItem*;
            using const_iterator = const TItem*;

        public:
            TVector();
            ~TVector() = default;

            TVector(const TItem* data, size_t size);
            TVector(std::initializer_list<TItem> data);
            template<size_t OtherLength>
            TVector(const TVector<TItem, OtherLength>& other);
            template<size_t OtherLength>
            TVector& operator=(const TVector<TItem, OtherLength>& other);

            const TItem* GetData() const {
                return Items;
            }
            size_t GetSize() const {
                return Size;
            }
            inline size_t GetCapacity() const {
                return Length;
            }
            bool IsFull() const {
                return GetSize() == GetCapacity();
            }
            bool IsEmpty() const {
                return GetSize() == 0;
            }

            void PushBack(const TItem& item);
            void PopBack();

            void Clear() {
                Size = 0;
            }

            iterator begin() {
                return &Items[0];
            }
            iterator end() {
                return &Items[Size];
            }
            const_iterator begin() const {
                return &Items[0];
            }
            const_iterator end() const {
                return &Items[Size];
            }
            iterator data() {
                return Items;
            }
            const_iterator data() const {
                return Items;
            }
            void Advance(size_t size) {
                Size += size;
                if (Size > GetCapacity()) {
                    Size = GetCapacity();
                }
            }

            TItem& operator[](size_t index);
            TItem& operator[](int index);
            const TItem& operator[](size_t index) const;
            const TItem& operator[](int index) const;

        private:
            TItem Items[Length];
            size_t Size = 0;

        private:
            void Copy(const TItem* items, size_t size);
        };

        template<typename TItem, size_t Length>
        TVector<TItem, Length>::TVector()
            : Items()
            , Size(0)
        {
        }


        template<typename TItem, size_t Length>
        TVector<TItem, Length>::TVector(const TItem* data, size_t size)
            : Items()
            , Size(0)
        {
            Copy(data, size);
        }

        template<typename TItem, size_t Length>
        template<size_t OtherLength>
        TVector<TItem, Length>::TVector(const TVector<TItem, OtherLength>& other)
            : Items()
            , Size(0)
        {
            Copy(other.begin(), other.GetSize());
        }

        template<typename TItem, size_t Length>
        template<size_t OtherLength>
        TVector<TItem, Length>& TVector<TItem, Length>::operator=(const TVector<TItem, OtherLength>& other) {
            Copy(other.begin(), other.GetSize());
            return *this;
        }

        template<typename TItem, size_t Length>
        void TVector<TItem, Length>::PushBack(const TItem& item) {
            if (IsFull()) {
                return;
            }
            Items[Size++] = item;
        }

        template<typename TItem, size_t Length>
        void TVector<TItem, Length>::PopBack() {
            if (IsEmpty()) {
                return;
            }
            Size--;
        }

        template<typename TItem, size_t Length>
        TItem& TVector<TItem, Length>::operator[](size_t index) {
            if (index > (Size - 1)) {
                return Items[0];
            }
            return Items[index];
        }

        template<typename TItem, size_t Length>
        const TItem& TVector<TItem, Length>::operator[](size_t index) const {
            if (index >= Size) {
                return Items[0];
            }
            return Items[index];
        }

        template<typename TItem, size_t Length>
        TItem& TVector<TItem, Length>::operator[](int index) {
            if (index < 0 || index >= static_cast<int>(Size)) {
                return Items[0];
            }
            return Items[index];
        }

        template<typename TItem, size_t Length>
        const TItem& TVector<TItem, Length>::operator[](int index) const {
            if (index < 0 || index >= Size) {
                return Items[0];
            }
            return Items[index];
        }

        template<typename TItem, size_t Length>
        void TVector<TItem, Length>::Copy(const TItem* items, size_t size) {
            size_t copySize = std::min(GetCapacity(), size);
            for (size_t i = 0; i < copySize; ++i) {
                Items[i] = items[i];
            }
            Size = copySize;
        }

        template<typename TItem, size_t Length>
        TVector<TItem, Length>::TVector(std::initializer_list<TItem> data) {
            for (auto& item : data) {
                PushBack(item);
            }
        }

        template<size_t Length>
        using TBuffer = TVector<uint8_t, Length>;
        using TBufferView = std::span<uint8_t>;
    }
}
