#pragma once

// LIS3DH driver

#include "registers.h"

#include <data/vector.h>
#include <lock/lock.h>
#include <stream/stream.h>
#include <util/point.h>
#include <util/util.h>

namespace NLibrary {
    namespace NLis {
        enum ESampleRate: uint8_t {
            SR_HZ_0 = 0b0000,
            SR_HZ_1 = 0b0001,
            SR_HZ_10 = 0b0010,
            SR_HZ_25 = 0b0011,
            SR_HZ_50 = 0b0100,
            SR_HZ_100 = 0b0101,
            SR_HZ_200 = 0b0110,
            SR_HZ_400 = 0b0111,
            SR_HZ_1600 = 0b1000,
            SR_HZ_5000 = 0b1001,
        };

        enum ERange: uint8_t {
            RNG_2 = 0x0,
            RNG_4 = 0x1,
            RNG_8 = 0x2,
            RNG_16 = 0x3,
        };

        enum class EMode: uint8_t {
            Normal,
            Sleep
        };

        struct TSetting {
            EMode Mode = EMode::Normal;
            ESampleRate SampleRate = SR_HZ_0;
            ERange Range = RNG_2;
        };

        using TPoint = NLibrary::NUtil::TPoint<int16_t>;
        using TPointF = NLibrary::NUtil::TPoint<float>;

        template<class TProvider>
        class TDevice {
        public:
            using TProviderPtr = TProvider*;

        public:
            TDevice() = default;
            virtual ~TDevice() = default;

            uint8_t DeviceId() {
                return Read<uint8_t>(R_WHO_AM_I);
            }

            TPoint GetLastPoint() const {
                return LastPoint;
            }

            TPoint GetRawPoint() {
                TPoint result({
                    Read<int16_t>(R_OUT_X_L),
                    Read<int16_t>(R_OUT_Y_L),
                    Read<int16_t>(R_OUT_Z_L)
                });

                LastPoint = result;
                LastPointTime = NLibrary::NTime::TTime::Now();

                return result;
            }

            EMode GetMode() const {
                return Mode;
            }

            ESampleRate GetSampleRate() const {
                return SampleRate;
            }

            ERange GetRange() const {
                return Range;
            }

            TPointF GetPoint() {
                return Transform(GetRawPoint());
            }

            void Reset() {
                Write(R_CTRL_5, BOOT);
            }

            bool SetNormalMode(ESampleRate sampleRate, ERange range) {
                Mode = EMode::Normal;
                SampleRate = sampleRate;
                Range = range;

                if (!Write(R_CTRL_1, XEN | YEN | ZEN | (sampleRate << 4))) {
                    return false;
                }
                if (!Write(R_CTRL_2, 0)) {
                    return false;
                }
                if (!Write(R_CTRL_3, I1_DRDY1)) {
                    return false;
                }
                if (!Write(R_CTRL_4, BDU | (range << 4) | HR)) {
                    return false;
                }
                if (!Write(R_CTRL_5, FIFO_EN)) {
                    return false;
                }
                if (!Write(R_CTRL_6, H_LACTIVE)) {
                    return false;
                }
                return true;
            }

            bool SetSleepMode(ESampleRate sampleRate, ERange range) {
                Mode = EMode::Sleep;
                SampleRate = sampleRate;
                Range = range;

                const uint16_t thMg = 52;
                const uint16_t fsMg = 2000;
                const uint16_t durationMs = 200;

                if (!Write(R_CTRL_1, XEN | YEN | ZEN | (sampleRate << 4))) {
                    return false;
                }
                if (!Write(R_CTRL_2, HPM1 | HPM0 | HPIS1)) {
                    return false;
                }
                if (!Write(R_CTRL_3, I1_AOI1)) {
                    return false;
                }
                if (!Write(R_CTRL_4, (range << 4) | HR)) {
                    return false;
                }
                if (!Write(R_CTRL_5, 0)) {
                    return false;
                }
                if (!Write(R_INT1_THS, IntTh(thMg, fsMg))) {
                    return false;
                }
                if (!Write(R_INT1_DUR, DurationTime(durationMs, GetValueOfSampleRate(sampleRate)))) {
                    return false;
                }

                GetRawPoint(); // Dummy read

                if (!Write(R_CTRL_6, H_LACTIVE)) {
                    return false;
                }
                if (!Write(R_INT1_CFG, ZHIE | YHIE | XHIE)) {
                    return false;
                }

                GetRawPoint(); // Dummy read
                return true;
            }

        private:
            EMode Mode = EMode::Normal;
            ESampleRate SampleRate = SR_HZ_0;
            ERange Range = RNG_2;

            TPoint LastPoint;
            NLibrary::NTime::TTime LastPointTime;

        private:
            TProviderPtr GetProvider() {
                return TProvider::GetInstance();
            }

            uint16_t GetValueOfSampleRate(ESampleRate sampleRate) {
                switch (sampleRate) {
                    case SR_HZ_0:
                        return 0;
                    case SR_HZ_1:
                        return 1;
                    case SR_HZ_10:
                        return 10;
                    case SR_HZ_25:
                        return 25;
                    case SR_HZ_50:
                        return 50;
                    case SR_HZ_100:
                        return 100;
                    case SR_HZ_200:
                        return 200;
                    case SR_HZ_400:
                        return 400;
                    case SR_HZ_1600:
                        return 1600;
                    case SR_HZ_5000:
                        return 5000;
                    default:
                        return 50;
                }
            }

            bool Write(ERegister reg, uint8_t value) {
                uint8_t dataToSend[] = {
                    reg,
                    value
                };

                auto provider = GetProvider();

                {
                    NLibrary::NLock::TLock lock(provider->GetLock());
                    provider->Transmit(dataToSend, sizeof(dataToSend));
                }

                return Read<uint8_t>(reg) == value;
            }

            template<typename TData>
            TData Read(ERegister reg) {
                TData result;
                constexpr size_t resultSize = sizeof(TData);

                uint8_t data = reg;
                data |= (1 << 7); // Set bit for read request

                if constexpr (resultSize > 1) {
                    data |= (1 << 6); // Set bit for auto increment
                }

                auto provider = GetProvider();

                {
                    NLibrary::NLock::TLock lock(provider->GetLock());
                    provider->TransmitReceive(&data, sizeof(data), reinterpret_cast<uint8_t*>(&result), resultSize);
                }

                return result;
            }
        };
    }
}

template<size_t Length>
inline NLibrary::NStream::IStringOutput<Length>& operator<<(NLibrary::NStream::IStringOutput<Length>& stream, const NLibrary::NLis::TPoint& point) {
    using namespace NLibrary::NUtil;
    stream << ToString(int32_t(point.X)) << ' ' << ToString(int32_t(point.Y)) << ' ' << ToString(int32_t(point.Z));
    return stream;
}

template<size_t Length>
inline NLibrary::NStream::IStringOutput<Length>& operator<<(NLibrary::NStream::IStringOutput<Length>& stream, const NLibrary::NLis::TPointF& point) {
    using namespace NLibrary::NUtil;
    stream << ToString(point.X, 6) << ' ' << ToString(point.Y, 6) << ' ' << ToString(point.Z, 6);
    return stream;
}
