#pragma once

#include <cstddef>
#include <cstdint>

namespace NLibrary {
    namespace NLis {
        enum ERegister: uint8_t {
            /*!
             *  STATUS_REG_AUX register
             *   321OR  1, 2 and 3 axis data overrun. Default value: 0
             *          (0: no overrun has occurred; 1: a new set of data has overwritten
             * the previous ones) 3OR    3 axis data overrun. Default value: 0 (0: no
             * overrun has occurred; 1: a new data for the 3-axis has overwritten the
             * previous one) 2OR    2 axis data overrun. Default value: 0 (0: no overrun has
             * occurred; 1: a new data for the 4-axis has overwritten the previous one) 1OR
             * 1 axis data overrun. Default value: 0 (0: no overrun has occurred; 1: a new
             * data for the 1-axis has overwritten the previous one) 321DA  1, 2 and 3 axis
             * new data available. Default value: 0 (0: a new set of data is not yet
             * available; 1: a new set of data is available) 3DA:   3 axis new data
             * available. Default value: 0 (0: a new data for the 3-axis is not yet
             * available; 1: a new data for the 3-axis is available) 2DA:   2 axis new data
             * available. Default value: 0 (0: a new data for the 2-axis is not yet
             * available; 1: a new data for the 2-axis is available) 1DA    1 axis new data
             * available. Default value: 0 (0: a new data for the 1-axis is not yet
             * available; 1: a new data for the 1-axis is available)
             */
            R_STATUS_1 = 0x07,
            R_OUT_ADC_1_L = 0x08,
            R_OUT_ADC_1_H = 0x09,
            R_OUT_ADC_2_L = 0x0A,
            R_OUT_ADC_2_H = 0x0B,
            R_OUT_ADC_3_L = 0x0C,
            R_OUT_ADC_3_H = 0x0D,

            R_INT_COUNTER = 0x0E, /**< INT_COUNTER register [IC7, IC6, IC5, IC4, IC3, IC2, IC1, IC0] */
            R_WHO_AM_I = 0x0F, /**< Device identification register. [0, 0, 1, 1, 0, 0, 1, 1] */

            /*!
             *  TEMP_CFG_REG
             *  Temperature configuration register.
             *   ADC_PD   ADC enable. Default value: 0
             *            (0: ADC disabled; 1: ADC enabled)
             *   TEMP_EN  Temperature sensor (T) enable. Default value: 0
             *            (0: T disabled; 1: T enabled)
             */
            R_TEMP_CONFIGE = 0x1F,

            /*!
             *  CTRL_REG1
             *  [ODR3, ODR2, ODR1, ODR0, LPen, Zen, Yen, Xen]
             *   ODR3-0  Data rate selection. Default value: 00
             *           (0000:50 Hz; Others: Refer to Datasheet Table 26, “Data rate
             * configuration”) LPen    Low power mode enable. Default value: 0 (0: normal
             * mode, 1: low power mode) Zen     Z axis enable. Default value: 1 (0: Z axis
             * disabled; 1: Z axis enabled) Yen     Y axis enable. Default value: 1 (0: Y
             * axis disabled; 1: Y axis enabled) Xen     X axis enable. Default value: 1 (0:
             * X axis disabled; 1: X axis enabled)
             */
            R_CTRL_1 = 0x20,

            /*!
             *  CTRL_REG2
             *  [HPM1, HPM0, HPCF2, HPCF1, FDS, HPCLICK, HPIS2, HPIS1]
             *   HPM1-0  High pass filter mode selection. Default value: 00
             *           Refer to Table 29, "High pass filter mode configuration"
             *   HPCF2-1 High pass filter cut off frequency selection
             *   FDS     Filtered data selection. Default value: 0
             *					 (0: internal filter bypassed; 1: data
             *from internal filter sent to output register and FIFO) HPCLICK High pass
             *filter enabled for CLICK function. (0: filter bypassed; 1: filter enabled)
             *HPIS2   X axis enable. Default value: 1 (0: X axis disabled; 1: X axis
             *enabled) HPIS1 High pass filter enabled for AOI function on interrupt 1, (0:
             *filter bypassed; 1: filter enabled)
             */
            R_CTRL_2 = 0x21,

            /*!
             *  CTRL_REG3
             *  [I1_CLICK, I1_AOI1, I1_AOI2, I1_DRDY1, I1_DRDY2, I1_WTM, I1_OVERRUN, --]
             *   I1_CLICK    CLICK interrupt on INT1. Default value 0.
             *						   (0: Disable; 1: Enable)
             *   I1_AOI1     AOI1 interrupt on INT1. Default value 0.
             *						   (0: Disable; 1: Enable)
             *   I1_AOI2     AOI2 interrupt on INT1. Default value 0.
             *               (0: Disable; 1: Enable)
             *   I1_DRDY1    DRDY1 interrupt on INT1. Default value 0.
             *               (0: Disable; 1: Enable)
             *   I1_DRDY2    DRDY2 interrupt on INT1. Default value 0.
             *               (0: Disable; 1: Enable)
             *   I1_WTM      FIFO Watermark interrupt on INT1. Default value 0.
             *               (0: Disable; 1: Enable)
             *   I1_OVERRUN  FIFO Overrun interrupt on INT1. Default value 0.
             * 							 (0: Disable; 1: Enable)
             */
            R_CTRL_3 = 0x22,

            /*!
             *  CTRL_REG4
             *  [BDU, BLE, FS1, FS0, HR, ST1, ST0, SIM]
             *   BDU      Block data update. Default value: 0
             *            (0: continuos update; 1: output registers not updated until MSB
             * and LSB reading) BLE      Big/little endian data selection. Default value 0.
             *            (0: Data LSB @ lower address; 1: Data MSB @ lower address)
             *   FS1-FS0  Full scale selection. default value: 00
             *            (00: +/- 2G; 01: +/- 4G; 10: +/- 8G; 11: +/- 16G)
             *   HR       High resolution output mode: Default value: 0
             *            (0: High resolution disable; 1: High resolution Enable)
             *   ST1-ST0  Self test enable. Default value: 00
             *            (00: Self test disabled; Other: See Table 34)
             *   SIM      SPI serial interface mode selection. Default value: 0
             *            (0: 4-wire interface; 1: 3-wire interface).
             */
            R_CTRL_4 = 0x23,

            /*!
             *  CTRL_REG5
             *  [BOOT, FIFO_EN, --, --, LIR_INT1, D4D_INT1, 0, 0]
             *   BOOT     Reboot memory content. Default value: 0
             *            (0: normal mode; 1: reboot memory content)
             *   FIFO_EN  FIFO enable. Default value: 0
             *            (0: FIFO disable; 1: FIFO Enable)
             *   LIR_INT1 Latch interrupt request on INT1_SRC register, with INT1_SRC
             * register cleared by reading INT1_SRC itself. Default value: 0. (0: interrupt
             * request not latched; 1: interrupt request latched) D4D_INT1 4D enable: 4D
             * detection is enabled on INT1 when 6D bit on INT1_CFG is set to 1.
             */
            R_CTRL_5 = 0x24,

            /*!
             *  CTRL_REG6
             *  [I2_CLICKen, I2_INT1, 0, BOOT_I1, 0, --, H_L, -]
             */
            R_CTRL_6 = 0x25,

            R_REFERENCE = 0x26, /**< REFERENCE/DATACAPTURE **/

            /*!
             *  STATUS_REG
             *  [ZYXOR, ZOR, YOR, XOR, ZYXDA, ZDA, YDA, XDA]
             *   ZYXOR    X, Y and Z axis data overrun. Default value: 0
             *            (0: no overrun has occurred; 1: a new set of data has overwritten
             * the previous ones) ZOR      Z axis data overrun. Default value: 0 (0: no
             * overrun has occurred; 1: a new data for the Z-axis has overwritten the
             * previous one) YOR      Y axis data overrun. Default value: 0 (0: no overrun
             * has occurred;  1: a new data for the Y-axis has overwritten the previous one)
             *   XOR      X axis data overrun. Default value: 0
             *            (0: no overrun has occurred; 1: a new data for the X-axis has
             * overwritten the previous one) ZYXDA    X, Y and Z axis new data available.
             * Default value: 0 (0: a new set of data is not yet available; 1: a new set of
             * data is available) ZDA      Z axis new data available. Default value: 0 (0: a
             * new data for the Z-axis is not yet available; 1: a new data for the Z-axis is
             * available) YDA      Y axis new data available. Default value: 0 (0: a new
             * data for the Y-axis is not yet available; 1: a new data for the Y-axis is
             * available)
             */
            R_STATUS_2 = 0x27,

            R_OUT_X_L = 0x28, /**< X-axis acceleration data. Low value */
            R_OUT_X_H = 0x29, /**< X-axis acceleration data. High value */
            R_OUT_Y_L = 0x2A, /**< Y-axis acceleration data. Low value */
            R_OUT_Y_H = 0x2B, /**< Y-axis acceleration data. High value */
            R_OUT_Z_L = 0x2C, /**< Z-axis acceleration data. Low value */
            R_OUT_Z_H = 0x2D, /**< Z-axis acceleration data. High value */

            /*!
             *  FIFO_CTRL_REG
             *  [FM1, FM0, TR, FTH4, FTH3, FTH2, FTH1, FTH0]
             *   FM1-FM0  FIFO mode selection. Default value: 00 (see Table 44)
             *   TR       Trigger selection. Default value: 0
             *            0: Trigger event liked to trigger signal on INT1
             *            1: Trigger event liked to trigger signal on INT2
             *   FTH4:0   Default value: 0
             */
            R_FIFO_CTRL = 0x2E,

            R_FIFO_SRC = 0x2F, /**< FIFO_SRC_REG [WTM, OVRN_FIFO, EMPTY, FSS4, FSS3, FSS2, FSS1, FSS0] */

            /*!
             *  INT1_CFG
             *  [AOI, 6D, ZHIE/ZUPE, ZLIE/ZDOWNE, YHIE/YUPE, XHIE/XUPE, XLIE/XDOWNE]
             *   AOI         And/Or combination of Interrupt events. Default value: 0. Refer
             * to Datasheet Table 48, "Interrupt mode" 6D          6 direction detection
             * function enabled. Default value: 0. Refer to Datasheet Table 48, "Interrupt
             * mode" ZHIE/ZUPE   Enable interrupt generation on Z high event or on Direction
             * recognition. Default value: 0. (0: disable interrupt request; 1: enable
             * interrupt request) ZLIE/ZDOWNE Enable interrupt generation on Z low event or
             * on Direction recognition. Default value: 0. YHIE/YUPE   Enable interrupt
             * generation on Y high event or on Direction recognition. Default value: 0. (0:
             * disable interrupt request; 1: enable interrupt request.) YLIE/YDOWNE Enable
             * interrupt generation on Y low event or on Direction recognition. Default
             * value: 0. (0: disable interrupt request; 1: enable interrupt request.)
             *   XHIE/XUPE   Enable interrupt generation on X high event or on Direction
             * recognition. Default value: 0. (0: disable interrupt request; 1: enable
             * interrupt request.) XLIE/XDOWNE Enable interrupt generation on X low event or
             * on Direction recognition. Default value: 0. (0: disable interrupt request; 1:
             * enable interrupt request.)
             */
            R_INT1_CFG = 0x30,

            /*!
             *  INT1_SRC
             *   [0, IA, ZH, ZL, YH, YL, XH, XL]
             *    IA  Interrupt active. Default value: 0
             *        (0: no interrupt has been generated; 1: one or more interrupts have
             * been generated) ZH  Z high. Default value: 0 (0: no interrupt, 1: Z High
             * event has occurred) ZL  Z low. Default value: 0 (0: no interrupt; 1: Z Low
             * event has occurred) YH  Y high. Default value: 0 (0: no interrupt, 1: Y High
             * event has occurred) YL  Y low. Default value: 0 (0: no interrupt, 1: Y Low
             * event has occurred) XH  X high. Default value: 0 (0: no interrupt, 1: X High
             * event has occurred) XL  X low. Default value: 0 (0: no interrupt, 1: X Low
             * event has occurred)
             *
             *    Interrupt 1 source register. Read only register.
             *    Reading at this address clears INT1_SRC IA bit (and the interrupt signal
             * on INT 1 pin) and allows the refreshment of data in the INT1_SRC register if
             * the latched option  was chosen.
             */
            R_INT1_SRC = 0x31,

            R_INT1_THS = 0x32, /**< INT1_THS register [0, THS6, THS5, THS4, THS3, THS1, THS0] */
            R_INT1_DUR = 0x33, /**< INT1_DURATION [0, D6, D5, D4, D3, D2, D1, D0] */

            /*!
             *  CLICK_CFG
             *   [--, --, ZD, ZS, YD, YS, XD, XS]
             *   ZD  Enable interrupt double CLICK-CLICK on Z axis. Default value: 0
             *       (0: disable interrupt request;
             *        1: enable interrupt request on measured accel. value higher than
             * preset threshold) ZS  Enable interrupt single CLICK-CLICK on Z axis. Default
             * value: 0 (0: disable interrupt request; 1: enable interrupt request on
             * measured accel. value higher than preset threshold) YD  Enable interrupt
             * double CLICK-CLICK on Y axis. Default value: 0 (0: disable interrupt request;
             *        1: enable interrupt request on measured accel. value higher than
             * preset threshold) YS  Enable interrupt single CLICK-CLICK on Y axis. Default
             * value: 0 (0: disable interrupt request; 1: enable interrupt request on
             * measured accel. value higher than preset threshold) XD  Enable interrupt
             * double CLICK-CLICK on X axis. Default value: 0 (0: disable interrupt request;
             * 1: enable interrupt request on measured accel. value higher than preset
             * threshold) XS  Enable interrupt single CLICK-CLICK on X axis. Default value:
             * 0 (0: disable interrupt request; 1: enable interrupt request on measured
             * accel. value higher than preset threshold)
             */
            R_CLICK_CFG = 0x38,
            /*!
             *  CLICK_SRC
             *   [-, IA, DCLICK, SCLICK, Sign, Z, Y, X]
             *   IA  Interrupt active. Default value: 0
             *       (0: no interrupt has been generated; 1: one or more interrupts have
             * been generated) DCLICK  Double CLICK-CLICK enable. Default value: 0 (0:double
             * CLICK-CLICK detection disable, 1: double CLICK-CLICK detection enable) SCLICK
             * Single CLICK-CLICK enable. Default value: 0 (0:Single CLICK-CLICK detection
             * disable, 1: single CLICK-CLICK detection enable) Sign    CLICK-CLICK Sign.
             *           (0: positive detection, 1: negative detection)
             *   Z       Z CLICK-CLICK detection. Default value: 0
             *           (0: no interrupt, 1: Z High event has occurred)
             *   Y       Y CLICK-CLICK detection. Default value: 0
             *           (0: no interrupt, 1: Y High event has occurred)
             *   X       X CLICK-CLICK detection. Default value: 0
             *           (0: no interrupt, 1: X High event has occurred)
             */
            R_CLICK_SRC = 0x39,

            /*!
             *  CLICK_THS
             *   [-, Ths6, Ths5, Ths4, Ths3, Ths2, Ths1, Ths0]
             *   Ths6-Ths0  CLICK-CLICK threshold. Default value: 000 0000
             */
            R_CLICK_THS = 0x3A,

            /*!
             *  TIME_LIMIT
             *   [-, TLI6, TLI5, TLI4, TLI3, TLI2, TLI1, TLI0]
             *   TLI7-TLI0  CLICK-CLICK Time Limit. Default value: 000 0000
             */
            R_TIME_LIMIT = 0x3B,

            /*!
             *  TIME_LATANCY
             *   [-, TLA6, TLIA5, TLA4, TLA3, TLA2, TLA1, TLA0]
             *   TLA7-TLA0  CLICK-CLICK Time Latency. Default value: 000 0000
             */
            R_TIME_LATENCY = 0x3C,

            /*!
             *  TIME_WINDOW
             *   [TW7, TW6, TW5, TW4, TW3, TW2, TW1, TW0]
             *   TW7-TW0  CLICK-CLICK Time window
             */
            R_TIME_WINDOW = 0x3D,
        };

        constexpr uint8_t CalculateBit(uint8_t bit) {
            return 1 << bit;
        }

        constexpr uint8_t DurationTime(uint16_t ms, uint16_t odr) {
            return static_cast<uint8_t>((ms * odr) / 1000);
        }

        constexpr uint8_t IntTh(uint16_t thMg, uint16_t fsMg) {
            return static_cast<uint8_t>((thMg * (255 / 2)) / fsMg);
        }

        constexpr uint8_t ODR_LIS3DH_BIT = CalculateBit(4);
        constexpr uint8_t ODR3 = CalculateBit(7);
        constexpr uint8_t ODR2 = CalculateBit(6);
        constexpr uint8_t ODR1 = CalculateBit(5);
        constexpr uint8_t ODR0 = CalculateBit(4);
        constexpr uint8_t LPEN = CalculateBit(3);
        constexpr uint8_t ZEN = CalculateBit(2);
        constexpr uint8_t YEN = CalculateBit(1);
        constexpr uint8_t XEN = CalculateBit(0);

        //CONTROL REGISTER 2
        constexpr uint8_t HPM1 = CalculateBit(7);
        constexpr uint8_t HPM0 = CalculateBit(6);
        constexpr uint8_t HPCF = CalculateBit(4);
        constexpr uint8_t FDS = CalculateBit(3);
        constexpr uint8_t HPCLICK = CalculateBit(2);
        constexpr uint8_t HPIS2 = CalculateBit(1);
        constexpr uint8_t HPIS1 = CalculateBit(0);

        //CONTROL REGISTER 3
        constexpr uint8_t I1_CLICK = CalculateBit(7);
        constexpr uint8_t I1_AOI1 = CalculateBit(6);
        constexpr uint8_t I1_AOI2 = CalculateBit(5);
        constexpr uint8_t I1_DRDY1 = CalculateBit(4);
        constexpr uint8_t I1_DRDY2 = CalculateBit(3);
        constexpr uint8_t I1_WTM = CalculateBit(2);
        constexpr uint8_t I1_ORUN = CalculateBit(1);

        //CONTROL REGISTER 6
        constexpr uint8_t I2_CLICK = CalculateBit(7);
        constexpr uint8_t I2_INT1 = CalculateBit(6);
        constexpr uint8_t I2_BOOT = CalculateBit(4);
        constexpr uint8_t H_LACTIVE = CalculateBit(1);

        //TEMPERATURE CONFIG REGISTER
        constexpr uint8_t ADC_PD = CalculateBit(7);
        constexpr uint8_t TEMP_EN = CalculateBit(6);

        //CONTROL REGISTER 4
        constexpr uint8_t BDU = CalculateBit(7);
        constexpr uint8_t BLE = CalculateBit(6);
        constexpr uint8_t FS = CalculateBit(4);
        constexpr uint8_t HR = CalculateBit(3);
        constexpr uint8_t ST = CalculateBit(1);
        constexpr uint8_t SIM = CalculateBit(0);

        //CONTROL REGISTER 5
        constexpr uint8_t BOOT = CalculateBit(7);
        constexpr uint8_t FIFO_EN = CalculateBit(6);
        constexpr uint8_t LIR_INT1 = CalculateBit(3);
        constexpr uint8_t D4D_INT1 = CalculateBit(2);

        //REFERENCE/DATA_CAPTURE
        constexpr uint8_t REF = CalculateBit(0);

        //STATUS_REG_AXIES
        constexpr uint8_t ZYXOR = CalculateBit(7);
        constexpr uint8_t ZOR = CalculateBit(6);
        constexpr uint8_t YOR = CalculateBit(5);
        constexpr uint8_t XOR = CalculateBit(4);
        constexpr uint8_t ZYXDA = CalculateBit(3);
        constexpr uint8_t ZDA = CalculateBit(2);
        constexpr uint8_t YDA = CalculateBit(1);
        constexpr uint8_t XDA = CalculateBit(0);

        //STATUS_REG_AUX
        constexpr uint8_t STATUS_AUX = 0x07;

        //INTERRUPT 1 CONFIGURATION
        constexpr uint8_t ANDOR = CalculateBit(7);
        constexpr uint8_t INT_6D = CalculateBit(6);
        constexpr uint8_t ZHIE = CalculateBit(5);
        constexpr uint8_t ZLIE = CalculateBit(4);
        constexpr uint8_t YHIE = CalculateBit(3);
        constexpr uint8_t YLIE = CalculateBit(2);
        constexpr uint8_t XHIE = CalculateBit(1);
        constexpr uint8_t XLIE = CalculateBit(0);

        //FIFO CONTROL REGISTER
        constexpr uint8_t FM = CalculateBit(6);
        constexpr uint8_t TR = CalculateBit(5);
        constexpr uint8_t FTH = CalculateBit(0);

        //CONTROL REG3 LIS3DH_BIT mask
        constexpr uint8_t CLICK_ON_PIN_INT1_ENABLE = 0x80;
        constexpr uint8_t CLICK_ON_PIN_INT1_DISABLE = 0x00;
        constexpr uint8_t I1_INT1_ON_PIN_INT1_ENABLE = 0x40;
        constexpr uint8_t I1_INT1_ON_PIN_INT1_DISABLE = 0x00;
        constexpr uint8_t I1_INT2_ON_PIN_INT1_ENABLE = 0x20;
        constexpr uint8_t I1_INT2_ON_PIN_INT1_DISABLE = 0x00;
        constexpr uint8_t I1_DRDY1_ON_INT1_ENABLE = 0x10;
        constexpr uint8_t I1_DRDY1_ON_INT1_DISABLE = 0x00;
        constexpr uint8_t I1_DRDY2_ON_INT1_ENABLE = 0x08;
        constexpr uint8_t I1_DRDY2_ON_INT1_DISABLE = 0x00;
        constexpr uint8_t WTM_ON_INT1_ENABLE = 0x04;
        constexpr uint8_t WTM_ON_INT1_DISABLE = 0x00;
        constexpr uint8_t INT1_OVERRUN_ENABLE = 0x02;
        constexpr uint8_t INT1_OVERRUN_DISABLE = 0x00;

        //CONTROL REG6 LIS3DH_BIT mask
        constexpr uint8_t CLICK_ON_PIN_INT2_ENABLE = 0x80;
        constexpr uint8_t CLICK_ON_PIN_INT2_DISABLE = 0x00;
        constexpr uint8_t I2_INT1_ON_PIN_INT2_ENABLE = 0x40;
        constexpr uint8_t I2_INT1_ON_PIN_INT2_DISABLE = 0x00;
        constexpr uint8_t I2_INT2_ON_PIN_INT2_ENABLE = 0x20;
        constexpr uint8_t I2_INT2_ON_PIN_INT2_DISABLE = 0x00;
        constexpr uint8_t I2_BOOT_ON_INT2_ENABLE = 0x10;
        constexpr uint8_t I2_BOOT_ON_INT2_DISABLE = 0x00;
        constexpr uint8_t INT_ACTIVE_HIGH = 0x00;
        constexpr uint8_t INT_ACTIVE_LOW = 0x02;

        //INT1_CFG LIS3DH_BIT mask
        constexpr uint8_t INT1_AND = 0x80;
        constexpr uint8_t INT1_OR = 0x00;
        constexpr uint8_t INT1_ZHIE_ENABLE = 0x20;
        constexpr uint8_t INT1_ZHIE_DISABLE = 0x00;
        constexpr uint8_t INT1_ZLIE_ENABLE = 0x10;
        constexpr uint8_t INT1_ZLIE_DISABLE = 0x00;
        constexpr uint8_t INT1_YHIE_ENABLE = 0x08;
        constexpr uint8_t INT1_YHIE_DISABLE = 0x00;
        constexpr uint8_t INT1_YLIE_ENABLE = 0x04;
        constexpr uint8_t INT1_YLIE_DISABLE = 0x00;
        constexpr uint8_t INT1_XHIE_ENABLE = 0x02;
        constexpr uint8_t INT1_XHIE_DISABLE = 0x00;
        constexpr uint8_t INT1_XLIE_ENABLE = 0x01;
        constexpr uint8_t INT1_XLIE_DISABLE = 0x00;

        //INT1_SRC LIS3DH_BIT mask
        constexpr uint8_t INT1_SRC_IA = 0x40;
        constexpr uint8_t INT1_SRC_ZH = 0x20;
        constexpr uint8_t INT1_SRC_ZL = 0x10;
        constexpr uint8_t INT1_SRC_YH = 0x08;
        constexpr uint8_t INT1_SRC_YL = 0x04;
        constexpr uint8_t INT1_SRC_XH = 0x02;
        constexpr uint8_t INT1_SRC_XL = 0x01;

        //FIFO Source Register LIS3DH_BIT Mask
        constexpr uint8_t FIFO_SRC_WTM = 0x80;
        constexpr uint8_t FIFO_SRC_OVRUN = 0x40;
        constexpr uint8_t FIFO_SRC_EMPTY = 0x20;

        //INTERRUPT CLICK CONFIGURATION LIS3DH_BIT mask
        constexpr uint8_t ZD_ENABLE = 0x20;
        constexpr uint8_t ZD_DISABLE = 0x00;
        constexpr uint8_t ZS_ENABLE = 0x10;
        constexpr uint8_t ZS_DISABLE = 0x00;
        constexpr uint8_t YD_ENABLE = 0x08;
        constexpr uint8_t YD_DISABLE = 0x00;
        constexpr uint8_t YS_ENABLE = 0x04;
        constexpr uint8_t YS_DISABLE = 0x00;
        constexpr uint8_t XD_ENABLE = 0x02;
        constexpr uint8_t XD_DISABLE = 0x00;
        constexpr uint8_t XS_ENABLE = 0x01;
        constexpr uint8_t XS_DISABLE = 0x00;

        //INTERRUPT CLICK SOURCE REGISTER LIS3DH_BIT mask
        constexpr uint8_t IA = 0x40;
        constexpr uint8_t DCLICK = 0x20;
        constexpr uint8_t SCLICK = 0x10;
        constexpr uint8_t CLICK_SIGN = 0x08;
        constexpr uint8_t CLICK_Z = 0x04;
        constexpr uint8_t CLICK_Y = 0x02;
        constexpr uint8_t CLICK_X = 0x01;

        //STATUS REGISTER LIS3DH_BIT mask
        constexpr uint8_t STATUS_REG_ZYXOR = 0x80;      // 1	:	new data set has over written the previous one
                                                        // 0	:	no overrun has occurred (default)
        constexpr uint8_t STATUS_REG_ZOR = 0x40;        // 0	:	no overrun has occurred (default)
                                                        // 1	:	new Z-axis data has over written the previous one
        constexpr uint8_t STATUS_REG_YOR = 0x20;        // 0	:	no overrun has occurred (default)
                                                        // 1	:	new Y-axis data has over written the previous one
        constexpr uint8_t STATUS_REG_XOR = 0x10;        // 0	:	no overrun has occurred (default)
                                                        // 1	:	new X-axis data has over written the previous one
        constexpr uint8_t STATUS_REG_ZYXDA = 0x08;      // 0	:	a new set of data is not yet avvious one
                                                        // 1	:	a new set of data is available
        constexpr uint8_t STATUS_REG_ZDA = 0x04;        // 0	:	a new data for the Z-Axis is not availvious one
                                                        // 1	:	a new data for the Z-Axis is available
        constexpr uint8_t STATUS_REG_YDA = 0x02;        // 0	:	a new data for the Y-Axis is not available
                                                        // 1	:	a new data for the Y-Axis is available
        constexpr uint8_t STATUS_REG_XDA = 0x01;        // 0	:	a new data for the X-Axis is not available

        constexpr uint8_t DATAREADY_LIS3DH_BIT = STATUS_REG_ZYXDA;


        //STATUS AUX REGISTER LIS3DH_BIT mask
        constexpr uint8_t STATUS_AUX_321OR = 0x80;
        constexpr uint8_t STATUS_AUX_3OR = 0x40;
        constexpr uint8_t STATUS_AUX_2OR = 0x20;
        constexpr uint8_t STATUS_AUX_1OR = 0x10;
        constexpr uint8_t STATUS_AUX_321DA = 0x08;
        constexpr uint8_t STATUS_AUX_3DA = 0x04;
        constexpr uint8_t STATUS_AUX_2DA = 0x02;
        constexpr uint8_t STATUS_AUX_1DA = 0x01;
    }
}
