#pragma once

#include <algorithm>
#include <array>
#include <cassert>
#include <limits>

#include <os.h>

#include <data/expected.h>
#include <data/vector.h>

namespace NLibrary {
    namespace NFilesystem {
        namespace NVega {
            struct TSetting {
                size_t SectorSize;
                size_t SectorCount;
            };

            class THeader {
            public:
                uint8_t Busy = 0;
                uint16_t Address = 0;
                uint32_t WearLevel = 0;
                uint8_t OutMoveStart = 0;
                uint8_t InMoveEnd = 0;

                THeader() = default;
                THeader(uint8_t busy, uint16_t address, uint32_t wearLevel, uint8_t outMoveStart, uint8_t inMoveEnd);

                THeader(const THeader& other) = default;
                THeader& operator=(const THeader& other) = default;

                static constexpr size_t GetSize() {
                    return sizeof(Busy) +
                           sizeof(Address) +
                           sizeof(WearLevel) +
                           sizeof(OutMoveStart) +
                           sizeof(InMoveEnd);
                }

                static constexpr size_t BusyOffset() {
                    return 0;
                }
                static constexpr size_t AddressOffset() {
                    return BusyOffset() + sizeof(Busy);
                }
                static constexpr size_t WearLevelOffset() {
                    return AddressOffset() + sizeof(Address);
                }
                static constexpr size_t OutMoveStartOffset() {
                    return WearLevelOffset() + sizeof(WearLevel);
                }
                static constexpr size_t InMoveEndOffset() {
                    return OutMoveStartOffset() + sizeof(OutMoveStart);
                }

                void Pack(uint8_t* data, size_t size) const;
                void Unpack(const uint8_t* data, size_t size);
            };

            class TLogicalAddressItem {
            public:
                bool Busy = false;
                uint16_t Address = 0;

                TLogicalAddressItem() = default;
                TLogicalAddressItem(bool busy, uint16_t address);
                TLogicalAddressItem(const THeader& header);
            };

            struct TAddress {
                uint16_t Sector = 0;
                size_t Address = 0;
            };

            class TLogicalAddressRecoveryInfo {
            public:
                bool EventFlag = false;
                bool WithCorrupt = false;
                bool Status = false;
                uint16_t Sector = 0;

                TLogicalAddressRecoveryInfo() = default;
                TLogicalAddressRecoveryInfo(bool eventFlag, bool withCorrupt, bool status, uint16_t sector);

                constexpr size_t GetSize() const {
                    return sizeof(uint16_t);
                }
                void Pack(uint8_t* data, size_t size) const;
                void Unpack(const uint8_t* data, size_t size);
            };

            template<class TFlash, size_t WearLeveledSector>
            class TFlashTranslation {
                static_assert(WearLeveledSector > 0);

            public:
                using TLogicalAddressArray = std::array<TLogicalAddressItem, WearLeveledSector>;

            public:
                const size_t MaximumWearDisbalance = 10;
                const size_t FreeSectorCount = 16;

            public:
                TFlashTranslation();

                bool GetFormattedFlag() const {
                    return FormattedFlag;
                }

                bool Init(const TSetting& setting);
                void DeInit();

                bool Write(uint32_t address, NData::TBufferView buffer);
                template<size_t Length>
                NData::TExpected<NData::TBuffer<Length>> Read(uint32_t address);
                bool DirectWrite(size_t sector, uint32_t address, NData::TBufferView buffer);
                template<size_t Length>
                NData::TExpected<NData::TBuffer<Length>> DirectRead(size_t sector, uint32_t address);
                bool DirectErase(size_t sector);

                void Idle();
                std::tuple<bool, size_t, size_t, bool> CheckLogicalAddressTable();
                bool Format();

                size_t GetMemorySize() const {
                    return (WearLeveledSector - FreeSectorCount) * Setting.SectorSize - THeader::GetSize() * WearLeveledSector;
                }

            private:
                TFlash Flash;
                TSetting Setting;
                TLogicalAddressArray LogicalAddressTable;
                TLogicalAddressRecoveryInfo LogicalAddressRecoveryInfo;
                bool FormattedFlag = false;

            private:
                bool Validate(const TSetting& setting);
                bool Validate(size_t sector, uint32_t address, size_t bufferSize);
                THeader ReadHeader(size_t sector);
                void WriteHeader(size_t sector, const THeader& header);
                bool IsFormatted();
                bool LoadLogicalAddressTable();
                void MoveSector(size_t source, size_t destination, const uint8_t* buffer = nullptr, size_t size = 0, size_t addressInSector = 0);
                void WriteSector(TAddress address, NData::TBufferView data);
                bool Recovery();
                NData::TExpected<TAddress> FindPhysicalAddress(uint32_t virtualAddress);
                NData::TExpected<size_t> FindFreeSector(size_t currentSector);

                uint32_t GetAddress(size_t index) {
                    return index * Setting.SectorSize;
                }
            };

            template<class TFlash, size_t WearLeveledSector>
            TFlashTranslation<TFlash, WearLeveledSector>::TFlashTranslation()
                : Flash()
                , Setting()
            {
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::Init(const TSetting& setting) {
                if (!Validate(setting)) {
                    return false;
                }

                LogicalAddressRecoveryInfo.EventFlag = false;
                FormattedFlag = false;

                Setting = setting;

                if (!Flash.Init()) {
                    return false;
                }

                if (IsFormatted()) {
                    LoadLogicalAddressTable();

                    auto [result, addressDublication, notFoundAddress, rangeOk] = CheckLogicalAddressTable();

                    if (!result && addressDublication == 0 && notFoundAddress == 1) {
                        result = Recovery();
                    }

                    if (!result) {
                        Format();
                        FormattedFlag = true;
                    }
                } else {
                    Format();
                    FormattedFlag = true;
                }

                return true;
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::Validate(const TSetting& setting) {
                return setting.SectorSize > 0 && setting.SectorCount > 0;
            }

            template<class TFlash, size_t WearLeveledSector>
            THeader TFlashTranslation<TFlash, WearLeveledSector>::ReadHeader(size_t sector) {
                THeader result;
                NData::TBuffer<THeader::GetSize()> data;
                data.Advance(data.GetCapacity());
                Flash.Read(sector, 0, data);
                result.Unpack(data.GetData(), data.GetSize());
                return result;
            }

            template<class TFlash, size_t WearLeveledSector>
            void TFlashTranslation<TFlash, WearLeveledSector>::WriteHeader(size_t sector, const THeader& header) {
                NData::TBuffer<THeader::GetSize()> data;
                header.Pack(data.end(), data.GetCapacity());
                data.Advance(data.GetCapacity());
                Flash.Write(sector, 0, data);
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::IsFormatted() {
                const size_t wearedSuccessCount = 1;
                size_t notWearedCount = 0;

                for (size_t i = 0; i < WearLeveledSector; ++i) {
                    auto header = ReadHeader(i);

                    if (header.WearLevel == std::numeric_limits<uint32_t>::max()) {
                        notWearedCount++;
                    }
                }

                if (notWearedCount > wearedSuccessCount) {
                    return false;
                }

                return true;
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::LoadLogicalAddressTable() {
                THeader errorInHeader;
                size_t errorOutSector = 0;
                size_t errorInSector = 0;
                size_t maximumWear = 0;

                for(size_t i = 0; i < WearLeveledSector; i++) {
                    auto header = ReadHeader(i);
                    LogicalAddressTable[i] = header;

                    if(header.WearLevel > maximumWear && header.WearLevel != std::numeric_limits<uint32_t>::max()) {
                        maximumWear = header.WearLevel;
                    }

                    // Search sector which in transfer not completed
                    if (header.Busy && header.InMoveEnd != 0x00) {
                        errorInSector = i + 1;
                        errorInHeader = header;
                    }

                    // Search sector which out transfer not completed
                    if (header.OutMoveStart == 0x00 && header.Busy) {
                        errorOutSector = i + 1;
                    }
                }

                if (errorInSector && !errorOutSector && errorInHeader.WearLevel == std::numeric_limits<uint32_t>::max()) {
                    errorInSector--;

                    Flash.SectorErase(errorInSector);
                    THeader header(0, 0, maximumWear, 0xFF, 0xFF);
                    WriteHeader(errorInSector, header);

                    LogicalAddressTable[errorInSector] = TLogicalAddressItem(false, 0);
                } else if (errorInSector && !errorOutSector && errorInHeader.WearLevel != std::numeric_limits<uint32_t>::max()) {
                    errorInSector--;
                    LogicalAddressTable[errorInSector] = TLogicalAddressItem(false, 0);
                    NData::TBuffer<3> busyAndAddress = { 0, 0, 0 };

                    Flash.Write(errorInSector, 0, busyAndAddress);
                } else if (errorInSector && errorOutSector && errorInHeader.InMoveEnd != 0) {
                    errorInSector--;
                    errorOutSector--;
                    MoveSector(errorOutSector, errorInSector);
                } else if (!errorInSector && errorOutSector) {
                    NData::TBuffer<1> busy = { 0 };
                    errorOutSector--;
                    Flash.Write(errorOutSector, THeader::BusyOffset(), busy);

                    auto& logicalAddressItem = LogicalAddressTable[errorOutSector];
                    logicalAddressItem.Busy = false;
                }

                return true;
            }

            template<class TFlash, size_t WearLeveledSector>
            void TFlashTranslation<TFlash, WearLeveledSector>::MoveSector(size_t source, size_t destination, const uint8_t* buffer, size_t size, size_t addressInSector) {
                auto sourceHeader = ReadHeader(source);
                auto destinationHeader = ReadHeader(destination);

                destinationHeader.Address = sourceHeader.Address;
                Flash.Erase(destination);

                destinationHeader.WearLevel++;
                Flash.Write(
                    destination, THeader::WearLevelOffset(),
                    NData::TBufferView(reinterpret_cast<uint8_t*>(&destinationHeader.WearLevel), sizeof(destinationHeader.WearLevel))
                );
                uint8_t moveStart[] = {0};
                Flash.Write(source, THeader::OutMoveStartOffset(), NData::TBufferView(moveStart, sizeof(moveStart)));
                size_t bufferIndex = 0;
                uint8_t moveBuffer[256];
                for(size_t i = THeader::GetSize(); i < Setting.SectorSize;) { // 31 ms
                    size_t readWriteSize = sizeof(moveBuffer);
                    if (i + readWriteSize >= Setting.SectorSize) {
                        readWriteSize = Setting.SectorSize - i;//NOR_FTL_SECTOR_SIZE - i;
                    }
                    Flash.Read(source, i, NData::TBufferView(moveBuffer, readWriteSize));
                    if (size && buffer != nullptr) {
                        for (size_t j = i; j < i + readWriteSize; j++) {
                            if (j >= addressInSector && j < addressInSector + size)  {
                                moveBuffer[j - i] = buffer[bufferIndex];
                                bufferIndex++;
                            }
                        }
                    }
                    Flash.Write(destination, i, NData::TBufferView(moveBuffer, readWriteSize));
                    i += readWriteSize;
                }

                Flash.Write(
                    destination, THeader::AddressOffset(),
                    NData::TBufferView(reinterpret_cast<uint8_t*>(&destinationHeader.Address), sizeof(destinationHeader.Address))
                );

                destinationHeader.InMoveEnd = 0x00;
                Flash.Write(
                    destination, THeader::InMoveEndOffset(),
                    NData::TBufferView(reinterpret_cast<uint8_t*>(&destinationHeader.InMoveEnd), sizeof(destinationHeader.InMoveEnd))
                );

                NData::TBuffer<1> busy = { 0 };
                Flash.Write(source, THeader::BusyOffset(), busy);

                LogicalAddressTable[source] = TLogicalAddressItem(false, 0);
                LogicalAddressTable[destination] = TLogicalAddressItem(true, destinationHeader.Address);
            }

            template<class TFlash, size_t WearLeveledSector>
            std::tuple<bool, size_t, size_t, bool> TFlashTranslation<TFlash, WearLeveledSector>::CheckLogicalAddressTable() {
                size_t addressDublication = 0;
                size_t notFoundAddress = WearLeveledSector;
                bool rangeOk = true;
                bool result = false;

                size_t min = std::numeric_limits<size_t>::max();
                size_t max = std::numeric_limits<size_t>::min();

                for (size_t i = 0; i < WearLeveledSector; ++i) {
                    auto& logicalAddressItem = LogicalAddressTable[i];
                    if (logicalAddressItem.Busy) {
                        min = std::min(size_t(logicalAddressItem.Address), min);
                        max = std::max(size_t(logicalAddressItem.Address), max);
                    }

                    for (size_t j = 0; j < WearLeveledSector; ++j) {
                        auto& currentLogicaAddressItem = LogicalAddressTable[j];

                        if (i != j && logicalAddressItem.Address == currentLogicaAddressItem.Address && logicalAddressItem.Busy && currentLogicaAddressItem.Busy) {
                            addressDublication++;
                        }
                    }
                }

                if ((addressDublication % 2) > 0) {
                    addressDublication++;
                }

                addressDublication /= 2;

                if (min != 0 || max != (WearLeveledSector - FreeSectorCount - 1)) {
                    rangeOk = false;
                }

                if (max <= WearLeveledSector && min < max) {
                    notFoundAddress = 0;

                    for (size_t sectorAddress = min; sectorAddress <= max; ++sectorAddress) {
                        size_t findedIndex = 0;
                        for (size_t i = 0; i < WearLeveledSector; ++i) {
                            auto& logicalAddressItem = LogicalAddressTable[i];

                            if (logicalAddressItem.Busy) {
                                if (sectorAddress == logicalAddressItem.Address) {
                                    findedIndex = i;
                                    break;
                                }
                            }
                        }

                        auto& logicalAddressItem = LogicalAddressTable[findedIndex];
                        if (findedIndex < WearLeveledSector && sectorAddress == logicalAddressItem.Address) {
                            continue;
                        }

                        notFoundAddress++;

                        if (notFoundAddress >= WearLeveledSector) {
                            break;
                        }
                    }
                }
                result = !notFoundAddress && !addressDublication && rangeOk;

                return { result, addressDublication, notFoundAddress, rangeOk };
            }

            template<class TFlash, size_t WearLeveledSector>
            auto TFlashTranslation<TFlash, WearLeveledSector>::FindFreeSector(size_t currentSector) -> NData::TExpected<size_t> {
                for(size_t sector = currentSector; sector < WearLeveledSector; ++sector) {
                    if(!LogicalAddressTable[sector].Busy) {
                        return { true, sector };
                    }
                }
                for(size_t sector = 0; sector < currentSector; ++sector) {
                    if (!LogicalAddressTable[sector].Busy) {
                        return { true, sector };
                    }
                }
                return { false, 0 };
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::Recovery() {
                const size_t min = 0;
                const size_t max = WearLeveledSector - FreeSectorCount - 1;

                LogicalAddressRecoveryInfo.EventFlag = true;

                for(uint16_t lostSector = min; lostSector <= max; ++lostSector) {
                    size_t index = 0;

                    for(index = 0; index < WearLeveledSector; ++index) {
                        auto& current = LogicalAddressTable[index];
                        if (current.Busy && lostSector == current.Address) {
                            break;
                        }
                    }

                    auto& logicalAddressItem = LogicalAddressTable[index];
                    if (index < WearLeveledSector && lostSector == logicalAddressItem.Address) {
                        continue;
                    }

                    LogicalAddressRecoveryInfo.Sector = lostSector;

                    for(size_t i = 0; i < WearLeveledSector; ++i) {
                        auto& current = LogicalAddressTable[i];
                        if(current.Busy == 0x00 && current.Address == lostSector) {
                            auto header = ReadHeader(i);

                            if (header.Busy == 0x00 && header.Address == lostSector &&
                                header.WearLevel != 0xFFFFFFFF && header.OutMoveStart == 0xFF &&
                                header.InMoveEnd == 0x00) {

                                auto [resultFind, freeSector] = FindFreeSector(i + 1);
                                if(!resultFind) {
                                    break;
                                }

                                MoveSector(i, freeSector);

                                LogicalAddressRecoveryInfo.WithCorrupt = false;
                                LogicalAddressRecoveryInfo.Status = true;
                                return true;
                            }
                        }
                    }

                    if (lostSector < 3) {
                        break;
                    }

                    auto [findResult, freeSector] = FindFreeSector(0);

                    if(!findResult) {
                        break;
                    }

                    auto header = ReadHeader(0);
                    Flash.Erase(GetAddress(freeSector));
                    auto& currentSector = LogicalAddressTable[freeSector];

                    currentSector = TLogicalAddressItem(true, lostSector);
                    header = THeader(0xFF, currentSector.Address, ++header.WearLevel, 0xFF, 0x00);

                    if (header.WearLevel == 0xFFFFFFFF) {
                        header.WearLevel /= 2;
                    }

                    WriteHeader(GetAddress(freeSector), header);

                    LogicalAddressRecoveryInfo.WithCorrupt = true;
                    LogicalAddressRecoveryInfo.Status = true;

                    return true;
                }

                LogicalAddressRecoveryInfo.Status = false;
                return false;
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::Format() {
                Flash.FullErase();
                for(size_t sector = 0; sector < WearLeveledSector; ++sector) {
                    auto& current = LogicalAddressTable[sector];
                    current.Address = sector;

                    if (sector < WearLeveledSector - FreeSectorCount) {
                        current.Busy = true;
                    } else {
                        current = TLogicalAddressItem(false, 0);
                    }

                    auto header = THeader(0xFF, current.Address, 0, 0xFF, 0x00);
                    if (!current.Busy) {
                        header.Busy = 0x00;
                    }
                    WriteHeader(sector, header);
                }
                return true;
            }

            template<class TFlash, size_t WearLeveledSector>
            void TFlashTranslation<TFlash, WearLeveledSector>::DeInit() {
                Flash.DeInit();
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::Write(uint32_t address, NData::TBufferView buffer) {
                if (address + buffer.size() > GetMemorySize()) {
                    return false;
                }
                if (buffer.size() > Setting.SectorSize) {
                    return false;
                }

                TAddress second;
                size_t firstSize = 0;
                size_t secondSize = 0;
                auto [result, first] = FindPhysicalAddress(address);
                if (!result) {
                    return false;
                }

                if (first.Address + buffer.size() > Setting.SectorSize) {
                    uint32_t secondAddress = address + Setting.SectorSize - first.Address;
                    bool localResult = false;
                    std::tie(localResult, second) = FindPhysicalAddress(secondAddress);
                    if (!localResult) {
                        return false;
                    }

                    firstSize = Setting.SectorSize - first.Address;
                    secondSize = buffer.size() - firstSize;
                } else {
                    firstSize = buffer.size();
                }

                if (firstSize > 0) {
                    WriteSector(first, buffer);
                }

                if (secondSize > 0) {
                    WriteSector(second, NData::TBufferView(buffer.data() + firstSize, secondSize));
                }

                return true;
            }

            template<class TFlash, size_t WearLeveledSector>
            auto TFlashTranslation<TFlash, WearLeveledSector>::FindPhysicalAddress(uint32_t virtualAddress) -> NData::TExpected<TAddress> {
                bool result = false;
                TAddress address;

                auto sectorAddress = virtualAddress / (Setting.SectorSize - THeader::GetSize());

                for (size_t i = 0; i < WearLeveledSector; ++i) {
                    auto& current = LogicalAddressTable[i];
                    if (!current.Busy && current.Address == sectorAddress) {
                        address = {
                            .Sector = static_cast<uint16_t>(i),
                            .Address = (virtualAddress % (Setting.SectorSize - THeader::GetSize())) + THeader::GetSize(),
                        };
                    }
                }

                return { result, address };
            }

            template<class TFlash, size_t WearLeveledSector>
            void TFlashTranslation<TFlash, WearLeveledSector>::WriteSector(TAddress address, NData::TBufferView data) {
                bool needMoveSector = false;

                uint32_t realAddress = GetAddress(address.Sector) + address.Address;
                for (size_t i = realAddress; i < realAddress + data.size(); ++i) {
                    uint8_t temp = 0;
                    Flash.Read(i, &temp, sizeof(temp));

                    if (temp != 0xFF) {
                        needMoveSector = true;
                        break;
                    }
                }

                if (needMoveSector) {
                    auto [findResult, sector] = FindFreeSector();
                    MoveSector(address.Sector, GetAddress(sector), data.data(), data.size(), address.Address);
                } else {
                    Flash.Write(address.Sector, address.Address, data);
                }
            }

            template<class TFlash, size_t WearLeveledSector>
            template<size_t Length>
            auto TFlashTranslation<TFlash, WearLeveledSector>::Read(uint32_t address) -> NData::TExpected<NData::TBuffer<Length>> {
                NData::TBuffer<Length> result;

                if (address + Length > GetMemorySize()) {
                    return { false, result };
                }

                if (Length > Setting.SectorSize) {
                    return { false, result };
                }

                auto [firstFind, firstAddress] = FindPhysicalAddress(address);
                if (!firstFind) {
                    return { false, result };
                }

                if (Setting.SectorSize - firstAddress.Address > Length) {
                    Flash.Read(GetAddress(firstAddress.Sector) + firstAddress.Address, result.data(), result.GetCapacity());
                    result.Advance(result.GetCapacity());
                } else {
                    size_t firstSize = Setting.SectorSize - firstAddress.Address;
                    size_t secondSize = Length - firstSize;

                    Flash.Read(GetAddress(firstAddress.Sector) + firstAddress.Address, result.data(), firstSize);
                    result.Advance(firstSize);

                    auto [secondFind, secondAddress] = FindPhysicalAddress(address + firstSize);

                    Flash.Read(GetAddress(secondAddress.Sector) + secondAddress.Address, result.end(), secondSize);
                    result.Advance(secondSize);
                }

                return { true, result };
            }

            template<class TFlash, size_t WearLeveledSector>
            void TFlashTranslation<TFlash, WearLeveledSector>::Idle() {
                size_t minimumWearSector = 0;
                size_t maximumWearSector = 0;
                uint32_t minimumWear = std::numeric_limits<uint32_t>::max();
                uint32_t maximumWear = 0;

                for (size_t sector = 0; sector < WearLeveledSector; ++sector) {
                    auto header = ReadHeader(sector);
                    if (header.Busy > 0 && header.WearLevel < minimumWear) {
                        minimumWear = header.WearLevel;
                        minimumWearSector = sector;
                    }
                    if (header.Busy == 0 && header.WearLevel > maximumWear) {
                        maximumWear = header.WearLevel;
                        maximumWearSector = sector;
                    }
                }

                if (maximumWear > minimumWear && (maximumWear - minimumWear) > MaximumWearDisbalance) {
                    MoveSector(minimumWearSector, maximumWearSector);
                }
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::DirectWrite(size_t sector, uint32_t address, NData::TBufferView buffer) {
                sector += WearLeveledSector;

                if (!Validate(sector, address, buffer.size())) {
                    return false;
                }

                return Flash.Write(sector, address, buffer);
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::Validate(size_t sector, uint32_t address, size_t bufferSize) {
                if (sector >= Setting.SectorCount) {
                    return false;
                }
                if (address + bufferSize > Setting.SectorSize) {
                    return false;
                }
                return true;
            }

            template<class TFlash, size_t WearLeveledSector>
            template<size_t Length>
            NData::TExpected<NLibrary::NData::TBuffer<Length>>
            TFlashTranslation<TFlash, WearLeveledSector>::DirectRead(size_t sector, uint32_t address) {
                NLibrary::NData::TBuffer<Length> result;

                sector += WearLeveledSector;
                if (!Validate(sector, address, Length)) {
                    return { false, result };
                }

                bool readResult = Flash.Read(sector, address, result);
                return { readResult, result };
            }

            template<class TFlash, size_t WearLeveledSector>
            bool TFlashTranslation<TFlash, WearLeveledSector>::DirectErase(size_t sector) {
                sector += WearLeveledSector;
                if (sector > Setting.SectorCount) {
                    return false;
                }

                Flash.SectorErase(sector);
                return true;
            }
        }
    }
}
