#pragma once

#include <cstdint>

#include <data/string.h>

#include <time/time.h>

namespace NLibrary {
    namespace NStream {
        class IOutput {
        public:
            virtual ~IOutput() = default;

            virtual void Save(uint8_t data) = 0;
            virtual void Flush() = 0;
        };

        class IInput {
        public:
            virtual ~IInput() = default;

            virtual int Avaliable() = 0;
            virtual int Read() = 0;
            virtual int Peek() = 0;

            NTime::TTime GetTimeout() const {
                return Timeout;
            }
            void SetTimeout(const NTime::TTime& timeout) {
                Timeout = timeout;
            }

            size_t Load(uint8_t* data, size_t size);
            size_t Load(uint8_t* data, size_t size, uint8_t terminator);

        protected:
            NTime::TTime Timeout;

        protected:
            int TimedRead();
            int TimedPeek();
        };

        template<size_t Length>
        class IStringOutput: public IOutput {
        public:
            using TData = NData::TString<Length>;

        public:
            virtual ~IStringOutput() = default;

            void Save(uint8_t data) override {
                if (Data.IsFull()) {
                    Flush();
                }
                Data.Append(static_cast<char>(data));
            }

            virtual void Flush() override {
                Data.Clear();
            }

            IStringOutput& Write(char data) {
                Save(static_cast<uint8_t>(data));
                return *this;
            }

            IStringOutput& Write(const char* data) {
                size_t length = strlen(data);

                for (size_t i = 0; i < length; ++i) {
                    Save(static_cast<uint8_t>(data[i]));
                }
                return *this;
            }

            template<size_t OtherLength>
            IStringOutput& Write(NData::TString<OtherLength> data) {
                for (char item : data) {
                    Save(static_cast<uint8_t>(item));
                }
                return *this;
            }

            IStringOutput& operator<<(char data) {
                return Write(data);
            }

            IStringOutput& operator<<(const char* data) {
                return Write(data);
            }

            template<size_t OtherLength>
            IStringOutput& operator<<(NData::TString<OtherLength> data) {
                return Write(data);
            }

            IStringOutput& operator<<(void(*function)(IStringOutput& stream)) {
                if (function) {
                    function(*this);
                }
                return *this;
            }

        protected:
            TData Data;

        protected:
            void CheckFlush() {
                if (!Data.IsEmpty()) {
                    Flush();
                }
            }
        };

        template<size_t Length>
        inline void Endl(IStringOutput<Length>& stream) {
            stream << "\r\n";
        }
    }
}
