#pragma once

#include <os.h>

namespace NLibrary {
    namespace NTime {
        class TTime {
        public:
            using TTick = NOs::NTask::TTick;

        public:
            static constexpr TTick MillisecondInSecond = 1000;
            static constexpr TTick SecondInMinute = 60;
            static constexpr TTick MinuteInHour = 60;
            static constexpr TTick HourInDay = 24;

        public:
            TTime() = default;
            explicit constexpr TTime(TTick tick)
                : Tick(tick)
            {
            }

            TTime(const TTime& other) = default;
            TTime& operator=(const TTime& other) = default;

            static TTime Now() {
                return TTime(NOs::NTask::Current());
            }
            static void Delay(const TTime& time) {
                NOs::NTask::Delay(time.GetTick());
            }
            static constexpr TTime MilliSecond(TTick tick) {
                return TTime(tick);
            }
            static constexpr TTime Second(TTick tick) {
                return MilliSecond(tick ) * MillisecondInSecond;
            }
            static constexpr TTime Minute(TTick tick) {
                return Second(tick) * SecondInMinute;
            }
            static constexpr TTime Hour(TTick tick) {
                return Minute(tick) * MinuteInHour;
            }
            static constexpr TTime Day(TTick tick) {
                return Hour(tick) * HourInDay;
            }

            constexpr TTick GetTick() const {
                return Tick;
            }

            friend constexpr bool operator==(const TTime& lhs, const TTime& rhs) {
                return lhs.GetTick() == rhs.GetTick();
            }

            friend constexpr bool operator>(const TTime& lhs, const TTime& rhs) {
                return lhs.GetTick() > rhs.GetTick();
            }

            friend constexpr bool operator<(const TTime& lhs, const TTime& rhs) {
                return lhs.GetTick() < rhs.GetTick();
            }

            friend constexpr bool operator>=(const TTime& lhs, const TTime& rhs) {
                return lhs.GetTick() >= rhs.GetTick();
            }

            friend constexpr bool operator<=(const TTime& lhs, const TTime& rhs) {
                return lhs.GetTick() <= rhs.GetTick();
            }

            friend constexpr TTime operator+(const TTime& lhs, const TTime& rhs) {
                return TTime(lhs.GetTick() + rhs.GetTick());
            }

            friend constexpr TTime operator+(const TTime& lhs, TTime::TTick rhs) {
                return TTime(lhs.GetTick() + rhs);
            }

            friend constexpr TTime operator-(const TTime& lhs, const TTime& rhs) {
                return TTime(lhs.GetTick() - rhs.GetTick());
            }

            friend constexpr TTime operator-(const TTime& lhs, TTime::TTick rhs) {
                return TTime(lhs.GetTick() - rhs);
            }

            friend constexpr TTime operator*(const TTime& lhs, const TTime& rhs) {
                return TTime(lhs.GetTick() * rhs.GetTick());
            }

            friend constexpr TTime operator*(const TTime& lhs, TTime::TTick rhs) {
                return TTime(lhs.GetTick() * rhs);
            }

            friend constexpr TTime operator/(const TTime& lhs, const TTime& rhs) {
                return TTime(lhs.GetTick() / rhs.GetTick());
            }

            friend constexpr TTime operator/(const TTime& lhs, TTime::TTick rhs) {
                return TTime(lhs.GetTick() / rhs);
            }

        private:
            TTick Tick;
        };
    }
}
