#pragma once

#include <concepts>

#include <data/string.h>

namespace NLibrary {
    namespace NUtil {
        enum EIntegerBase: uint8_t {
            Dec = 10,
            Hex = 16,
            Oct = 8,
            Bin = 2
        };

        using TIntegerString = NData::TString<32>;
        using TFloatString = NData::TString<32>;

        template<typename T>
        concept TBoolType = std::same_as<T, bool>;

        template<typename T>
        concept TIntegralType = std::integral<T> && !TBoolType<T>;

        inline TIntegerString ToString(TBoolType auto number) {
            TIntegerString result;

            if (number) {
                result.Append("true");
            } else {
                result.Append("false");
            }

            return result;
        }

        inline TIntegerString ToString(TIntegralType auto number, EIntegerBase base = EIntegerBase::Dec) {
            TIntegerString result;

            bool isSigned = number < 0;
            if (isSigned) {
                number = -number;
            }

            do {
                char symbol = number % base;
                number /= base;

                symbol = symbol < 10 ? symbol + '0' : symbol + 'A' - 10;
                result.Append(symbol);
            } while (number > 0);

            if (isSigned) {
                result.Append('-');
            }

            std::reverse(result.begin(), result.end());

            return result;
        }

        inline TFloatString ToString(std::floating_point auto number, uint8_t digits = 2) {
            TFloatString result;

            if (number < 0.0f) {
                result.Append(static_cast<uint8_t>('-'));
                number = -number;
            }

            double rounding = 0.5;
            for (uint8_t i = 0; i < digits; ++i) {
                rounding /= 10.0;
            }

            number += static_cast<float>(rounding);

            auto intPart = static_cast<uint32_t>(number);
            double remainder = number - static_cast<double>(intPart);

            result.Append(ToString(intPart));

            if (digits > 0) {
                result.Append(static_cast<uint8_t>('.'));
            }

            while (digits-- > 0) {
                remainder *= 10.0;
                auto toPrint = static_cast<uint8_t>(remainder);
                result.Append(ToString(static_cast<uint32_t>(toPrint)));
                remainder -= toPrint;
            }

            return result;
        }
    }
}
