#pragma once

#include <cassert>

#include <lock/lock.h>

#include <os_hal.h>

namespace NOs {
    namespace NMutex {
        TId Create();
        bool Lock(TId id);
        void Unlock(TId id);
        void Remove(TId id);

        struct Implementation {
            using TId = NOs::NMutex::TId;
            static constexpr TId EmptyId = NOs::NMutex::EmptyId;

            static TId Create() {
                return NOs::NMutex::Create();
            }
            static bool Lock(TId id) {
                return NOs::NMutex::Lock(id);
            }
            static void Unlock(TId id) {
                NOs::NMutex::Unlock(id);
            }
            static void Remove(TId id) {
                NOs::NMutex::Remove(id);
            }
        };
    }
}

namespace NOs {
    namespace NRecursiveMutex {
        TId Create();
        bool Lock(TId id);
        void Unlock(TId id);
        void Remove(TId id);

        struct Implementation {
            using TID = NOs::NRecursiveMutex::TId;
            static constexpr TId EmptyId = NOs::NRecursiveMutex::EmptyId;

            static TId Create() {
                return NOs::NRecursiveMutex::Create();
            }
            static bool Lock(TId id) {
                return NOs::NRecursiveMutex::Lock(id);
            }
            static void Unlock(TId id) {
                NOs::NRecursiveMutex::Unlock(id);
            }
            static void Remove(TId id) {
                NOs::NRecursiveMutex::Remove(id);
            }
        };
    }
}

namespace NOs {
    bool IsStarted();

    template<class TImplementation>
    class MutexBase final : public NLibrary::NLock::ILockable {
    public:
        using TId = typename TImplementation::TId;

    public:
        MutexBase()
            : ILockable(), Id()
        {
            if (NOs::IsStarted()) {
                Create();
            }
        }

        virtual ~MutexBase() = default;

        void Lock() override {
            if (!NOs::IsStarted()) {
                return;
            }
            if (Id == TImplementation::EmptyId) {
                Create();
            }

            TImplementation::Lock(Id);
        }

        void Unlock() override {
            if (Id == TImplementation::EmptyId) {
                return;
            }
            TImplementation::Unlock(Id);
        }

    private:
        TId Id;

    private:
        void Create() {
            Id = TImplementation::Create();
            assert(Id != TImplementation::EmptyId);
        }
    };

    using TMutex = MutexBase<NOs::NMutex::Implementation>;
    using TRecursiveMutex = MutexBase<NOs::NRecursiveMutex::Implementation>;
}
