#pragma once

#include <cassert>
#include <cstddef>

#include <os_hal.h>

namespace NOs {
    namespace NQueue {
        TId Create(size_t count, size_t size);
        bool Put(TId id, const void* data, size_t size);
        bool Get(TId id, void* data, size_t size);
        bool Peek(TId id, void* data, size_t size);
        size_t GetSize(TId id);
        void Remove(TId id);
        void Clear(TId id);

        template<typename TItem, size_t Length>
        class TQueue {
        public:
            using TId = NOs::NQueue::TId;

        public:
            TQueue()
                : Id()
            {
                if constexpr (Length > 0) {
                    Id = NOs::NQueue::Create(Length, sizeof(TItem));
                    assert(Id != NOs::NQueue::EmptyId);
                }
            }

            virtual ~TQueue() = default;

            void Deinit() {
                if constexpr (Length > 0) {
                    NOs::NQueue::Remove(Id);
                }
            }

            bool Put(const TItem& item) {
                if constexpr (Length > 0) {
                    return NOs::NQueue::Put(Id, reinterpret_cast<const void*>(&item), sizeof(TItem));
                }
                return false;
            }

            TId GetId() {
                return Id;
            }

            size_t GetSize() const {
                if constexpr (Length > 0) {
                    return NOs::NQueue::GetSize(Id);
                } else {
                    return 0;
                }
            }
            inline size_t GetCapacity() const {
                return Length;
            }
            bool IsFull() const {
                return GetSize() == GetCapacity();
            }
            bool IsEmpty() const {
                return GetSize() == 0;
            }

            TItem Get() {
                TItem result;

                if constexpr (Length > 0) {
                    NOs::NQueue::Get(Id, reinterpret_cast<void*>(&result), sizeof(TItem));
                }

                return result;
            }

            void Clear() {
                if constexpr (Length > 0) {
                    NOs::NQueue::Clear(Id);
                }
            }

        private:
            TId Id;
        };
    }
}