#include <devices/lis/device.h>
#include <devices/lis/registers.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/testing/unittest/registar.h>

namespace {
    class TDummyLisSpi {
    public:
        using TRegister = TMap<ui8, ui8>;

        class TDummyLock: public NLibrary::NLock::ILockable {
        public:
            TDummyLock() = default;
            ~TDummyLock() = default;

            void Lock() override {
            }
            void Unlock() override {
            }
        };

    public:
        TDummyLisSpi()
            : Registers()
            , Lock()
        {
            Clear();
        }

        TRegister& GetRegisters() {
            return Registers;
        }

        void Clear() {
            Registers[NLibrary::NLis::R_CTRL_1] = 0;
            Registers[NLibrary::NLis::R_CTRL_2] = 0;
            Registers[NLibrary::NLis::R_CTRL_3] = 0;
            Registers[NLibrary::NLis::R_CTRL_4] = 0;
            Registers[NLibrary::NLis::R_CTRL_5] = 0;
            Registers[NLibrary::NLis::R_CTRL_6] = 0;
            Registers[NLibrary::NLis::R_OUT_X_L] = 0;
            Registers[NLibrary::NLis::R_OUT_X_H] = 0;
            Registers[NLibrary::NLis::R_OUT_Y_L] = 0;
            Registers[NLibrary::NLis::R_OUT_Y_H] = 0;
            Registers[NLibrary::NLis::R_OUT_Z_L] = 0;
            Registers[NLibrary::NLis::R_OUT_Z_H] = 0;
            Registers[NLibrary::NLis::R_INT1_CFG] = 0;
            Registers[NLibrary::NLis::R_INT1_SRC] = 0;
            Registers[NLibrary::NLis::R_INT1_THS] = 0;
            Registers[NLibrary::NLis::R_INT1_DUR] = 0;
        }

        TDummyLock& GetLock() {
            return Lock;
        }

        void Transmit(const uint8_t data[], size_t size) {
            Y_ENSURE(size == 2);

            uint8_t reg = data[0];
            uint8_t value = data[1];

            Y_ENSURE(Registers.contains(reg));
            Registers[reg] = value;
        }

        void TransmitReceive(const uint8_t txData[], size_t txSize, uint8_t rxData[], size_t rxSize) {
            uint8_t request = txData[0];

            Y_ENSURE(txSize == 1);

            bool isReadRequest = (request >> 7) & 0x01;
            bool isAutoIncrement = (request >> 6) & 0x01;

            if (!isReadRequest) {
                rxData[0] = 0xFF;
                return;
            }

            if (rxSize > 1) {
                Y_ENSURE(isAutoIncrement);
            }

            request &= 0x3F;
            ui8 start = request;
            for (size_t i = 0 ; i < rxSize; ++i) {
                Y_ENSURE(Registers.contains(start));
                rxData[i] = Registers[start];
                start++;
            }
        }

        static TDummyLisSpi* GetInstance() {
            return Singleton<TDummyLisSpi>();
        }

    private:
        TRegister Registers;
        TDummyLock Lock;
    };

    using TLis = NLibrary::NLis::TDevice<TDummyLisSpi>;
}

Y_UNIT_TEST_SUITE(Lis) {
    Y_UNIT_TEST(Reset) {
        using namespace NLibrary::NLis;
        auto& spi = *TDummyLisSpi::GetInstance();
        spi.Clear();

        TLis lis;

        lis.Reset();
        auto& registers = spi.GetRegisters();
        UNIT_ASSERT_EQUAL(registers[R_CTRL_5], BOOT);
    }

    Y_UNIT_TEST(NormalMode) {
        using namespace NLibrary::NLis;
        auto& spi = *TDummyLisSpi::GetInstance();
        spi.Clear();

        TLis lis;
        lis.SetNormalMode(SR_HZ_400, RNG_2);

        auto& registers = spi.GetRegisters();
        UNIT_ASSERT_EQUAL(registers[R_CTRL_1], 119);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_2], 0);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_3], 16);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_4], 136);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_5], 64);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_6], 2);
    }

    Y_UNIT_TEST(SleepMode) {
        using namespace NLibrary::NLis;
        auto& spi = *TDummyLisSpi::GetInstance();
        spi.Clear();

        TLis lis;
        lis.SetSleepMode(SR_HZ_50, RNG_2);

        auto& registers = spi.GetRegisters();
        UNIT_ASSERT_EQUAL(registers[R_CTRL_1], 71);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_2], 193);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_3], 64);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_4], 8);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_5], 0);
        UNIT_ASSERT_EQUAL(registers[R_CTRL_6], 2);
        UNIT_ASSERT_EQUAL(registers[R_INT1_CFG], 42);
        UNIT_ASSERT_EQUAL(registers[R_INT1_THS], 3);
        UNIT_ASSERT_EQUAL(registers[R_INT1_DUR], 10);
    }

    Y_UNIT_TEST(Point) {
        using namespace NLibrary::NLis;
        auto& spi = *TDummyLisSpi::GetInstance();
        auto& registers = spi.GetRegisters();
        spi.Clear();

        auto defaultPoint = NLibrary::NLis::TPoint(1632, -32, -3241);
        auto wrongPoint = NLibrary::NLis::TPoint(-1632, 32, 3241);
        registers[R_OUT_X_L] = (defaultPoint.X >> 0) & 0xFF;
        registers[R_OUT_X_H] = (defaultPoint.X >> 8) & 0xFF;
        registers[R_OUT_Y_L] = (defaultPoint.Y >> 0) & 0xFF;
        registers[R_OUT_Y_H] = (defaultPoint.Y >> 8) & 0xFF;
        registers[R_OUT_Z_L] = (defaultPoint.Z >> 0) & 0xFF;
        registers[R_OUT_Z_H] = (defaultPoint.Z >> 8) & 0xFF;

        TLis lis;
        auto readedPoint = lis.GetRawPoint();
        UNIT_ASSERT_EQUAL(defaultPoint, readedPoint);
        UNIT_ASSERT_UNEQUAL(wrongPoint, readedPoint);
    }
}
